//----------------------------------------------------------------------------
//  EDGE WIN32 CD Handling Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
// -ACB- 1999/11/09 Written
//

#include "..\i_defs.h"
#include "i_sysinc.h"

#define CDDEVICE "cdaudio"

typedef struct
{
  WORD id;
  int currenttrack;
  int startpos;
  int finishpos;
  int pausedpos;
}
cdinfo_t;

static cdinfo_t *currcd = NULL;
static MCIERROR errorcode;

//
// I_CDStartPlayback
//
// Initialises playing a CD-Audio Track, returns false on failure.
//
boolean_t I_CDStartPlayback(int tracknum)
{
  MCI_OPEN_PARMS openparm;
  MCI_PLAY_PARMS playparm;
  MCI_SET_PARMS setparm;
  MCI_STATUS_PARMS statusparm;
  int numoftracks;
  char errordesc[256];

  // clear error description
  memset(errordesc, 0, sizeof(char)*256);

  if (currcd)
    I_CDStopPlayback();

  // create cd info object
  currcd = malloc(sizeof(cdinfo_t));
  if (!currcd)
  {
    I_PostMusicError("I_CDPlayTrack: Unable to open CD-Audio device\n");
    return false;
  }

  // open parameters
  openparm.dwCallback      = (DWORD)mainwindow;
  openparm.lpstrDeviceType = CDDEVICE;

  // Open MCI CD-Audio
  errorcode = mciSendCommand(0, MCI_OPEN, MCI_OPEN_TYPE, (DWORD)&openparm);
  if (errorcode)
  {
    if (!mciGetErrorString(errorcode, errordesc, 128))
      I_PostMusicError("Unknown Error");
    else
      I_PostMusicError(errordesc);

    free(currcd);
    currcd = NULL;
    return false;
  }

  // Set global deviceid
  currcd->id = openparm.wDeviceID;

  // Get the status of MCI CD-Audio
  statusparm.dwCallback = (DWORD)mainwindow;
  statusparm.dwItem     = MCI_STATUS_MEDIA_PRESENT;

  errorcode = mciSendCommand(currcd->id, MCI_STATUS, MCI_STATUS_ITEM, (DWORD)&statusparm);
  if (errorcode || !statusparm.dwReturn)
  {
    if (!mciGetErrorString(errorcode, errordesc, 128))
      I_PostMusicError("Unknown Error");
    else
      I_PostMusicError(errordesc);

    mciSendCommand(currcd->id, MCI_CLOSE, 0, (DWORD)NULL);
    free(currcd);
    currcd = NULL;
    return false;
  }

  // Get the number of CD Tracks
  statusparm.dwCallback = (DWORD)mainwindow;
  statusparm.dwItem     = MCI_STATUS_NUMBER_OF_TRACKS;

  errorcode = mciSendCommand(currcd->id, MCI_STATUS, MCI_STATUS_ITEM, (DWORD)&statusparm);
  if (errorcode)
  {
    if (!mciGetErrorString(errorcode, errordesc, 128))
      I_PostMusicError("Unknown Error");
    else
      I_PostMusicError(errordesc);

    mciSendCommand(currcd->id, MCI_CLOSE, 0, (DWORD)NULL);
    free(currcd);
    currcd = NULL;
    return false;
  }

  numoftracks = statusparm.dwReturn;
  if (tracknum >= numoftracks)
  {
    mciSendCommand(currcd->id, MCI_CLOSE, 0, (DWORD)NULL);
    I_PostMusicError("Track exceeds available tracks");
    free(currcd);
    currcd = NULL;
    return false;
  }

  // Get the status of the CD Track
  statusparm.dwCallback = (DWORD)mainwindow;
  statusparm.dwItem     = MCI_CDA_STATUS_TYPE_TRACK;
  currcd->currenttrack  = statusparm.dwTrack    = tracknum;

  errorcode = mciSendCommand(currcd->id, MCI_STATUS, MCI_STATUS_ITEM|MCI_TRACK, (DWORD)&statusparm);
  if (errorcode)
  {
    if (!mciGetErrorString(errorcode, errordesc, 128))
      I_PostMusicError("Unknown Error");
    else
      I_PostMusicError(errordesc);

    mciSendCommand(currcd->id, MCI_CLOSE, 0, (DWORD)NULL);
    free(currcd);
    currcd = NULL;
    return false;
  }

  // Check its an audio track....
  if (statusparm.dwReturn != MCI_CDA_TRACK_AUDIO)
  {
    mciSendCommand(currcd->id, MCI_CLOSE, 0, (DWORD)NULL);
    I_PostMusicError("Track is not Audio\n");
    free(currcd);
    currcd = NULL;
    return false;
  }

  // Setup time format
  setparm.dwTimeFormat = MCI_FORMAT_TMSF;
  mciSendCommand(currcd->id, MCI_SET, MCI_SET_TIME_FORMAT, (DWORD)(LPVOID)&setparm);

  // Setup play parameters
  playparm.dwCallback = (DWORD)mainwindow;
  currcd->startpos = playparm.dwFrom = MCI_MAKE_TMSF(tracknum, 0, 0, 0);

  // Check if last track...
  if (tracknum == (numoftracks-1))
  {
    currcd->finishpos = playparm.dwTo = 0L;
    errorcode = mciSendCommand(currcd->id, MCI_PLAY, MCI_NOTIFY|MCI_FROM, (DWORD)(LPVOID)&playparm);
  }
  else
  {
    currcd->finishpos = playparm.dwTo = MCI_MAKE_TMSF(tracknum+1, 0, 0, 0);
    errorcode = mciSendCommand(currcd->id, MCI_PLAY, MCI_NOTIFY|MCI_FROM|MCI_TO, (DWORD)(LPVOID)&playparm);
  }

  if (errorcode)
  {
    if (!mciGetErrorString(errorcode, errordesc, 128))
      I_PostMusicError("Unknown Error");
    else
      I_PostMusicError(errordesc);

    mciSendCommand(currcd->id, MCI_CLOSE, 0, (DWORD)NULL);
    free(currcd);
    currcd = NULL;
    return false;
  }

  return true;
}

//
// I_CDPausePlayback
//
boolean_t I_CDPausePlayback(void)
{
  MCI_SET_PARMS setparm;
  MCI_STATUS_PARMS statusparm;
  char errordesc[256];

  // clear error description
  memset(errordesc, 0, sizeof(char)*256);

  if (!currcd)
  {
    I_PostMusicError("CD has not been started\n");
    return false;
  }

  setparm.dwTimeFormat = MCI_FORMAT_TMSF;
  mciSendCommand(currcd->id, MCI_SET, MCI_SET_TIME_FORMAT, (DWORD)(LPVOID)&setparm);

  statusparm.dwCallback  = (DWORD)mainwindow;
  statusparm.dwItem = MCI_STATUS_POSITION;

  errorcode = mciSendCommand(currcd->id, MCI_STATUS, MCI_STATUS_ITEM, (DWORD)&statusparm);
  if (errorcode)
  {
    if (!mciGetErrorString(errorcode, errordesc, 128))
      I_PostMusicError("Unknown Error");
    else
      I_PostMusicError(errordesc);

    return false;
  }

  currcd->pausedpos = statusparm.dwReturn;

  mciSendCommand(currcd->id, MCI_STOP, 0, (DWORD)NULL);

  return true;
}

//
// I_CDResumePlayback
//
boolean_t I_CDResumePlayback(void)
{
  MCI_SET_PARMS setparm;
  MCI_PLAY_PARMS playparm;
  char errordesc[256];

  // clear error description
  memset(errordesc, 0, sizeof(char)*256);

  if (!currcd)
  {
    I_PostMusicError( "CD has not been started\n");
    return false;
  }

  setparm.dwTimeFormat = MCI_FORMAT_TMSF;
  mciSendCommand(currcd->id, MCI_SET, MCI_SET_TIME_FORMAT, (DWORD)(LPVOID) &setparm);

  playparm.dwCallback = (DWORD)mainwindow;
  playparm.dwFrom     = currcd->pausedpos;
  playparm.dwTo       = currcd->finishpos;

  if (!currcd->finishpos)
    errorcode = mciSendCommand(currcd->id, MCI_PLAY, MCI_NOTIFY|MCI_FROM, (DWORD)(LPVOID)&playparm);
  else
    errorcode = mciSendCommand(currcd->id, MCI_PLAY, MCI_NOTIFY|MCI_FROM|MCI_TO, (DWORD)(LPVOID)&playparm);

  if (errorcode)
  {
    if (!mciGetErrorString(errorcode, errordesc, 128))
      I_PostMusicError("Unknown Error");
    else
      I_PostMusicError(errordesc);

    free(currcd);
    currcd = NULL;
    return false;
  }

  return true;
}

//
// I_CDStopPlayback
//
void I_CDStopPlayback(void)
{
  if (!currcd)
    return;

  mciSendCommand(currcd->id, MCI_STOP, 0, (DWORD)NULL);
  mciSendCommand(currcd->id, MCI_CLOSE, 0, (DWORD)NULL);
  free(currcd);
  currcd = NULL;
  return;
}

//
// I_CDFinished
//
// Has the CD Finished playing
//
boolean_t I_CDFinished(void)
{
  MCI_PLAY_PARMS playparm;
  MCI_STATUS_PARMS statusparm;

  if (!currcd)
    return false;

  // Get the status of MCI CD-Audio
  statusparm.dwCallback = (DWORD)mainwindow;
  statusparm.dwItem     = MCI_STATUS_MODE;

  mciSendCommand(currcd->id, MCI_STATUS, MCI_STATUS_ITEM, (DWORD)&statusparm);

  switch(statusparm.dwReturn)
  {
    case MCI_MODE_NOT_READY:
    case MCI_MODE_PLAY:
    case MCI_MODE_SEEK:
      return false;
  }

  return true;
}
