//----------------------------------------------------------------------------
//  EDGE WIN32 Input System
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------

#include "..\i_defs.h"
#include "i_sysinc.h"

#include "..\e_main.h"

// INPUT DEVICES
LPDIRECTINPUT diobject;

// CONTROL BUFFERING
typedef struct buffyevent_s
{
  int type;
  int data1;
  int data2;
}
buffyevent_t;

#define BUFQUESIZE 32
static buffyevent_t buffevent[BUFQUESIZE];
static int numbuffevent;
static boolean_t lockcontrol = true; // by default don't read

// KEYBOARD
#define KEYBUFSIZE 256
LPDIRECTINPUTDEVICE keyboard;
static byte buffer[KEYBUFSIZE];
static boolean_t keyisdown[KEYBUFSIZE];
static boolean_t syskeydown[KEYBUFSIZE]; // for message handled keys

// Key Conversion Tables...
static unsigned char EDGE2SCAN[256];
static unsigned char SCAN2EDGE[256];

// ASCII Key table - Borrowed from allegro.
static unsigned char KeyASCTable[128] =
{
   0,   27,  '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=', 8,   9,       /* 0 */
   'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', '[', ']', 13,  0,   'a', 's',     /* 1 */
   'd', 'f', 'g', 'h', 'j', 'k', 'l', ';', 39,  '`', 0,   92,  'z', 'x', 'c', 'v',     /* 2 */
   'b', 'n', 'm', ',', '.', '/', 0,   '*', 0,   ' ', 0,   3,   3,   3,   3,   8,       /* 3 */
   3,   3,   3,   3,   3,   0,   0,   0,   0,   0,   '-', 0,   0,   0,   '+', 0,       /* 4 */
   0,   0,   0,   127, 0,   0,   92,  3,   3,   0,   0,   0,   0,   0,   0,   0,       /* 5 */
   13,  0,   '/', 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   127,     /* 6 */
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   '/', 0,   0,   0,   0,   0        /* 7 */
};

// MOUSE
#define WINMOUSEBUTTONS 4
static LPDIRECTINPUTDEVICE mouse;
static boolean_t buttondown[WINMOUSEBUTTONS];

// BUFFER EVENT CODE

//
// AddBufferEvent
//
static void AddBufferEvent(int type, int data1, int data2)
{
  // Check for overflow
  if (numbuffevent >= BUFQUESIZE)
    return;

  buffevent[numbuffevent].type = type;
  buffevent[numbuffevent].data1 = data1;
  buffevent[numbuffevent].data2 = data2;
  numbuffevent++;
  return;
}

// KEYBOARD CODE

//
// ScanCodeConvert
//
static int ScanCodeConvert(int a)
{
  switch (a)
  {
    case DIK_GRAVE:        return KEYD_TILDE;  
    case DIK_UP:           return KEYD_UPARROW;
    case DIK_DOWN:         return KEYD_DOWNARROW;
    case DIK_LEFT:         return KEYD_LEFTARROW;
    case DIK_RIGHT:        return KEYD_RIGHTARROW;
    case DIK_RETURN:       return KEYD_ENTER;
    case DIK_ESCAPE:       return KEYD_ESCAPE;
    case DIK_TAB:          return KEYD_TAB;
    case DIK_SPACE:        return KEYD_SPACE;
    case DIK_F1:           return KEYD_F1;
    case DIK_F2:           return KEYD_F2;
    case DIK_F3:           return KEYD_F3;
    case DIK_F4:           return KEYD_F4;
    case DIK_F5:           return KEYD_F5;
    case DIK_F6:           return KEYD_F6;
    case DIK_F7:           return KEYD_F7;
    case DIK_F8:           return KEYD_F8;
    case DIK_F9:           return KEYD_F9;
    case DIK_F10:          return KEYD_F10;
    case DIK_F11:          return KEYD_F11;
    case DIK_F12:          return KEYD_F12;
    case DIK_BACKSPACE:    return KEYD_BACKSPACE;
    case DIK_EQUALS:       return KEYD_EQUALS;
    case DIK_NUMPADEQUALS: return KEYD_EQUALS;
    case DIK_MINUS:        return KEYD_MINUS;
    case DIK_NUMPADMINUS:  return KEYD_MINUS;
    case DIK_LSHIFT:       return KEYD_RSHIFT;
    case DIK_RSHIFT:       return KEYD_RSHIFT;
    case DIK_LCONTROL:     return KEYD_RCTRL;
    case DIK_RCONTROL:     return KEYD_RCTRL;
    case DIK_LMENU:        return KEYD_RALT;
    case DIK_RMENU:        return KEYD_RALT;
    case DIK_HOME:         return KEYD_HOME;
    case DIK_PRIOR:        return KEYD_PGUP;
    case DIK_END:          return KEYD_END;
    case DIK_NEXT:         return KEYD_PGDN;
    case DIK_INSERT:       return KEYD_INSERT;
    case DIK_DELETE:       return KEYD_DELETE;

    // -ACB- 2000/04/18 NUMLOCK, CAPS LOCK & SCROLL LOCK
    case DIK_SCROLL:       return KEYD_SCRLOCK;
    case DIK_NUMLOCK:      return KEYD_NUMLOCK;
    case DIK_CAPITAL:      return KEYD_CAPSLOCK;
  }

  if (a >= 0x100)
    return a;
  else if (KeyASCTable[a] > 8)
    return KeyASCTable[a];
  else
    return a + 0x80;
}

//
// StartupKeyboard 
//
static void StartupKeyboard(void)
{
  HRESULT result;
  int i, k;

  // Retrieve a pointer to an IDirectInputDevice interface 
  result = diobject->lpVtbl->CreateDevice(diobject, &GUID_SysKeyboard, &keyboard, NULL);
  if (result != DI_OK)     
    I_Error("CreateDevice (Keyboard) Failed!");
  
  // Set the data format using the predefined keyboard data 
  // format provided by the DirectInput object for keyboards. 
  result = keyboard->lpVtbl->SetDataFormat(keyboard, &c_dfDIKeyboard);
  if (result != DI_OK)
    I_Error("Keyboard->SetDataFormat Failed");
  
  // Set the cooperative level 
  result =
    keyboard->lpVtbl->SetCooperativeLevel(keyboard, mainwindow,
                                             DISCL_FOREGROUND |
                                                 DISCL_NONEXCLUSIVE);

  if (result != DI_OK)
    I_Error("SetCooperativeLevel Failed");

  // Set the keyisdown array to false. All keys are assumed to be up to start
  memset(keyisdown, false, sizeof(boolean_t)*KEYBUFSIZE);
  memset(syskeydown, false, sizeof(boolean_t)*KEYBUFSIZE);

  // Setup EDGE keycode conversion tables
  i=0;
  while (i<256)
  {
    SCAN2EDGE[i] = ScanCodeConvert(i);
    i++;
  }

  i=0;
  while (i<256)
  {
    k=0;
    while (k<256 && SCAN2EDGE[k] != i)
      k++;
 
    if (k != 256)
      EDGE2SCAN[i] = k;
    else
      EDGE2SCAN[i] = 0;
 
    i++;
  }

  I_Printf("StartupKeyboard: DirectInput KeyboardInit OK\n"); 
}

//
// ReadKeyboard
//
static void ReadKeyboard(void)
{
  HRESULT result;
  int i;

  result = keyboard->lpVtbl->GetDeviceState(keyboard, sizeof(byte)*KEYBUFSIZE, &buffer);

  // -ACB- 2000/06/05 Don't Throw Fatal Error
  if (result == DIERR_INPUTLOST || result == DIERR_NOTACQUIRED)
  {
    result = keyboard->lpVtbl->Acquire(keyboard);

    if (result == DI_OK)
      result = keyboard->lpVtbl->GetDeviceState(keyboard, sizeof(byte)*KEYBUFSIZE, &buffer);
    else
      return;
  }

  if (result != DI_OK)
    return;

  i=0;
  while(i<256)
  {
    if (buffer[i]&0x80)
    {
      if (!keyisdown[i])
      {
        AddBufferEvent(ev_keydown, SCAN2EDGE[i], 0);
        keyisdown[i] = true;
      }
    }  
    else
    {
      if (keyisdown[i])
      {
        AddBufferEvent(ev_keyup, SCAN2EDGE[i], 0);
        keyisdown[i] = false;
      }
    }

    i++;
  }
}

//
// Shutdownkeyboard
//
static void ShutdownKeyboard(void)
{
  if (keyboard) 
  { 
    keyboard->lpVtbl->Unacquire(keyboard); 
    keyboard->lpVtbl->Release(keyboard);
    keyboard = NULL; 
  } 
}

//
// I_HandleKeypress
//
// Handles keys that are generated by the message loop and not direct
// input. The reason for this distinction is because DirectInput does
// not handle some keys well.
//
// -ACB- 1999/10/18
//
void I_HandleKeypress(int key, boolean_t keydown)
{
  event_t event;

  if (keydown)
  {
    if (!syskeydown[key])
    {
      AddBufferEvent(ev_keydown, key, 0);
      syskeydown[key] = true;
    }
  }  
  else
  {
    if (syskeydown[key])
    {
      AddBufferEvent(ev_keyup, key, 0);
      syskeydown[key] = false;
    }
  }

  return;
}

// MOUSE CODE

//
// StartupMouse
//
static void StartupMouse(void)
{
  HRESULT result;

  result = diobject->lpVtbl->CreateDevice(diobject, &GUID_SysMouse, &mouse, NULL);
  if (result != DI_OK)
    I_Error("StartupMouse: CreateDevice (MOUSE) Failed\n");

  result = mouse->lpVtbl->SetDataFormat(mouse, &c_dfDIMouse);
  if (result != DI_OK)
    I_Error("StartupMouse: SetDataFormat (MOUSE) Failed\n");

  result = mouse->lpVtbl->SetCooperativeLevel(mouse, mainwindow, DISCL_EXCLUSIVE | DISCL_FOREGROUND);
  if (result != DI_OK)
    I_Error("StartupMouse: SetCooperativeLevel (MOUSE) Failed\n");

  I_Printf("StartupMouse: DirectInput MouseInit OK\n"); 

  // Set the buttondown array to false. All buttons are assumed to be up to start
  memset(buttondown, false, sizeof(boolean_t)*WINMOUSEBUTTONS);
}

//
// ReadMouse
//
static void ReadMouse(void)
{
  DIMOUSESTATE mousestate;
  HRESULT result;
  int i;
  int amount;

  result = mouse->lpVtbl->GetDeviceState(mouse, sizeof(DIMOUSESTATE), &mousestate);

  // -ACB- 2000/06/05 Don't Throw Fatal Error
  if (result == DIERR_INPUTLOST || result == DIERR_NOTACQUIRED)
  {
    result = mouse->lpVtbl->Acquire(mouse);

    if (result == DI_OK)
      result = mouse->lpVtbl->GetDeviceState(mouse, sizeof(DIMOUSESTATE), &mousestate);
    else
      return;
  }

  if (result != DI_OK)
    return;

  if (mousestate.lX != 0)
  {
    amount = (mousestate.lX*4)*(mouseSensitivity+5);
    AddBufferEvent(ev_analogue, mouse_xaxis, amount);
  }

  if (mousestate.lY != 0)
  {
    amount = (mousestate.lY*-4)*(mouseSensitivity+5);

    if (invertmouse)
      amount = -amount;

    AddBufferEvent(ev_analogue, mouse_yaxis, amount);
  }
  
  // Treat mouse buttons like keys...
  i=0;
  while (i<WINMOUSEBUTTONS)
  {
    if (mousestate.rgbButtons[i] & 0x80)
    {
      if (!buttondown[i])
      {
        AddBufferEvent(ev_keydown, KEYD_MOUSE1+i, 0);
        buttondown[i] = true;
      }
    }
    else
    {
      if (buttondown[i])
      {
        AddBufferEvent(ev_keyup, KEYD_MOUSE1+i, 0);
        buttondown[i] = false;
      }
    }
    i++;
  }
}

//
// ShutdownMouse
//
static void ShutdownMouse(void)
{
  if (mouse) 
  { 
    mouse->lpVtbl->Unacquire(mouse); 
    mouse->lpVtbl->Release(mouse);
    mouse = NULL; 
  } 
}

// JOYSTICK CODE
void StartupJoystick(void)
{
}

void ReadJoystick(void)
{
}

void ShutdownJoystick(void)
{
}

void I_CalibrateJoystick(int ch)
{
}

// MAIN

//
// I_ControlTicker
//
// WIN32 Internal Control Ticker
//
void I_ControlTicker(void)
{
  if (lockcontrol || !appactive)
    return;

  ReadKeyboard();
  ReadMouse();
  ReadJoystick();
}

//
// I_StartupControl
//
// Gets a directinput object up and running....
//
void I_StartupControl(void)
{
  HRESULT result;

  SetFocus(mainwindow);
  result = DirectInputCreate(maininstance, DIRECTINPUT_VERSION, &diobject, NULL);

  if (result != DI_OK) 
    I_Error("DirectInputCreate Failed!");
  
  StartupKeyboard();
  StartupMouse();
  StartupJoystick();

  // Allow input buffering now
  lockcontrol = false;

  return;
}

//
// I_ControlGetEvents
//
// Gets all the input devices to post their events
//
void I_ControlGetEvents(void)
{
  int i;
  event_t event;

  // lock control routine - we're reading
  lockcontrol = true;

  // Process Buffered Events
  i=0;
  while (i<numbuffevent)
  {
    memset(&event, sizeof(event_t), 0);

    switch (buffevent[i].type)
    {
      case ev_keyup:
        event.type = ev_keyup;
        event.value.key = buffevent[i].data1;
        E_PostEvent(&event);
        break;

      case ev_keydown:
        event.type = ev_keydown;
        event.value.key = buffevent[i].data1;
        E_PostEvent(&event);
        break;

      case ev_analogue:
        event.type = ev_analogue;
        event.value.analogue.axis = buffevent[i].data1;
        event.value.analogue.amount = buffevent[i].data2;
        E_PostEvent(&event);
        break;

      default:
        break;
    }
    i++;
  }

  // Zeroise buffer count
  numbuffevent = 0;

  // unlock control routine
  lockcontrol = false;
}

//
// I_ShutdownControl
//
// Return input devices to system control
//
void I_ShutdownControl(void)
{
  if (diobject)
  { 
    // Stop input buffering now
    lockcontrol = true;

    ShutdownJoystick();
    ShutdownMouse();
    ShutdownKeyboard();

    diobject->lpVtbl->Release(diobject);
    diobject = NULL;
  } 
}

