//----------------------------------------------------------------------------
//  EDGE WIN32 MP3 Handling Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//

#include "..\i_defs.h"
#include "..\l_mp3.h"
#include "i_sysinc.h"

// #include "..\common\l_mp3.h"

#define _MP3_BUFFERSIZE 16384
#define _MP3_BUFHLFSIZE (_MP3_BUFFERSIZE/2)

static int _MP3_OUTFREQ    = 11025;  // Output Frequency
static int _MP3_CHANNELS   = 1;      // 1==Mono, 2==Stereo
static int _MP3_SAMPLEBITS = 8;      // 8 or 16 bit only

#define _MP3_MAXVOL     15

// ==== PRIVATE VARIABLES ====
static LPDIRECTSOUNDBUFFER mp3buffer;  // Music Buffer
static boolean_t inited = false;       // Initialised
static boolean_t playing = false;      // Playing?
static boolean_t looping = false;      // Looping?
static boolean_t pausing = false;      // Paused?
static int half[2];                    // Progress by half
static int voldiff;                    // Difference between MAX and MIN vol
// ===========================

// ============================= INTERNALS ==============================

//
// CalcVolume
//
// This is a conversion of seperation value from
// the range 0(quietest)-15(loudest) to the
// DirectSound Values. Due to DirectSound handling
// the calculation makes DirectSound half-vol the
// quietest (quality reasons).
//
static int CalcVolume(int vol)
{
  if (vol)
  {
    vol *= voldiff;
    vol /= _MP3_MAXVOL;
  }

  //
  // This should not be neccessary, but for some reason volume below half
  // of volume is inaudible
  //
  vol = DSBVOLUME_MIN + (voldiff/2) + (vol/2);

  return vol;
}

//
// LoadSampleData
//
static void LoadSampleData(boolean_t firsthalf)
{
  LPVOID buff1;
  LPVOID buff2;
  DWORD len1;
  DWORD len2;
  int startpos;
  byte *snddata;

  if (firsthalf)
    startpos = 0;
  else        
    startpos = _MP3_BUFHLFSIZE;

  snddata = L_MP3ReadBuffer((_MP3_BUFHLFSIZE)/(_MP3_CHANNELS));
  if (!snddata)
    return;

  L_MP3ReadAdvance((_MP3_BUFHLFSIZE)/(_MP3_CHANNELS));

  if (mp3buffer->lpVtbl->Lock(mp3buffer, startpos, _MP3_BUFHLFSIZE, &buff1, &len1, &buff2, &len2, 0) != DS_OK)
    return;

  memcpy(buff1, snddata, len1);

  if (mp3buffer->lpVtbl->Unlock(mp3buffer, buff1, len1, buff2, len2) != DS_OK)
    return;

  return;
}

//
// CreateBuffer
//
// Creates a DirectSound Secondary Buffer of the correct length
//
static LPDIRECTSOUNDBUFFER CreateBuffer(LPDIRECTSOUND dsobj, int length)
{
  LPDIRECTSOUNDBUFFER buffer;
  DSBUFFERDESC bufferdesc;
  PCMWAVEFORMAT pcmwf;

  // Set up wave format structure.
  memset(&pcmwf, 0, sizeof(PCMWAVEFORMAT));
  pcmwf.wf.wFormatTag         = WAVE_FORMAT_PCM;      
  pcmwf.wf.nChannels          = _MP3_CHANNELS;
  pcmwf.wf.nSamplesPerSec     = _MP3_OUTFREQ;
  pcmwf.wf.nBlockAlign        = (_MP3_SAMPLEBITS/8);               
  pcmwf.wf.nAvgBytesPerSec    = _MP3_OUTFREQ*_MP3_CHANNELS;
  pcmwf.wBitsPerSample        = (WORD)_MP3_SAMPLEBITS;

  // Set up DSBUFFERDESC structure.
  memset(&bufferdesc, 0, sizeof(DSBUFFERDESC)); 
  bufferdesc.dwSize        = sizeof(DSBUFFERDESC);
  bufferdesc.dwFlags       = DSBCAPS_CTRLVOLUME|DSBCAPS_GETCURRENTPOSITION2;
  bufferdesc.dwBufferBytes = length;
  bufferdesc.lpwfxFormat   = (LPWAVEFORMATEX)&pcmwf;  
    
  if (dsobj->lpVtbl->CreateSoundBuffer(dsobj, &bufferdesc, &buffer, NULL) != DS_OK)
    return NULL;
 
  return buffer;
}
// ========================== END OF INTERNALS ==========================

//
// I_MP3Init
//
boolean_t I_StartupMP3(void)
{
  LPDIRECTSOUND soundobj;

  // Already started?
  if (inited)
    return false;

  // Get sound object
  soundobj = I_SoundReturnObject();
  if (!soundobj)
    return false;

  // Create MP3 buffer
  mp3buffer = CreateBuffer(soundobj, _MP3_BUFFERSIZE);
  if (!mp3buffer)
    return false;

  // Calc Volume Range
  voldiff = DSBVOLUME_MAX - DSBVOLUME_MIN;
  if (voldiff<0)
    voldiff = 0 - voldiff;

  L_MP3Init(_MP3_OUTFREQ, _MP3_SAMPLEBITS, _MP3_CHANNELS);

  inited = true;
  return true;
}

//
// I_MP3PlayTrack
//
int I_MP3PlayTrack(char *filename, boolean_t loopy)
{
  int samplerate;
  byte* snddata;
  byte* buffdata;

  // Sanity Checks...
  if (!filename || !inited)
    return -1;

  // -AJA- shouldn't happen, but if MP3 is playing, shut it off
  if (playing)
  {
    L_WriteDebug("I_MP3PlayTrack: didn't clear MP3 !\n");
    L_MP3ClearMusicFile();
    playing = false;
  }
    
  // Load Music into library
  if (!L_MP3SetMusicFile(filename, &samplerate))
    return -1;

  L_WriteDebug("MP3 Freq: %d\n", samplerate);

  // The half values exist to show the age of the sound in the buffer: the
  // half which has the higher value is the most recent.
  half[0] = 0; half[1] = -1;

  if (mp3buffer->lpVtbl->SetVolume(mp3buffer, 0) != DS_OK)
  {
    L_WriteDebug("I_MP3PlayTrack: DX SetVolume() failed\n");
    L_MP3ClearMusicFile();
    return -1;
  }

  mp3buffer->lpVtbl->SetCurrentPosition(mp3buffer, 0);

  LoadSampleData(true);

  if (mp3buffer->lpVtbl->Play(mp3buffer, 0, 0, DSBPLAY_LOOPING) != DS_OK)
  {
    L_WriteDebug("I_MP3PlayTrack: DX Play() failed\n");
    L_MP3ClearMusicFile();
    return -1;
  }

  L_WriteDebug("%s started\n", filename); 

  playing = true;
  pausing = false;
  looping = loopy;

  return 1;
}
                 
//
// I_MP3BufferFill
//
void I_MP3BufferFill(void)
{
  byte *snddata;
  DWORD pos;

  if (!playing || pausing || !appactive || !inited)
    return;

  L_MP3FillBuffer();
}

//
// I_MP3Ticker
//
void I_MP3Ticker(void)
{
  byte *snddata;
  DWORD pos;

  if (!playing || pausing || !appactive || !inited)
    return;

  mp3buffer->lpVtbl->GetCurrentPosition(mp3buffer, &pos, NULL);

  if (!L_MP3FillBuffer())
  {
    if (!looping)
      I_MP3Stop();
    else
      L_MP3RestartMusicFile();

    return;
  }

  if (pos < _MP3_BUFHLFSIZE) // Read pos in first half
  {
    if (half[0] > half[1])        // The second half has old data - Update
    {
      half[1] = half[0] + 1;
      LoadSampleData(false);
    }
  }
  else                            // Read pos in second half
  {
    if (half[0] < half[1])        // The first half has old data - Update
    {
      half[0] = half[1] + 1;
      LoadSampleData(true);
    }
  }

  return;  
}

//
// I_MP3Pause
//
void I_MP3Pause(void)
{
  if (!playing || pausing || !inited)
    return;

  mp3buffer->lpVtbl->Stop(mp3buffer);
  pausing = true;
}

//
// I_MP3Resume
//
void I_MP3Resume(void)
{
  if (!playing || !pausing || !inited)
    return;

  mp3buffer->lpVtbl->Play(mp3buffer, 0, 0, DSBPLAY_LOOPING);
  pausing = false;
}

//
// I_MP3Stop
//
void I_MP3Stop(void)
{
  LPVOID buff1;
  LPVOID buff2;
  DWORD len1;
  DWORD len2;

  if (!playing || !inited)
    return;

  if (pausing)
    I_MP3Resume();

  mp3buffer->lpVtbl->Stop(mp3buffer);

  mp3buffer->lpVtbl->Lock(mp3buffer, 0, 0, &buff1, &len1, &buff2, &len2, DSBLOCK_ENTIREBUFFER);
  memset(buff1, 128, len1);
  memset(buff2, 128, len2);
  mp3buffer->lpVtbl->Unlock(mp3buffer, buff1, len1, buff2, len2);

  L_MP3ClearMusicFile();
  playing = false;
}

//
// I_ShutdownMP3
//
void I_ShutdownMP3(void)
{       
  if (!inited)
    return;

  if (playing)
    I_MP3Stop();

  L_MP3Shutdown();

  if (mp3buffer)
  {
    mp3buffer->lpVtbl->Release(mp3buffer);
    mp3buffer = NULL;
  }

  inited = false;
}

//
// I_MP3Playing
//
boolean_t I_MP3Playing(void)
{
  if (!inited)
    return false;

  return playing;
}

//
// I_MP3SetVolume
//
void I_MP3SetVolume(int vol)
{
  LONG actualvol;

  if (!inited)
    return;

  actualvol = CalcVolume(vol);
  mp3buffer->lpVtbl->SetVolume(mp3buffer, actualvol);
}

