//----------------------------------------------------------------------------
//  EDGE Win32 OpenGL Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
// -ACB- 1999/11/06 Written
//
#ifdef USE_GL

#include "..\i_defs.h"

#include "..\dm_defs.h"
#include "..\dm_state.h"
#include "..\i_system.h"
#include "..\st_stuff.h"
#include "..\m_argv.h"
#include "..\v_res.h"
#include "..\v_screen.h"
#include "..\w_wad.h"
#include "..\z_zone.h"
#include "..\v_colour.h"

#include "i_sysinc.h"

static boolean_t alreadystarted = false;
static boolean_t changedepth;
static HDC devicecontext;
static HGLRC glRendercontext;

static screen_t dummy_screen;
static int bitspp = 8;
static byte *buffer;

static int winwidth;
static int winheight;

// Possible Screen Modes
static screenmode_t possresmode[] =
{
  // windowed modes
  { 640, 400,  16, true},
  { 640, 480,  16, true},
  { 800, 600,  16, true},
  {1024, 768,  16, true},
  {1152, 864,  16, true},
  {1280, 1024, 16, true},
  {1600, 1200, 16, true},

  {  -1,  -1, -1}
};

// ========================= INTERNALS ==========================

//
// MakeColour
//
static int MakeColour(byte red, byte green, byte blue)
{
  red   = ((red   >> 3)&0x1F);
  green = ((green >> 2)&0x3F);
  blue  = ((blue  >> 3)&0x1F);

  return ((red << 11)&0xF800) + ((green << 5)&0x7E0) + (blue&0x1F);
}

//
// GraphicsBuildModeList
//
void GraphicsBuildModeList(void)
{
  DEVMODE dm;
  int mode, i;

  mode = 0;
  while(EnumDisplaySettings(NULL, mode, &dm))
  {
    screenmode_t edgeMode;

    edgeMode.width    = dm.dmPelsWidth;
    edgeMode.height   = dm.dmPelsHeight;
    edgeMode.depth    = dm.dmBitsPerPel;
    edgeMode.windowed = false;
    
    if (dm.dmBitsPerPel == 16)
      V_AddAvailableResolution(&edgeMode);

    mode++;
  }

  // handle windowing modes
  for (i=0; possresmode[i].width != -1; i++)
  {
    V_AddAvailableResolution(possresmode + i);
  }

  return;
}

//
// GraphicsSetDepthSwitch
//
// Sets whether the operating system can change the colour depth
//
// -ACB- 1999/09/19 OSR2_BUILD_NUMBER replaced by MinorVersion check.
//
void GraphicsSetDepthSwitch(void)
{
  OSVERSIONINFO vinfo;

  vinfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

  changedepth = false;

  if (GetVersionEx(&vinfo))
  {
    if (vinfo.dwMajorVersion > 4)
    {
      changedepth = true;
    }
    else if (vinfo.dwMajorVersion == 4)
    {
      if (vinfo.dwPlatformId == VER_PLATFORM_WIN32_NT)
      {
        changedepth = true;
      }
      else if (vinfo.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS)
      {
        if (vinfo.dwMinorVersion > 0)
          changedepth = true;
      }
    }
  }
}

//
// GraphicsSetVideoMode
//
static boolean_t GraphicsSetVideoMode(screenmode_t *mode)
{
  DEVMODE dm;

  RECT winrect;
  RECT clientrect;
  POINT point;

  L_WriteDebug("Tried: %dx%dx%d\n", mode->width, mode->height, mode->depth);

  if (!changedepth)
  {
    I_Warning("This version of Windows cannot change color depth.\n"
              "Please request different video mode settings or adjust\n"
              "your desktop colour depth.\n"
              "Current depth '%d', attempt depth '%d'\n",
              BPP, mode->depth);

//!!! FIXME:    return false;
  }

  memset(&dm, 0, sizeof(dm));
  dm.dmSize = sizeof(dm);

  dm.dmPelsWidth  = mode->width;
  dm.dmPelsHeight = mode->height;
  dm.dmFields     = DM_PELSWIDTH | DM_PELSHEIGHT;

  dm.dmBitsPerPel = mode->depth;
  dm.dmFields |= DM_BITSPERPEL;

  if (ChangeDisplaySettings(&dm, CDS_FULLSCREEN) != DISP_CHANGE_SUCCESSFUL)
  {
    I_Warning("Unable to change windows screen size\n");
    return false;
  }

  if (0)  //!!!! FIXME: mode->windowed)
  {
    GetWindowRect(mainwindow, &winrect);

    GetClientRect(mainwindow, &clientrect);
    point.x = clientrect.left;
    point.y = clientrect.top;
    ClientToScreen(mainwindow, &point);

    winwidth = ((point.x-winrect.left)+3) + mode->width;
    winheight = ((point.y-winrect.top)+3) + mode->height;

    SetWindowPos(mainwindow, HWND_TOP, 0, 0, winwidth, winheight, SWP_NOMOVE);
  }
  else
  {
    SetWindowPos(mainwindow, HWND_TOP, 0, 0, mode->width, mode->height, SWP_NOMOVE);
  }

  return true;
}

// ====================== END OF INTERNALS ======================

//
// I_GetTruecolInfo
//
// -ACB- 1999/11/06 Returns 24-bit colour always (OpenGL).
//
void I_GetTruecolInfo(truecol_info_t *info)
{
  info->red_bits = 5;
  info->red_shift = 11;
  info->red_mask = 0xF800;

  info->green_bits = 6;
  info->green_shift = 5;
  info->green_mask = 0x07E0;

  info->blue_bits = 5;
  info->blue_shift = 0;
  info->blue_mask = 0x001F;

  info->grey_mask = 0xFFDF;
  return;
/*
  info->red_bits = 8;
  info->red_shift = 16;
  info->red_mask = 0xFF0000;

  info->green_bits = 8;
  info->green_shift = 8;
  info->green_mask = 0xFF00;

  info->blue_bits = 8;
  info->blue_shift = 0;
  info->blue_mask = 0xFF;

  return;
*/
}

//
// I_StartupGraphics
//
void I_StartupGraphics(void)
{
  GraphicsBuildModeList();
  GraphicsSetDepthSwitch();
  I_Printf("I_StartupGraphics: Win32 OpenGL Init OK\n");
}

//
// I_SetScreenSize
//
// Sets the screen size to the given parameters
//
boolean_t I_SetScreenSize(screenmode_t *mode)
{
  int actdepth;
  int pixformatnum;

  PIXELFORMATDESCRIPTOR pixformatdesc =
  {
    sizeof(PIXELFORMATDESCRIPTOR),  // size of this pfd
    1,                       // version number
    PFD_DRAW_TO_WINDOW |     // support window
    PFD_SUPPORT_OPENGL |     // support OpenGL
    PFD_DOUBLEBUFFER,        // double buffered
    PFD_TYPE_RGBA,           // RGBA type
    24,                      // 24-bit color depth
    0, 0, 0, 0, 0, 0,        // color bits ignored
    0,                       // no alpha buffer
    0,                       // shift bit ignored
    0,                       // no accumulation buffer
    0, 0, 0, 0,              // accum bits ignored
    32,                      // 32-bit z-buffer
    0,                       // no stencil buffer
    0,                       // no auxiliary buffer
    PFD_MAIN_PLANE,          // main layer
    0,                       // reserved
    0, 0, 0                  // layer masks ignored
  };

  if (mode->depth != 16)
    return false;

  // -AJA- don't allow windowed mode (FIXME)
  if (mode->windowed)
    return false;

  // Kill any existing windows...
  if (alreadystarted)
    I_ShutdownGraphics();

  // Set the desired video mode and/or color depth
  if (!GraphicsSetVideoMode(mode))
    return false;

  devicecontext = GetDC(mainwindow);

  // we want to get the pixel format for the above descriptor
  if ((pixformatnum = ChoosePixelFormat(devicecontext, &pixformatdesc)) == 0)
    return false;

  // got the number now set the pixel format
  if (SetPixelFormat(devicecontext, pixformatnum, &pixformatdesc) == FALSE)
    return false;

  // Get the render context...
  if ((glRendercontext = wglCreateContext(devicecontext)) == 0)
    return false;

  // make this render context current: we want to work with it
  if (!wglMakeCurrent(devicecontext, glRendercontext))
    return false;

  glClearColor (0.0f, 0.0f, 0.0f, 0.0f);
  glViewport(0, 0, SCREENWIDTH, SCREENHEIGHT);

  I_Printf("I_SetScreenSize: Win32 OpenGL Screen Size OK\n");

  //
  // NOTE: we don't need main_scr, BUT current code relies on it
  //
  actdepth = V_GetPitch(mode->width, mode->depth / 8);

  // -ES- 1998/08/20 Destroy bitmap
  if (buffer)
  {
    Z_Free(buffer);
    buffer = NULL;
  }

  buffer = Z_ClearNew(byte, actdepth * mode->height);

  dummy_screen.width = mode->width;
  dummy_screen.height = mode->height;
  dummy_screen.pitch = actdepth;
  dummy_screen.bytepp = mode->depth / 8;
  dummy_screen.parent = NULL;
  dummy_screen.data = buffer;

  main_scr = V_CreateSubScreen(&dummy_screen, 0, 0, mode->width, mode->height);

  SCREENPITCH = actdepth;

  // Switch of the cursor
  ShowCursor(FALSE);

  alreadystarted = true;

  return true;
}

//
// I_StartFrame
//
// -ACB- 1999/11/06
//
void I_StartFrame(void)
{
}

//
// I_FinishFrame
//
void I_FinishFrame(void)
{
  //
  // This just calls whatever OpenGL implementation
  // uses to swap the video buffers. Win32 uses:
  //
  if (!SwapBuffers(devicecontext))
    L_WriteDebug("Failed frame swap at %d", I_GetTime());
}

//
// I_SetPalette
//
// Sets the palette.
// palette is 256 RGB triplets. Each R, G and B element ranges from 0 to 255.
//
// -ACB- 1999/11/06 UNUSED: OpenGL does not use palettes.
//
void I_SetPalette(byte palette[256][3])
{
}

//
// I_Colour2Pixel
//
// Gets a colour from a pixel value
//
// -AJA- 1999/07/05: added this.
// -ACB- 1999/11/06: Returns 24-bit RGB value for OpenGL
//
long I_Colour2Pixel(byte palette[256][3], int col)
{
  return (long)MakeColour(palette[col][0], palette[col][1], palette[col][2]);
}

//
// I_SizeWindow
//
void I_SizeWindow(void)
{
#if 0
  if (SCREENWINDOW)
  {
    SetWindowPos(mainwindow, HWND_TOP, 0, 0, winwidth, winheight,
        SWP_NOMOVE | SWP_SHOWWINDOW);
  }
#endif
}

//
// I_ShutdownGraphics
//
void I_ShutdownGraphics(void)
{
  if (glRendercontext)
  {
    wglMakeCurrent(NULL, NULL);
    wglDeleteContext(glRendercontext);
  }

  if (devicecontext)
    ReleaseDC(mainwindow, devicecontext);

  // restore original GFX mode
  ChangeDisplaySettings(0,0);

  alreadystarted = false;
}

#endif // USE_GL
