//----------------------------------------------------------------------------
//  EDGE Win32 Video Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
// -AJA- 1999/07/01: Moved the translucency lookup tables & calculation
//        functions to v_colour.c.
//
// -AJA- 1999/07/03: Added I_MakeColourmapRange for 16-bit colourmap
//                   generation.
//
// -AJA- 1999/07/05: Added I_Colour2Pixel.
//

#ifndef USE_GL

#include "..\i_defs.h"

#include "..\dm_defs.h"
#include "..\dm_state.h"
#include "..\i_system.h"
#include "..\m_argv.h"
#include "..\st_stuff.h"
#include "..\v_res.h"
#include "..\v_screen.h"
#include "..\w_wad.h"
#include "..\z_zone.h"
#include "..\v_colour.h"

#include "i_sysinc.h"

// DirectDraw - Use DX7 -ACB- 2001/01/29
static LPDIRECTDRAW7        drawobj = NULL;     // DirectDraw Object
static LPDIRECTDRAWSURFACE7 screen = NULL;      // DirectDraw Surface
static LPDIRECTDRAWPALETTE  scrpalette = NULL;

static screen_t dummy_screen;
static int bitspp = 8;
static byte *buffer;

// -ES- 2000/03/04 Added.
static boolean_t nogfxmode = false;

// Top-left hand corner of the window
static int topleft_x;
static int topleft_y;
static int winwidth;
static int winheight;

// Possible Screen Modes
static screenmode_t possresmode[] =
{
  // windowed modes
  { 320, 200,  8,  true},
  { 320, 240,  8,  true},
  { 400, 300,  8,  true},
  { 512, 384,  8,  true},
  { 640, 400,  8,  true},
  { 640, 480,  8,  true},
  { 800, 600,  8,  true},
  {1152, 864,  8,  true},
  {1024, 768,  8,  true},
  {1280, 1024, 8,  true},

  { 320, 200,  16, true},
  { 320, 240,  16, true},
  { 400, 300,  16, true},
  { 512, 384,  16, true},
  { 640, 400,  16, true},
  { 640, 480,  16, true},
  { 800, 600,  16, true},
  {1024, 768,  16, true},
  {1152, 864,  16, true},
  {1280, 1024, 16, true},

  {  -1,  -1, -1}
};

//
// MakeColour
//
static int MakeColour(byte red, byte green, byte blue)
{
  if (bitspp == 15)
  {
    red   = red   >> 3;
    green = green >> 3;
    blue  = blue  >> 3;

    return (((red << 10)&0xEC00) + ((green << 5)&0x3E0) + (blue&0x1F));
  }

  if (bitspp == 16)
  {
    red   = ((red   >> 3)&0x1F);
    green = ((green >> 2)&0x3F);
    blue  = ((blue  >> 3)&0x1F);

    return ((red << 11)&0xF800) + ((green << 5)&0x7E0) + (blue&0x1F);
  }

  return 0;
}

//
// EnumerateDisplayModes
//
static HRESULT CALLBACK EnumerateDisplayModes(LPDDSURFACEDESC2 ddsd, LPVOID lpContext)
{
  screenmode_t edgeMode;

  edgeMode.width = ddsd->dwWidth;
  edgeMode.height = ddsd->dwHeight;
  edgeMode.depth = ddsd->ddpfPixelFormat.dwRGBBitCount;
  edgeMode.windowed = false;

  if (edgeMode.depth == 8 || edgeMode.depth == 16)  
    V_AddAvailableResolution(&edgeMode);
  
  return DDENUMRET_OK;
}


//
// I_StartupGraphics
//
void I_StartupGraphics(void)
{
  HRESULT value;
  LPDIRECTDRAW ddobject;
  int i;

  // -ES- 2000/03/04 Added nogfxmode.
  if (M_CheckParm("-nogfxmode"))
    nogfxmode = true;
  else
    nogfxmode = false;

  if (nogfxmode)
    return;

  // create a directdraw object if we don't already have one
  if (!drawobj)
  {
    // Create the main DirectDraw object
    value = DirectDrawCreate(NULL, &ddobject, NULL);
    if (value != DD_OK)
      I_Error("I_GraphicsStartup: DirectDrawCreate FAILED");

    // Fetch DirectDraw interface
    value =
      ddobject->lpVtbl->QueryInterface(ddobject, &IID_IDirectDraw7, (LPVOID *)&drawobj);

    if (value != DD_OK)
      I_Error("I_GraphicsStartup: QueryInterface FAILED");

    // See what fullscreen display modes are available...
    // -ACB- 2000/07/05 NT Machines don't like STANDARDVGAMODES
    drawobj->lpVtbl->EnumDisplayModes(drawobj, DDEDM_REFRESHRATES, NULL, 0, EnumerateDisplayModes);
  
    // handle windowing modes
    for (i=0; possresmode[i].width != -1; i++)
    {
      V_AddAvailableResolution(possresmode + i);
    }
  }

  I_Printf("I_StartupGraphics: DirectDraw Init OK\n");
}

//
// I_GetTruecolInfo
//
// -AJA- 1999/07/04: added this.
// -ACB- 1999/09/19: Moved from I_ALLEGV.C
//
void I_GetTruecolInfo(truecol_info_t *info)
{
  if (bitspp == 15)
  {
    info->red_bits = 5;
    info->red_shift = 10;
    info->red_mask = 0xEC00;

    info->green_bits = 5;
    info->green_shift = 5;
    info->green_mask = 0x03E0;

    info->blue_bits = 5;
    info->blue_shift = 0;
    info->blue_mask = 0x001F;

    info->grey_mask = 0xFFFF;
    return;
  }

  if (bitspp == 16)
  {
    info->red_bits = 5;
    info->red_shift = 11;
    info->red_mask = 0xF800;

    info->green_bits = 6;
    info->green_shift = 5;
    info->green_mask = 0x07E0;

    info->blue_bits = 5;
    info->blue_shift = 0;
    info->blue_mask = 0x001F;

    info->grey_mask = 0xFFDF;
    return;
  }
}

//
// I_SetScreenSize
//
// -ACB- 1999/09/19 Moved from I_Allegv.C
//                  Removed DEBUG info.
//                  Removed blit selector.
//                  New parameters.
//
boolean_t I_SetScreenSize(screenmode_t *mode)
{
  HRESULT value;
  DDSURFACEDESC2 ddsd;
  int setdepth;
  int actdepth;
  int flags;
  DWORD changedelay;

  RECT winrect;
  RECT clientrect;
  POINT point;
  DWORD envwidth;
  DWORD envheight;

  // if we have a frontbuffer it will need to be destroyed
  if (screen != NULL)
  {
    screen->lpVtbl->Release(screen);
    screen = NULL;
  }

  setdepth = mode->depth;

  bitspp = setdepth;

  if (!nogfxmode)
  {
    value = drawobj->lpVtbl->SetCooperativeLevel(drawobj, mainwindow, 
        DDSCL_EXCLUSIVE|DDSCL_FULLSCREEN);

    if (value != DD_OK)
      return false;

    // Set the video mode
    if (mode->windowed)
    {
      GetWindowRect(mainwindow, &winrect);

      GetClientRect(mainwindow, &clientrect);
      point.x = clientrect.left;
      point.y = clientrect.top;
      ClientToScreen(mainwindow, &point);

      topleft_x = point.x;
      topleft_y = point.y;

      winwidth = ((topleft_x-winrect.left)+3) + mode->width;
      winheight = ((topleft_y-winrect.top)+3) + mode->height;

      envwidth = GetSystemMetrics(SM_CXSCREEN);
      envheight = GetSystemMetrics(SM_CYSCREEN);

      value=drawobj->lpVtbl->SetDisplayMode(drawobj, envwidth, envheight, 
          setdepth, 0, 0);

      if (value != DD_OK)
        return false;

      I_SizeWindow();
    }
    else
    {
      value=drawobj->lpVtbl->SetDisplayMode(drawobj, mode->width, mode->height, 
          setdepth, 0, 0);

      if (value != DD_OK)
        return false;
    }

    // Create the primary surface with no back buffer
    ZeroMemory(&ddsd, sizeof(ddsd));
    ddsd.dwSize = sizeof(ddsd);
    ddsd.dwFlags = DDSD_CAPS;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

    value = drawobj->lpVtbl->CreateSurface(drawobj, &ddsd, &screen, NULL);
    if (value != DD_OK)
      return false;

    // -ACB- 2000/04/09 Check for alpha bits...
    if (bitspp == 16)
    {
      ZeroMemory(&ddsd, sizeof(ddsd));
      ddsd.dwSize = sizeof(ddsd);
      screen->lpVtbl->GetSurfaceDesc(screen, &ddsd);

      // Cheap hack that detects 15-bit modes...
      if (ddsd.ddpfPixelFormat.dwGBitMask == 0x03e0)
        bitspp = 15;
    }

    // Switch of the cursor
    ShowCursor(FALSE);

    // -ACB- 2000/06/18 Experimental routine to stop res change crashes
    changedelay = timeGetTime();
    while (timeGetTime() > (changedelay+250)) {};
  }

  actdepth = V_GetPitch(mode->width, mode->depth / 8);

  // -ES- 1998/08/20 Destroy bitmap
  if (buffer)
  {
    Z_Free(buffer);
    buffer = NULL;
  }

  buffer = Z_ClearNew(byte, actdepth * mode->height);

  dummy_screen.width = mode->width;
  dummy_screen.height = mode->height;
  dummy_screen.pitch = actdepth;
  dummy_screen.bytepp = mode->depth / 8;
  dummy_screen.parent = NULL;
  dummy_screen.data = buffer;

  main_scr = V_CreateSubScreen(&dummy_screen, 0, 0, mode->width, mode->height);

  SCREENPITCH = actdepth;

  return true;
}

//
// I_StartFrame
//
// Called to prepare screen for rendering if necessary
//
void I_StartFrame(void)
{
}

//
// I_GraphicsRestore
//
// Checks if the surfaces need restoring...
//
boolean_t I_GraphicsRestore(void)
{
  if (screen)
  {
    if (screen->lpVtbl->IsLost(screen))
    {
      screen->lpVtbl->Restore(screen);
      return false;
    }
  }

  return true;
}

//
// I_FinishFrame
//
// -ACB- 1999/09/19 Simplified to the blitting function only.
//
void I_FinishFrame(void)
{
  DDSURFACEDESC2 ddsd;
  HRESULT hresult;
  int ycount;
  int winscrwidth;
  short *dest;
  short *src;
  byte *destbyte;
  byte *srcbyte;

  if (I_GraphicsRestore() == false)
    return;

  if (nogfxmode)
    return;

  if (BPP == 1)
  {
    // get corner point
    srcbyte = buffer;

    // get surface...
    ddsd.dwSize = sizeof(ddsd);
    screen->lpVtbl->GetSurfaceDesc(screen, &ddsd);
    hresult = screen->lpVtbl->Lock(screen, NULL, &ddsd, 
        DDLOCK_SURFACEMEMORYPTR|DDLOCK_WAIT, NULL);

    if (hresult == DD_OK)
    {
      winscrwidth = ddsd.lPitch/sizeof(byte);

      destbyte = (byte*)ddsd.lpSurface;

      if (SCREENWINDOW)
        destbyte += (topleft_x+(topleft_y*winscrwidth));

      for (ycount=0; ycount < SCREENHEIGHT; ycount++)
      {
        memcpy(destbyte, srcbyte, sizeof(byte)*SCREENWIDTH);
        srcbyte+=SCREENPITCH;
        destbyte+=winscrwidth;
      }

      screen->lpVtbl->Unlock(screen, NULL);
    }
    else
    {
      I_Error("I_UpdateScreen: Unable to Lock DD Surface");
    } 
  }
  else
  {
    // get corner point
    src = (short*)buffer;

    // get surface...
    ddsd.dwSize = sizeof(ddsd);
    screen->lpVtbl->GetSurfaceDesc(screen, &ddsd);
    hresult = screen->lpVtbl->Lock(screen, NULL, &ddsd, DDLOCK_SURFACEMEMORYPTR|DDLOCK_WAIT, NULL);

    if (hresult == DD_OK)
    {
      winscrwidth = ddsd.lPitch/sizeof(short);

      dest = (short*)ddsd.lpSurface;

      if (SCREENWINDOW)
        dest += (topleft_x+(topleft_y*winscrwidth));

      for (ycount=0; ycount < SCREENHEIGHT; ycount++)
      {
        memcpy(dest, src, sizeof(short)*SCREENWIDTH);
        src+=(SCREENPITCH/sizeof(short));
        dest+=winscrwidth;
      }

      screen->lpVtbl->Unlock(screen, NULL);
    }
    else
    {
      I_Error("I_UpdateScreen: Unable to Lock DD Surface");
    }
  }
}

//
// I_SetPalette
//
// -AJA- 1999/07/03: removed `redness' parameter, and moved the 16-bit
//                   handling to I_MakeColourmapRange.
//
void I_SetPalette(byte palette[256][3])
{
  PALETTEENTRY pe[256];
  HRESULT value;
  const byte *gtable = gammatable[usegamma];
  int i;

  if (nogfxmode)
    return;

  for (i = 0; i < 256; i++)
  {
    pe[i].peRed   = gtable[palette[i][0]];
    pe[i].peGreen = gtable[palette[i][1]];
    pe[i].peBlue  = gtable[palette[i][2]];
  }

  // Create the palette
  if (!scrpalette)
  {
    value = drawobj->lpVtbl->CreatePalette(drawobj, DDPCAPS_8BIT|DDPCAPS_ALLOW256|DDPCAPS_INITIALIZE, pe, &scrpalette, NULL);

    if (value != DD_OK)
      I_Error("I_SetPalette: CreatePalette failed\n");
  }
  else
  {  
    scrpalette->lpVtbl->SetEntries(scrpalette, 0, 0, 256, pe);
  }

  if (screen)
  {
    value = screen->lpVtbl->SetPalette(screen, scrpalette);
 
    if (value != DD_OK)
      I_Warning("I_SetPalette: Palette routine has gone pear-shaped\n");
  }
}

//
// I_Colour2Pixel
//
// -AJA- 1999/07/05: added this.
//
long I_Colour2Pixel(byte palette[256][3], int col)
{
  if (BPP > 1)
    return (long)MakeColour(palette[col][0], palette[col][1], palette[col][2]);

  return col;
}

//
// I_SizeWindow
//
void I_SizeWindow(void)
{
  if (SCREENWINDOW)
  {
    SetWindowPos(mainwindow, HWND_TOP, 0, 0, winwidth, winheight,
        SWP_NOMOVE | SWP_SHOWWINDOW);
  }
}

//
// I_ShutdownGraphics
//
void I_ShutdownGraphics(void)
{
  if (drawobj == NULL)
    return;

  if (screen != NULL)
  {
    screen->lpVtbl->Release(screen);
    screen = NULL;
  }

  drawobj->lpVtbl->Release(drawobj);
  drawobj = NULL;
}

#endif // !USE_GL
