//----------------------------------------------------------------------------
//  EDGE Wipe Main
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------

#include "i_defs.h"
#include "wp_main.h"

#include "v_res.h"
#include "z_zone.h"

wipeinfo_t *WIPE_CreateWipeInfo(void)
{
  wipeinfo_t *wi;
  wi = Z_ClearNew(wipeinfo_t, 1);
  return wi;
}

void WIPE_DestroyWipeInfo(wipeinfo_t * wi)
{
  if (wi->parms.data && wi->type->destroy_info)
    wi->type->destroy_info(&wi->parms);

  Z_Free(wi);
}

void WIPE_StopWipe(screen_t * dest, screen_t * start, screen_t * end, wipeinfo_t * wi)
{
#ifndef USE_GL
  if (wi->parms.data && wi->type->destroy_info)
    wi->type->destroy_info(&wi->parms);

  if (dest && end && dest->id == wi->dest.id && end->id == wi->end.id && wi->end.id != wi->dest.id)
  {
    // the dest and end screens are still valid.
    // Output the final screen.
    V_CopyScreen(wi->dest.scr, wi->end.scr);
  }

  V_DestroyScreen(wi->end.scr);
  V_DestroyScreen(wi->start.scr);
  V_DestroyScreen(wi->dest.scr);

  Z_Clear(wi, wipeinfo_t, 1);
#endif
}

// internally used by InitWipe
static void InitWipeScr(wipescr_t * ws, screen_t * scr, int x, int y, int width,
    int height, int isconst, int allow_sharing, screen_t * dest)
{
  boolean_t subscreen = true;

  ws->id = scr->id;

  // hack: dest is set to null when handling dest. Special case,
  // the wi->dest.scr will always be a subscreen.
  if (dest)
  {
    // if the screen intersects with the destination, and
    // we don't have the very special case where scr is exactly dest and
    // this is allowed, then we must use a shadow of the screen.
    if (V_ScreensIntersect(scr, dest) && !(allow_sharing && scr == dest && !isconst))
    {
      subscreen = false;

      if (isconst)
      {
        ws->update_scr = false;
      }
      else
      {
        ws->update_scr = true;
        ws->update_x = x;
        ws->update_y = y;
      }
    }
    else
    {
      subscreen = true;
    }
  }

  if (subscreen)
  {
    ws->scr = V_CreateSubScreen(scr, x, y, width, height);
    ws->update_scr = false;
  }
  else
  {
    ws->scr = V_CreateScreen(width, height, scr->bytepp);
    if (!ws->update_scr)
      // if scr isn't updated each frame, it must be updated once now.
      V_CopyRect(ws->scr, scr, x, y, width, height, 0, 0);
  }
}

// dest/start/end/x/y/width/height: positions of the wiping
// cstart/cend: Tells whether start/end screens are constant.
// duration tells the number of steps the effect should last, pass -1
//  to use the default.
// wi is the wipeinfo that should be used for this effect. It can be a good
//  idea to recycle wipeinfos, and use a static variable for each place where
//  wiping can be used: if one wiping interrupts another one, the new one
//  will kill off the old one if the same wi is used (otherwise everything
//  will look very strange).
// reverse tells whether the effect should be reversed.
// effect shows which effect to use
//
wipeinfo_t *WIPE_InitWipe(screen_t * dest, int destx, int desty,
    screen_t * start, int startx, int starty, int cstart,
    screen_t * end, int endx, int endy, int cend,
    int width, int height, wipeinfo_t * wi,
    int duration, boolean_t reversed, wipetype_e effect)
{
  if (!wi)
    wi = WIPE_CreateWipeInfo();
  if (wi->active)
    WIPE_StopWipe(dest, start, end, wi);

#ifndef USE_GL

  wi->active = true;
  wi->reversed = reversed;

  wi->type = &wipes[effect];

  if (dest->bytepp != start->bytepp || dest->bytepp != end->bytepp)
    I_Error("WIPE_InitWipe: Screens have different BPP!");

  if (reversed)
  {
    InitWipeScr(&wi->start, start, startx, starty, width, height, cstart, wi->type->allow_end_dest, dest);
    InitWipeScr(&wi->end, end, endx, endy, width, height, cend, wi->type->allow_start_dest, dest);
    wi->parms.end = wi->start.scr;
    wi->parms.start = wi->end.scr;
  }
  else
  {
    InitWipeScr(&wi->start, start, startx, starty, width, height, cstart, wi->type->allow_start_dest, dest);
    InitWipeScr(&wi->end, end, endx, endy, width, height, cend, wi->type->allow_end_dest, dest);
    wi->parms.start = wi->start.scr;
    wi->parms.end = wi->end.scr;
  }

  InitWipeScr(&wi->dest, dest, destx, desty, width, height, false, false, NULL);

  wi->width = wi->dest.scr->width;
  wi->height = wi->dest.scr->height;

  // NOTE: duration can depend on wi->height
  if (duration < 0)
    duration = wi->type->def_duration(wi);
  
  wi->duration = duration;

  wi->parms.dest = wi->dest.scr;

  if (wi->type->init_info)
    wi->type->init_info(&wi->parms);
#endif // USE_GL

  return wi;
}

boolean_t WIPE_DoWipe(screen_t * dest, screen_t * start, screen_t * end,
    int progress, wipeinfo_t * wi)
{
#ifdef USE_GL
  return true;
#else
  
  // the wipe must be active
  if (!wi->active)
    return true;

  if (progress < 0)
    progress = 0;

  // we kill the wiping if any of the screens has changed
  // or if the wiping it completed
  // 
  if (!dest || !start || !end || dest->id != wi->dest.id ||
      start->id != wi->start.id || end->id != wi->end.id ||
      progress >= wi->duration)
  {
    WIPE_StopWipe(dest, start, end, wi);
    return true;
  }

  if (wi->reversed)
    wi->parms.progress = ((wi->duration - progress) - 0.5f) / wi->duration;
  else
    wi->parms.progress = (progress + 0.5f) / wi->duration;

  if (wi->start.update_scr)
    V_CopyRect(wi->start.scr, start, wi->start.update_x, wi->start.update_y,
        wi->width, wi->height, 0, 0);
  
  if (wi->end.update_scr)
    V_CopyRect(wi->end.scr, end, wi->end.update_x, wi->end.update_y,
        wi->width, wi->height, 0, 0);

  wi->type->do_wipe(&wi->parms);

  return false;
#endif // USE_GL
}
