/*
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation. NO WARRANTY.
*/

#include "doomstat.h"
#include "m_bbox.h"
#include "r_main.h"
#include "p_maputl.h"
#include "p_map.h"
#include "p_setup.h"

fixed_t P_AproxDistance(fixed_t dx, fixed_t dy)
{
  dx = abs(dx);
  dy = abs(dy);
  if (dx < dy)
    return dx+dy-(dx>>1);
  return dx+dy-(dy>>1);
}

int P_PointOnLineSide(fixed_t x, fixed_t y, line_t *line)
{
  return
    !line->dx ? x <= line->v1->x ? line->dy > 0 : line->dy < 0 :
    !line->dy ? y <= line->v1->y ? line->dx < 0 : line->dx > 0 :
    FixedMul(y-line->v1->y, line->dx>>FRACBITS) >=
    FixedMul(line->dy>>FRACBITS, x-line->v1->x);
}

int P_BoxOnLineSide(fixed_t *tmbox, line_t *ld)
{
  switch (ld->slopetype)
    {
      int p;
    default:
    case ST_HORIZONTAL:
      return
      (tmbox[BOXBOTTOM] > ld->v1->y) == (p = tmbox[BOXTOP] > ld->v1->y) ?
        p ^ (ld->dx < 0) : -1;
    case ST_VERTICAL:
      return
        (tmbox[BOXLEFT] < ld->v1->x) == (p = tmbox[BOXRIGHT] < ld->v1->x) ?
        p ^ (ld->dy < 0) : -1;
    case ST_POSITIVE:
      return
        P_PointOnLineSide(tmbox[BOXRIGHT], tmbox[BOXBOTTOM], ld) ==
        (p = P_PointOnLineSide(tmbox[BOXLEFT], tmbox[BOXTOP], ld)) ? p : -1;
    case ST_NEGATIVE:
      return
        (P_PointOnLineSide(tmbox[BOXLEFT], tmbox[BOXBOTTOM], ld)) ==
        (p = P_PointOnLineSide(tmbox[BOXRIGHT], tmbox[BOXTOP], ld)) ? p : -1;
    }
}

int P_PointOnDivlineSide(fixed_t x, fixed_t y, divline_t *line)
{
  return
    !line->dx ? x <= line->x ? line->dy > 0 : line->dy < 0 :
    !line->dy ? y <= line->y ? line->dx < 0 : line->dx > 0 :
    (line->dy^line->dx^(x -= line->x)^(y -= line->y)) < 0 ? (line->dy^x) < 0 :
    FixedMul(y>>8, line->dx>>8) >= FixedMul(line->dy>>8, x>>8);
}

void P_MakeDivline(line_t *li, divline_t *dl)
{
  dl->x = li->v1->x;
  dl->y = li->v1->y;
  dl->dx = li->dx;
  dl->dy = li->dy;
}

int P_InterceptVector(divline_t *v2, divline_t *v1)
{
  int den = FixedMul(v1->dy>>8, v2->dx) - FixedMul(v1->dx>>8, v2->dy);
  return den ? FixedDiv((FixedMul((v1->x-v2->x)>>8, v1->dy) +
                         FixedMul((v2->y-v1->y)>>8, v1->dx)), den) : 0;
}

fixed_t opentop;
fixed_t openbottom;
fixed_t openrange;
fixed_t lowfloor;

sector_t *openfrontsector;
sector_t *openbacksector;

void P_LineOpening(line_t *linedef,int in3dfloor)
{
  if (linedef->sidenum[1] == -1)
    {
      openrange = 0;
      return;
    }

  openfrontsector = linedef->frontsector;
  openbacksector = linedef->backsector;

  if (openfrontsector->ceilingheight < openbacksector->ceilingheight)
    opentop = openfrontsector->ceilingheight;
  else
    opentop = openbacksector->ceilingheight;

  if (openfrontsector->floorheight > openbacksector->floorheight)
    {
      openbottom = openfrontsector->floorheight;
      lowfloor = openbacksector->floorheight;
    }
  else
    {
      openbottom = openbacksector->floorheight;
      lowfloor = openfrontsector->floorheight;
    }

    if(in3dfloor)
    {
      extern mobj_t* tmthing;

      ffloor_t* rover;
      fixed_t lowestceiling = opentop;
      fixed_t highestfloor = openbottom;
      fixed_t lowestfloor = lowfloor;
      fixed_t delta1,delta2;
      int     thingtop = tmthing->z + tmthing->height;

      if(openfrontsector->ffloors)
        for(rover = openfrontsector->ffloors; rover; rover = rover->next)
        {
          if(rover->liquid)continue;
          delta1 = tmthing->z - (*rover->bottomheight+*rover->topheight)/2;
          delta2 = thingtop - (*rover->bottomheight+*rover->topheight)/2;
          if(*rover->bottomheight < lowestceiling && abs(delta1) >= abs(delta2))
            lowestceiling = *rover->bottomheight;
          if(*rover->topheight > highestfloor && abs(delta1) < abs(delta2))
            highestfloor = *rover->topheight;
          else if(*rover->topheight > lowestfloor && abs(delta1) < abs(delta2))
            lowestfloor = *rover->topheight;
        }

      if(openbacksector->ffloors)
        for(rover = openbacksector->ffloors; rover; rover = rover->next)
        {
          if(rover->liquid)continue;
          delta1 = tmthing->z - (*rover->bottomheight+*rover->topheight)/2;
          delta2 = thingtop - (*rover->bottomheight+*rover->topheight)/2;
          if(*rover->bottomheight < lowestceiling && abs(delta1) >= abs(delta2))
            lowestceiling = *rover->bottomheight;
          if(*rover->topheight > highestfloor && abs(delta1) < abs(delta2))
            highestfloor = *rover->topheight;
          else if(*rover->topheight > lowestfloor && abs(delta1) < abs(delta2))
            lowestfloor = *rover->topheight;
        }
      if(highestfloor > openbottom)
        openbottom = highestfloor;
      if(lowestceiling < opentop)
        opentop = lowestceiling;
      if(lowestfloor > lowfloor)
        lowfloor = lowestfloor;
    }

  openrange = opentop - openbottom;
}

void P_UnsetThingPosition (mobj_t *thing)
{
  if (!(thing->flags & MF_NOSECTOR))
    {
      mobj_t **sprev = thing->sprev;
      mobj_t  *snext = thing->snext;
      if ((*sprev = snext))
	snext->sprev = sprev;

      sector_list = thing->touching_sectorlist;
      thing->touching_sectorlist = NULL;
    }

  if (!(thing->flags & MF_NOBLOCKMAP))
    {
      mobj_t *bnext, **bprev = thing->bprev;
      if (bprev && (*bprev = bnext = thing->bnext))
	bnext->bprev = bprev;
    }
}

void P_SetThingPosition(mobj_t *thing)
{
 subsector_t *ss = thing->subsector = R_PointInSubsector(thing->x, thing->y);

  if (!(thing->flags & MF_NOSECTOR))
    {
      mobj_t **link = &ss->sector->thinglist;
      mobj_t *snext = *link;
      if ((thing->snext = snext))
	snext->sprev = &thing->snext;
      thing->sprev = link;
      *link = thing;
      P_CreateSecNodeList(thing, thing->x, thing->y);
      thing->touching_sectorlist = sector_list;
      sector_list = NULL;
    }

  if (!(thing->flags & MF_NOBLOCKMAP))
    {
      int blockx = (thing->x - bmaporgx)>>MAPBLOCKSHIFT;
      int blocky = (thing->y - bmaporgy)>>MAPBLOCKSHIFT;

      if (blockx>=0 && blockx < bmapwidth && blocky>=0 && blocky < bmapheight)
        {
	  mobj_t **link = &blocklinks[blocky*bmapwidth+blockx];
	  mobj_t *bnext = *link;
          if ((thing->bnext = bnext))
	    bnext->bprev = &thing->bnext;
	  thing->bprev = link;
          *link = thing;
        }
      else
        thing->bnext = NULL, thing->bprev = NULL;
    }
}

boolean ThingIsOnLine(mobj_t *t, line_t *l)
{
  int dx = l->dx >> FRACBITS;
  int dy = l->dy >> FRACBITS;
  int a = (l->v1->x >> FRACBITS) - (t->x >> FRACBITS);
  int b = (l->v1->y >> FRACBITS) - (t->y >> FRACBITS);
  int r = t->radius >> FRACBITS;

  if (abs(a*2+dx)-abs(dx) > r*2 || abs(b*2+dy)-abs(dy) > r*2)
    return 0;

  a *= dy;
  b *= dx;
  a -= b;
  b = dx + dy;
  b *= r;
  if (((a-b)^(a+b)) < 0)
    return 1;
  dy -= dx;
  dy *= r;
  b = a+dy;
  a -= dy;
  return (a^b) < 0;
}

boolean P_BlockLinesIterator(int x, int y, boolean func(line_t*))
{
  int        offset;
  const long *list;

  if (x<0 || y<0 || x>=bmapwidth || y>=bmapheight)
    return true;
  offset = y*bmapwidth+x;
  offset = *(blockmap+offset);
  list = blockmaplump+offset;

  if (!demo_compatibility)
    list++;
  for ( ; *list != -1 ; list++)
    {
      line_t *ld = &lines[*list];
      if (ld->validcount == validcount)
        continue;
      ld->validcount = validcount;
      if (!func(ld))
        return false;
    }
  return true;
}

boolean P_BlockThingsIterator(int x, int y, boolean func(mobj_t*))
{
  mobj_t *mobj;
  if (!(x<0 || y<0 || x>=bmapwidth || y>=bmapheight))
    for (mobj = blocklinks[y*bmapwidth+x]; mobj; mobj = mobj->bnext)
      if (!func(mobj))
        return false;
  return true;
}

static intercept_t *intercepts, *intercept_p;

static void check_intercept(void)
{
  static size_t num_intercepts;
  size_t offset = intercept_p - intercepts;
  if (offset >= num_intercepts)
    {
      num_intercepts = num_intercepts ? num_intercepts*2 : 128;
      intercepts = realloc(intercepts, sizeof(*intercepts)*num_intercepts);
      intercept_p = intercepts + offset;
    }
}

divline_t trace;

boolean PIT_AddLineIntercepts(line_t *ld)
{
  int       s1;
  int       s2;
  fixed_t   frac;
  divline_t dl;

  if (trace.dx >  FRACUNIT*16 || trace.dy >  FRACUNIT*16 ||
      trace.dx < -FRACUNIT*16 || trace.dy < -FRACUNIT*16)
    {
      s1 = P_PointOnDivlineSide (ld->v1->x, ld->v1->y, &trace);
      s2 = P_PointOnDivlineSide (ld->v2->x, ld->v2->y, &trace);
    }
  else
    {
      s1 = P_PointOnLineSide (trace.x, trace.y, ld);
      s2 = P_PointOnLineSide (trace.x+trace.dx, trace.y+trace.dy, ld);
    }

  if (s1 == s2)
    return true;

  P_MakeDivline(ld, &dl);
  frac = P_InterceptVector(&trace, &dl);

  if (frac < 0)
    return true;

  check_intercept();

  intercept_p->frac = frac;
  intercept_p->isaline = true;
  intercept_p->d.line = ld;
  intercept_p++;

  return true;
}

boolean PIT_AddThingIntercepts(mobj_t *thing)
{
  fixed_t   x1, y1;
  fixed_t   x2, y2;
  int       s1, s2;
  divline_t dl;
  fixed_t   frac;

  if ((trace.dx ^ trace.dy) > 0)
    {
      x1 = thing->x - thing->radius;
      y1 = thing->y + thing->radius;
      x2 = thing->x + thing->radius;
      y2 = thing->y - thing->radius;
    }
  else
    {
      x1 = thing->x - thing->radius;
      y1 = thing->y - thing->radius;
      x2 = thing->x + thing->radius;
      y2 = thing->y + thing->radius;
    }

  s1 = P_PointOnDivlineSide (x1, y1, &trace);
  s2 = P_PointOnDivlineSide (x2, y2, &trace);

  if (s1 == s2)
    return true;

  dl.x = x1;
  dl.y = y1;
  dl.dx = x2-x1;
  dl.dy = y2-y1;

  frac = P_InterceptVector (&trace, &dl);

  if (frac < 0)
    return true;

  check_intercept();

  intercept_p->frac = frac;
  intercept_p->isaline = false;
  intercept_p->d.thing = thing;
  intercept_p++;

  return true;
}

boolean P_TraverseIntercepts(traverser_t func, fixed_t maxfrac)
{
  intercept_t *in = NULL;
  int count = intercept_p - intercepts;
  while (count--)
    {
      fixed_t dist = MAXINT;
      intercept_t *scan;
      for (scan = intercepts; scan < intercept_p; scan++)
        if (scan->frac < dist)
          dist = (in=scan)->frac;
      if (dist > maxfrac)
        return true;
      if (!func(in))
        return false;
      in->frac = MAXINT;
    }
  return true;
}

boolean P_PathTraverse(long long x1, long long y1, long long x2, long long y2,
                       int flags, boolean trav(intercept_t *))
{
  int xt1, yt1, xt2, yt2, xstep, ystep, partial, xintercept, yintercept;
  int     mapx, mapy;
  int     mapxstep, mapystep;
  int     count;

  validcount++;
  intercept_p = intercepts;

  if (!((x1-bmaporgx)&(MAPBLOCKSIZE-1)))
    x1 += FRACUNIT;
  if (!((y1-bmaporgy)&(MAPBLOCKSIZE-1)))
    y1 += FRACUNIT;

  trace.x = x1;
  trace.y = y1;
  trace.dx = x2 - x1;
  trace.dy = y2 - y1;

  x1 -= bmaporgx;
  y1 -= bmaporgy;
  xt1 = x1>>MAPBLOCKSHIFT;
  yt1 = y1>>MAPBLOCKSHIFT;

  x2 -= bmaporgx;
  y2 -= bmaporgy;
  xt2 = x2>>MAPBLOCKSHIFT;
  yt2 = y2>>MAPBLOCKSHIFT;

  if (xt2 > xt1)
    {
      mapxstep = 1;
      partial = FRACUNIT - ((x1>>MAPBTOFRAC)&(FRACUNIT-1));
      ystep = FixedDiv (y2-y1,abs(x2-x1));
    }
  else
    if (xt2 < xt1)
      {
        mapxstep = -1;
        partial = (x1>>MAPBTOFRAC)&(FRACUNIT-1);
        ystep = FixedDiv (y2-y1,abs(x2-x1));
      }
    else
      {
        mapxstep = 0;
        partial = FRACUNIT;
        ystep = 256*FRACUNIT;
      }

  yintercept = (y1>>MAPBTOFRAC) + FixedMul(partial, ystep);

  if (yt2 > yt1)
    {
      mapystep = 1;
      partial = FRACUNIT - ((y1>>MAPBTOFRAC)&(FRACUNIT-1));
      xstep = FixedDiv (x2-x1,abs(y2-y1));
    }
  else
    if (yt2 < yt1)
      {
        mapystep = -1;
        partial = (y1>>MAPBTOFRAC)&(FRACUNIT-1);
        xstep = FixedDiv (x2-x1,abs(y2-y1));
      }
    else
      {
        mapystep = 0;
        partial = FRACUNIT;
        xstep = 256*FRACUNIT;
      }

  xintercept = (x1>>MAPBTOFRAC) + FixedMul (partial, xstep);

  mapx = xt1;
  mapy = yt1;

  for (count = 0; count < 64; count++)
    {
      if (flags & PT_ADDLINES)
        if (!P_BlockLinesIterator(mapx, mapy,PIT_AddLineIntercepts))
          return false;
      if (flags & PT_ADDTHINGS)
        if (!P_BlockThingsIterator(mapx, mapy,PIT_AddThingIntercepts))
          return false;
      if (mapx == xt2 && mapy == yt2)
        break;

      if ((yintercept >> FRACBITS) == mapy)
        {
          yintercept += ystep;
          mapx += mapxstep;
        }
      else
        if ((xintercept >> FRACBITS) == mapx)
          {
            xintercept += xstep;
            mapy += mapystep;
          }
    }
  return P_TraverseIntercepts(trav, FRACUNIT);
}