/*
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation. NO WARRANTY.
*/

#include "doomdef.h"
#include "doomstat.h"
#include "m_random.h"
#include "r_main.h"
#include "p_maputl.h"
#include "p_map.h"
#include "p_tick.h"
#include "sounds.h"
#include "st_stuff.h"
#include "info.h"
#include "g_game.h"
#include "p_inter.h"

void P_RemoveMobj (mobj_t *mobj)
{
  if (!((mobj->flags ^ MF_SPECIAL) & (MF_SPECIAL | MF_DROPPED))
      && mobj->type != MT_INV && mobj->type != MT_INS)
    {
      itemrespawnque[iquehead] = mobj->spawnpoint;
      itemrespawntime[iquehead++] = leveltime;
      if ((iquehead &= ITEMQUESIZE-1) == iquetail)
	iquetail = (iquetail+1)&(ITEMQUESIZE-1);
    }

  P_UnsetThingPosition (mobj);

  if (sector_list)
    {
      P_DelSeclist(sector_list);
      sector_list = NULL;
    }

  S_StopSound (mobj);

  if (demo_version >= 203)
    {
      P_SetTarget(&mobj->target,    NULL);
      P_SetTarget(&mobj->tracer,    NULL);
      P_SetTarget(&mobj->lastenemy, NULL);
    }

  P_RemoveThinker(&mobj->thinker);
}

boolean P_SetMobjState(mobj_t* mobj,statenum_t state)
{
  state_t*  st;

  static statenum_t seenstate_tab[NUMSTATES];
  statenum_t *seenstate = seenstate_tab;
  static int recursion;
  statenum_t i = state;
  boolean ret = true;
  statenum_t tempstate[NUMSTATES];

  if (recursion++)
    memset(seenstate=tempstate,0,sizeof tempstate);

  do
    {
      if (state == S_NULL)
	{
	  mobj->state = (state_t *) S_NULL;
	  P_RemoveMobj (mobj);
	  ret = false;
	  break;
	}

      st = &states[state];
      mobj->state = st;
      mobj->tics = st->tics;
      mobj->sprite = st->sprite;
      mobj->frame = st->frame;

      if (st->action)
	st->action(mobj);

      seenstate[state] = 1 + st->nextstate;
      state = st->nextstate;
    } 
  while (!mobj->tics && !seenstate[state]);
  if (ret && !mobj->tics)
    dprintf("Warning: State Cycle Detected");
  if (!--recursion)
    for (;(state=seenstate[i]);i=state-1)
      seenstate[i] = 0;
  return ret;
}

void P_ExplodeMissile (mobj_t* mo)
{
  mo->momx = mo->momy = mo->momz = 0;

  P_SetMobjState(mo, mobjinfo[mo->type].deathstate);

  mo->tics -= P_Random(pr_explode)&3;

  if (mo->tics < 1)
    mo->tics = 1;

  mo->flags &= ~MF_MISSILE;

  if (mo->info->deathsound)
    S_StartSound (mo, mo->info->deathsound);
}

void P_XYMovement (mobj_t* mo)
{
  player_t *player;
  fixed_t xmove, ymove;

  if (!(mo->momx | mo->momy))
    {
      if (mo->flags & MF_SKULLFLY)
	{
	  mo->flags &= ~MF_SKULLFLY;
	  mo->momz = 0;

	  P_SetMobjState(mo, mo->info->spawnstate);
	}
      return;
    }

  player = mo->player;

  if (mo->momx > MAXMOVE)
    mo->momx = MAXMOVE;
  else 
    if (mo->momx < -MAXMOVE)
      mo->momx = -MAXMOVE;

  if (mo->momy > MAXMOVE)
    mo->momy = MAXMOVE;
  else
    if (mo->momy < -MAXMOVE)
      mo->momy = -MAXMOVE;

  xmove = mo->momx;
  ymove = mo->momy;

  do
    {
      fixed_t ptryx, ptryy;

      if (xmove > MAXMOVE/2 || ymove > MAXMOVE/2 ||
	  ((xmove < -MAXMOVE/2 || ymove < -MAXMOVE/2) && demo_version >= 203))
	{
	  ptryx = mo->x + xmove/2;
	  ptryy = mo->y + ymove/2;
	  xmove >>= 1;
	  ymove >>= 1;
	}
      else
	{
	  ptryx = mo->x + xmove;
	  ptryy = mo->y + ymove;
	  xmove = ymove = 0;
	}

      if (!P_TryMove(mo, ptryx, ptryy, true))
	{
	  
	  if (!(mo->flags & MF_MISSILE) && demo_version >= 203 &&
	      (mo->flags & MF_BOUNCES || 
	       (!player && blockline &&
		variable_friction && mo->z <= mo->floorz &&
		P_GetFriction(mo, NULL) > ORIG_FRICTION)))
	    {
	      if (blockline)
		{
		  fixed_t r = ((blockline->dx >> FRACBITS) * mo->momx +
			       (blockline->dy >> FRACBITS) * mo->momy) /
		    ((blockline->dx >> FRACBITS)*(blockline->dx >> FRACBITS)+
		     (blockline->dy >> FRACBITS)*(blockline->dy >> FRACBITS));
		  fixed_t x = FixedMul(r, blockline->dx);
		  fixed_t y = FixedMul(r, blockline->dy);

		  mo->momx = x*2 - mo->momx;
		  mo->momy = y*2 - mo->momy;

		  if (!(mo->flags & MF_NOGRAVITY))
		    {
		      mo->momx = (mo->momx + x)/2;
		      mo->momy = (mo->momy + y)/2;
		    }
		}
	      else
		mo->momx = mo->momy = 0;
	    }
	  else
	    if (player)
	      P_SlideMove (mo);
	    else 
	      if (mo->flags & MF_MISSILE)
		{
		  if (ceilingline &&
		      ceilingline->backsector &&
		      ceilingline->backsector->ceilingpic == skyflatnum)
		    if (demo_compatibility ||
			mo->z > ceilingline->backsector->ceilingheight)
		      {
			P_RemoveMobj (mo);
			return;
		      }
		  P_ExplodeMissile (mo);
		}
	      else
		mo->momx = mo->momy = 0;
	}
    }
  while (xmove | ymove);

  if (mo->flags & (MF_MISSILE | MF_SKULLFLY) || mo->z > mo->floorz)
    return;

  if (((mo->flags & MF_BOUNCES && mo->z > mo->dropoffz) ||
       mo->flags & MF_CORPSE || mo->intflags & MIF_FALLING) &&
      (mo->momx > FRACUNIT/4 || mo->momx < -FRACUNIT/4 ||
       mo->momy > FRACUNIT/4 || mo->momy < -FRACUNIT/4) &&
      mo->floorz != mo->subsector->sector->floorheight)
    return;

  if (mo->momx > -STOPSPEED && mo->momx < STOPSPEED &&
      mo->momy > -STOPSPEED && mo->momy < STOPSPEED &&
      (!player || !(player->cmd.forwardmove | player->cmd.sidemove) ||
       (player->mo != mo && demo_version >= 203)))
    {
      if (player && (unsigned)(player->mo->state - states - S_PLAY_RUN1) < 4 
	  && (player->mo == mo || demo_version < 203))
	P_SetMobjState(player->mo, S_PLAY);

      mo->momx = mo->momy = 0;
      
      if (player && player->mo == mo)
	player->momx = player->momy = 0; 
    }
  else
    {
      fixed_t friction = P_GetFriction(mo, NULL);

      mo->momx = FixedMul(mo->momx, friction);
      mo->momy = FixedMul(mo->momy, friction);

      if (player && player->mo == mo)
	{
	  player->momx = FixedMul(player->momx, ORIG_FRICTION);
	  player->momy = FixedMul(player->momy, ORIG_FRICTION);
	}
    }
}

static void P_ZMovement (mobj_t* mo)
{
  if (mo->flags & MF_BOUNCES && mo->momz) {
    mo->z += mo->momz;
    if (mo->z <= mo->floorz) {
      mo->z = mo->floorz;
      if (mo->momz < 0) {
        mo->momz = -mo->momz;
  if (!(mo->flags & MF_NOGRAVITY)) {
    mo->momz = mo->flags & MF_FLOAT ?
      mo->flags & MF_DROPOFF ?
      FixedMul(mo->momz, (fixed_t)(FRACUNIT*.85)) :
      FixedMul(mo->momz, (fixed_t)(FRACUNIT*.70)) :
      FixedMul(mo->momz, (fixed_t)(FRACUNIT*.45)) ;

    if (abs(mo->momz) <= mo->info->mass*(GRAVITY*4/256))
      mo->momz = 0;
  }

  if (mo->flags & MF_TOUCHY && mo->intflags & MIF_ARMED
      && mo->health > 0)
    P_DamageMobj(mo, NULL, NULL, mo->health);
  else if (mo->flags & MF_FLOAT && sentient(mo))
    goto floater;
  return;
      }
    } else if (mo->z >= mo->ceilingz - mo->height) {
      mo->z = mo->ceilingz - mo->height;
      if (mo->momz > 0) {
  if (mo->subsector->sector->ceilingpic != skyflatnum)
    mo->momz = -mo->momz;
  else if (mo->flags & MF_MISSILE)
    P_RemoveMobj(mo);
  else if (mo->flags & MF_NOGRAVITY)
    mo->momz = -mo->momz;
  if (mo->flags & MF_FLOAT && sentient(mo))
    goto floater;

  return;
      }
    } else {
      if (!(mo->flags & MF_NOGRAVITY))
        mo->momz -= mo->info->mass*(GRAVITY/256);

      if (mo->flags & MF_FLOAT && sentient(mo)) goto floater;
      return;
    }

    mo->momz = 0;

    if (mo->flags & MF_MISSILE) {
  if (ceilingline &&
      ceilingline->backsector &&
      ceilingline->backsector->ceilingpic == skyflatnum &&
      mo->z > ceilingline->backsector->ceilingheight)
    P_RemoveMobj(mo);
  else
    P_ExplodeMissile(mo);
    }

    if (mo->flags & MF_FLOAT && sentient(mo)) goto floater;
    return;
  }

  if (mo->player && mo->player->mo == mo && mo->z < mo->floorz)
    {
    mo->player->viewheight -= mo->floorz-mo->z;
    mo->player->deltaviewheight = (VIEWHEIGHT - mo->player->viewheight)>>3;
    }

  mo->z += mo->momz;

floater:
  if ((mo->flags & MF_FLOAT) && mo->target)

    if (!((mo->flags ^ MF_FLOAT) & (MF_FLOAT | MF_SKULLFLY | MF_INFLOAT)) &&
  mo->target)
      {
  fixed_t delta;
  if (P_AproxDistance(mo->x - mo->target->x, mo->y - mo->target->y) <
      abs(delta = mo->target->z + (mo->height>>1) - mo->z)*3)
    mo->z += delta < 0 ? -FLOATSPEED : FLOATSPEED;
      }

  if (mo->z <= mo->floorz)
    {
    int correct_lost_soul_bounce = !demo_compatibility || (gamemission != doom2);

    if (correct_lost_soul_bounce && mo->flags & MF_SKULLFLY)
      mo->momz = -mo->momz;

    if (mo->momz < 0)
      {
  if (mo->flags & MF_TOUCHY && mo->intflags & MIF_ARMED && mo->health > 0)
    P_DamageMobj(mo, NULL, NULL, mo->health);
  else
    if (mo->player && mo->player->mo == mo && mo->momz < -GRAVITY*8)
      {
        mo->player->deltaviewheight = mo->momz>>3;
        if (mo->health) S_StartSound (mo, sfx_oof);
      }
    mo->momz = 0;
      }
    mo->z = mo->floorz;

    if (!correct_lost_soul_bounce && mo->flags & MF_SKULLFLY)
      mo->momz = -mo->momz;

    if ( (mo->flags & MF_MISSILE) && !(mo->flags & MF_NOCLIP) )
      {
      P_ExplodeMissile (mo);
      return;
      }
    }
  else
    if (!(mo->flags & MF_NOGRAVITY))
      {
  if (!mo->momz)
    mo->momz = -GRAVITY;
        mo->momz -= GRAVITY;
      }

  if (mo->z + mo->height > mo->ceilingz)
    {

    if (mo->momz > 0)
      mo->momz = 0;

    mo->z = mo->ceilingz - mo->height;

    if (mo->flags & MF_SKULLFLY)
      mo->momz = -mo->momz;

    if ( (mo->flags & MF_MISSILE) && !(mo->flags & MF_NOCLIP) )
      {
      P_ExplodeMissile (mo);
      return;
      }
    }
}

void P_NightmareRespawn(mobj_t* mobj)
{
  fixed_t      x;
  fixed_t      y;
  fixed_t      z;
  subsector_t* ss;
  mobj_t*      mo;
  mapthing_t*  mthing;

  x = mobj->spawnpoint.x << FRACBITS;
  y = mobj->spawnpoint.y << FRACBITS;

  if (!P_CheckPosition (mobj, x, y))
    return;

  mo = P_SpawnMobj(mobj->x, mobj->y,
		   mobj->subsector->sector->floorheight, MT_TFOG);

  S_StartSound (mo, sfx_telept);

  ss = R_PointInSubsector (x,y);

  mo = P_SpawnMobj (x, y, ss->sector->floorheight , MT_TFOG);

  S_StartSound (mo, sfx_telept);

  mthing = &mobj->spawnpoint;
  z = mobj->info->flags & MF_SPAWNCEILING ? ONCEILINGZ : ONFLOORZ;

  mo = P_SpawnMobj (x,y,z, mobj->type);
  mo->spawnpoint = mobj->spawnpoint;
  mo->angle = ANG45 * (mthing->angle/45);

  if (mthing->options & MTF_AMBUSH)
    mo->flags |= MF_AMBUSH;

  mo->flags = (mo->flags & ~MF_FRIEND) | (mobj->flags & MF_FRIEND);

  mo->reactiontime = 18;

  P_RemoveMobj (mobj);
}

void P_MobjThinker (mobj_t* mobj)
{
  if (mobj->momx | mobj->momy || mobj->flags & MF_SKULLFLY)
    {
      P_XYMovement(mobj);
      if (mobj->thinker.function == P_RemoveThinkerDelayed)
	return;
    } else
  if(mobj->subsector->sector->ffloors) P_TryMove(mobj,mobj->x,mobj->y,1);
  if (mobj->z != mobj->floorz || mobj->momz)
    {
      P_ZMovement(mobj);
      if (mobj->thinker.function == P_RemoveThinkerDelayed)
	return;
    }
  else
    if (!(mobj->momx | mobj->momy) && !sentient(mobj))
      {
	mobj->intflags |= MIF_ARMED;
	if (mobj->z > mobj->dropoffz &&
	    !(mobj->flags & MF_NOGRAVITY) &&
	    !comp[comp_falloff] && demo_version >= 203)
	  P_ApplyTorque(mobj);
	else
	  mobj->intflags &= ~MIF_FALLING, mobj->gear = 0;
      }

  if (mobj->tics != -1)
    {
      if (!--mobj->tics)
	P_SetMobjState(mobj, mobj->state->nextstate);
    }
  else                       
    if (mobj->flags & MF_COUNTKILL && respawnmonsters &&
	++mobj->movecount >= 12*35 && !(leveltime & 31) &&
	P_Random (pr_respawn) <= 4)
      P_NightmareRespawn(mobj);
}

mobj_t *P_SpawnMobj(fixed_t x, fixed_t y, fixed_t z, mobjtype_t type)
{
  mobj_t *mobj = Z_Malloc(sizeof *mobj, PU_LEVEL, NULL);
  mobjinfo_t *info = &mobjinfo[type];
  state_t    *st;

  memset(mobj, 0, sizeof *mobj);

  mobj->type = type;
  mobj->info = info;
  mobj->x = x;
  mobj->y = y;
  mobj->radius = info->radius;
  mobj->height = info->height;
  mobj->flags  = info->flags;

  if (demo_version < 203)
    mobj->flags &= ~(MF_BOUNCES | MF_FRIEND | MF_TOUCHY); 
  else
    if (type == MT_PLAYER)
      mobj->flags |= MF_FRIEND;

  mobj->health = info->spawnhealth;

  if (gameskill != sk_nightmare)
    mobj->reactiontime = info->reactiontime;

  mobj->lastlook = P_Random (pr_lastlook) % MAXPLAYERS;

  st = &states[info->spawnstate];

  mobj->state  = st;
  mobj->tics   = st->tics;
  mobj->sprite = st->sprite;
  mobj->frame  = st->frame;

  mobj->touching_sectorlist = NULL;

  P_SetThingPosition(mobj);

  mobj->dropoffz = mobj->floorz = mobj->subsector->sector->floorheight;
  mobj->ceilingz = mobj->subsector->sector->ceilingheight;
  
  mobj->z = z == ONFLOORZ ? mobj->floorz : z == ONCEILINGZ ?
  mobj->ceilingz - mobj->height : z;

  mobj->thinker.function = P_MobjThinker;
  mobj->above_thing = mobj->below_thing = 0;

  P_AddThinker(&mobj->thinker);

  return mobj;
}

static mapthing_t itemrespawnque[ITEMQUESIZE];
static int itemrespawntime[ITEMQUESIZE];
int iquehead, iquetail;

int P_FindDoomedNum(unsigned type)
{
  static struct { int first, next; } *hash;
  register int i;

  if (!hash)
    {
      hash = Z_Malloc(sizeof *hash * NUMMOBJTYPES, PU_CACHE, (void **) &hash);
      for (i=0; i<NUMMOBJTYPES; i++)
	hash[i].first = NUMMOBJTYPES;
      for (i=0; i<NUMMOBJTYPES; i++)
	if (mobjinfo[i].doomednum != -1)
	  {
	    unsigned h = (unsigned) mobjinfo[i].doomednum % NUMMOBJTYPES;
	    hash[i].next = hash[h].first;
	    hash[h].first = i;
	  }
    }
  
  i = hash[type % NUMMOBJTYPES].first;
  while (i < NUMMOBJTYPES && mobjinfo[i].doomednum != type)
    i = hash[i].next;
  return i;
}

void P_RespawnSpecials (void)
{
  fixed_t x, y, z;
  subsector_t*  ss;
  mobj_t*       mo;
  mapthing_t*   mthing;
  int           i;

  if (deathmatch != 2 ||
      iquehead == iquetail ||
      leveltime - itemrespawntime[iquetail] < 30*35)
    return;
  
  mthing = &itemrespawnque[iquetail];

  x = mthing->x << FRACBITS;
  y = mthing->y << FRACBITS;

  ss = R_PointInSubsector (x,y);
  mo = P_SpawnMobj(x, y, ss->sector->floorheight , MT_IFOG);
  S_StartSound(mo, sfx_itmbk);

  i = P_FindDoomedNum(mthing->type);

  z = mobjinfo[i].flags & MF_SPAWNCEILING ? ONCEILINGZ : ONFLOORZ;

  mo = P_SpawnMobj(x,y,z, i);
  mo->spawnpoint = *mthing;
  mo->angle = ANG45 * (mthing->angle/45);

  iquetail = (iquetail+1)&(ITEMQUESIZE-1);
}

void P_SpawnPlayer (mapthing_t* mthing)
{
  player_t* p;
  fixed_t   x, y, z;
  mobj_t*   mobj;
  int       i;

  if (!playeringame[mthing->type-1])
    return;

  p = &players[mthing->type-1];

  if (p->playerstate == PST_REBORN)
    G_PlayerReborn (mthing->type-1);

  x    = mthing->x << FRACBITS;
  y    = mthing->y << FRACBITS;
  z    = ONFLOORZ;
  mobj = P_SpawnMobj (x,y,z, MT_PLAYER);

  if (mthing->type > 1)
    mobj->flags |= (mthing->type-1)<<MF_TRANSSHIFT;
  
  mobj->angle      = ANG45 * (mthing->angle/45);
  mobj->player     = p;
  mobj->health     = p->health;

  p->mo            = mobj;
  p->playerstate   = PST_LIVE;
  p->refire        = 0;
  p->message       = NULL;
  p->damagecount   = 0;
  p->bonuscount    = 0;
  p->extralight    = 0;
  p->fixedcolormap = 0;
  p->viewheight    = VIEWHEIGHT;

  p->momx = p->momy = 0;
  
  P_SetupPsprites (p);

  if (deathmatch)
    for (i = 0 ; i < NUMCARDS ; i++)
      p->cards[i] = true;

  if (mthing->type-1 == consoleplayer)
    {
      ST_Start();
      HU_Start();
    }
}

void P_SpawnMapThing (mapthing_t* mthing)
{
  int    i;
  mobj_t *mobj;
  fixed_t x, y, z;

  if (demo_compatibility || 
      (demo_version >= 203 && mthing->options & MTF_RESERVED))
    mthing->options &= MTF_EASY|MTF_NORMAL|MTF_HARD|MTF_AMBUSH|MTF_NOTSINGLE;

  if (mthing->type == 11)
    {
      size_t offset = deathmatch_p - deathmatchstarts;

      if (offset >= num_deathmatchstarts)
	{
	  num_deathmatchstarts = num_deathmatchstarts ?
	    num_deathmatchstarts*2 : 16;
	  deathmatchstarts = realloc(deathmatchstarts,
				     num_deathmatchstarts *
				     sizeof(*deathmatchstarts));
	  deathmatch_p = deathmatchstarts + offset;
	}
      memcpy(deathmatch_p++, mthing, sizeof*mthing);
      return;
    }

  if (mthing->type <= 4 && mthing->type > 0)
    {
      playerstarts[mthing->type-1] = *mthing;
      if (!deathmatch)
	P_SpawnPlayer (mthing);
      return;
    }

  if (!netgame && mthing->options & MTF_NOTSINGLE)
    return;
  if (netgame && deathmatch && mthing->options & MTF_NOTDM)
    return;
  if (netgame && !deathmatch && mthing->options & MTF_NOTCOOP)
    return;
  if (gameskill == sk_baby || gameskill == sk_easy ? 
      !(mthing->options & MTF_EASY) :
      gameskill == sk_hard || gameskill == sk_nightmare ?
      !(mthing->options & MTF_HARD) : !(mthing->options & MTF_NORMAL))
    return;

  i = P_FindDoomedNum(mthing->type);

  if (i == NUMMOBJTYPES)
    {
      dprintf("Unknown Thing type %i at (%i, %i)",
	      mthing->type, mthing->x, mthing->y);
      return;
    }
  if (deathmatch && mobjinfo[i].flags & MF_NOTDMATCH)
    return;

  if (nomonsters && (i == MT_SKULL || (mobjinfo[i].flags & MF_COUNTKILL)))
    return;

  x = mthing->x << FRACBITS;
  y = mthing->y << FRACBITS;

  z = mobjinfo[i].flags & MF_SPAWNCEILING ? ONCEILINGZ : ONFLOORZ;

  mobj = P_SpawnMobj (x,y,z, i);
  mobj->spawnpoint = *mthing;

  if (mobj->tics > 0)
    mobj->tics = 1 + (P_Random (pr_spawnthing) % mobj->tics);

  if (!(mobj->flags & MF_FRIEND) &&
      mthing->options & MTF_FRIEND && 
      demo_version>=203)
    {
      mobj->flags |= MF_FRIEND;
      P_UpdateThinker(&mobj->thinker);
    }

  if (!((mobj->flags ^ MF_COUNTKILL) & (MF_FRIEND | MF_COUNTKILL)))
    totalkills++;

  if (mobj->flags & MF_COUNTITEM)
    totalitems++;

  mobj->angle = ANG45 * (mthing->angle/45);
  if (mthing->options & MTF_AMBUSH)
    mobj->flags |= MF_AMBUSH;
}

extern fixed_t attackrange;

void P_SpawnPuff(fixed_t x,fixed_t y,fixed_t z)
{
  mobj_t* th;
  int t = P_Random(pr_spawnpuff);
  z += (t - P_Random(pr_spawnpuff))<<10;

  th = P_SpawnMobj (x,y,z, MT_PUFF);
  th->momz = FRACUNIT;
  th->tics -= P_Random(pr_spawnpuff)&3;

  if (th->tics < 1)
    th->tics = 1;

  if (attackrange == MELEERANGE)
    P_SetMobjState (th, S_PUFF3);
}

void P_SpawnBlood(fixed_t x,fixed_t y,fixed_t z,int damage)
{
  mobj_t* th;
  int t = P_Random(pr_spawnblood);
  z += (t - P_Random(pr_spawnblood))<<10;
  th = P_SpawnMobj(x,y,z, MT_BLOOD);
  th->momz = FRACUNIT*2;
  th->tics -= P_Random(pr_spawnblood)&3;

  if (th->tics < 1)
    th->tics = 1;

  if (damage <= 12 && damage >= 9)
    P_SetMobjState(th,S_BLOOD2);
  else
    if (damage < 9)
      P_SetMobjState(th,S_BLOOD3);
}

void P_CheckMissileSpawn (mobj_t* th)
{
  th->tics -= P_Random(pr_missile)&3;
  if (th->tics < 1)
    th->tics = 1;

  th->x += th->momx>>1;
  th->y += th->momy>>1;
  th->z += th->momz>>1;

  if (!(th->flags & MF_MISSILE) && demo_version >= 203)
    return;

  if (!P_TryMove(th, th->x, th->y, false))
    P_ExplodeMissile (th);
}

mobj_t* P_SpawnMissile(mobj_t* source,mobj_t* dest,mobjtype_t type)
{
  angle_t an;
  int     dist;
  mobj_t *th = P_SpawnMobj (source->x,source->y,source->z + 4*8*FRACUNIT,type);

  if (th->info->seesound)
    S_StartSound (th, th->info->seesound);

  P_SetTarget(&th->target, source);
  an = R_PointToAngle2 (source->x, source->y, dest->x, dest->y);

  if (dest->flags & MF_SHADOW)
    {
      int t = P_Random(pr_shadow);
      an += (t - P_Random(pr_shadow))<<20;
    }

  th->angle = an;
  an >>= ANGLETOFINESHIFT;
  th->momx = FixedMul (th->info->speed, finecosine[an]);
  th->momy = FixedMul (th->info->speed, finesine[an]);

  dist = P_AproxDistance (dest->x - source->x, dest->y - source->y);
  dist = dist / th->info->speed;

  if (dist < 1)
    dist = 1;

  th->momz = (dest->z - source->z) / dist;
  P_CheckMissileSpawn(th);
  return th;
}

void P_SpawnPlayerMissile(mobj_t* source,mobjtype_t type)
{
  mobj_t *th;
  fixed_t x, y, z, slope = 0;

  angle_t an = source->angle;

    {
      int mask = demo_version < 203 ? 0 : MF_FRIEND;
      do
	{
	  slope = P_AimLineAttack(source, an, 16*64*FRACUNIT, mask);
	  if (!linetarget)
	    slope = P_AimLineAttack(source, an += 1<<26, 16*64*FRACUNIT, mask);
	  if (!linetarget)
	    slope = P_AimLineAttack(source, an -= 2<<26, 16*64*FRACUNIT, mask);
	  if (!linetarget)
	    an = source->angle, slope = 0;
	}
      while (mask && (mask=0, !linetarget));
    }

  x = source->x;
  y = source->y;
  z = source->z + 4*8*FRACUNIT;

  th = P_SpawnMobj (x,y,z, type);

  if (th->info->seesound)
    S_StartSound (th, th->info->seesound);

  P_SetTarget(&th->target, source);
  th->angle = an;
  th->momx = FixedMul(th->info->speed,finecosine[an>>ANGLETOFINESHIFT]);
  th->momy = FixedMul(th->info->speed,finesine[an>>ANGLETOFINESHIFT]);
  th->momz = FixedMul(th->info->speed,slope);

  P_CheckMissileSpawn(th);
}