/*
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation. NO WARRANTY.
*/

#include "z_zone.h"
#include "doomstat.h"
#include <dpmi.h>

#define CACHE_ALIGN 32
#define HEADER_SIZE 32
#define CHUNK_SIZE 32
#define MIN_BLOCK_SPLIT (1024)
#define LEAVE_ASIDE (128*1024)
#define MIN_RAM (1024*1024)
#define RETRY_AMOUNT (256*1024)
#define ZONEID  0x931d4a11
#define ZONE_HISTORY 4

typedef struct memblock {

  struct memblock *next,*prev;
  size_t size;
  void **user;
  unsigned char tag,vm;
} memblock_t;

static memblock_t *rover,*zone,*zonebase;
static size_t zonebase_size;
static memblock_t *blockbytag[PU_MAX];

static void Z_Close(void)
{
  (free)(zonebase);
  zone = rover = zonebase = NULL;
}

void Z_Init(void)
{
  size_t size = _go32_dpmi_remaining_physical_memory();
  if (size < MIN_RAM) size = MIN_RAM;
  size -= LEAVE_ASIDE;
  assert(HEADER_SIZE >= sizeof(memblock_t) && MIN_RAM > LEAVE_ASIDE);
  atexit(Z_Close);
  size = (size+CHUNK_SIZE-1) & ~(CHUNK_SIZE-1);

  while (!(zonebase=(malloc)(zonebase_size=size + HEADER_SIZE + CACHE_ALIGN)))
    if (size < (MIN_RAM-LEAVE_ASIDE < RETRY_AMOUNT ? RETRY_AMOUNT :
                                                     MIN_RAM-LEAVE_ASIDE))
      I_Error("Z_Init: fail to alloc %d bytes",(unsigned long)
              zonebase_size);
    else
      size -= RETRY_AMOUNT;

  zone = (memblock_t *) ((char *) zonebase + CACHE_ALIGN -
                         ((unsigned) zonebase & (CACHE_ALIGN-1)));

  rover = zone;
  zone->next = zone->prev = zone;
  zone->size = size;
  zone->tag = PU_FREE;
  zone->vm  = 0;
}

void *(Z_Malloc)(size_t size, int tag, void **user, const char *file, int line)
{
  register memblock_t *block;
  memblock_t *start;

  if (!size)
    return user ? *user = NULL : NULL;

  size = (size+CHUNK_SIZE-1) & ~(CHUNK_SIZE-1);

  block = rover;

  if (block->prev->tag == PU_FREE)
    block = block->prev;

  start = block;

  do
    {
      if (block->tag >= PU_PURGELEVEL)
        {
          start = block->prev;
          Z_Free((char *) block + HEADER_SIZE);
          block = start = start->next;
        }

      if (block->tag == PU_FREE && block->size >= size)
        {
          size_t extra = block->size - size;
          if (extra >= MIN_BLOCK_SPLIT + HEADER_SIZE)
            {
              memblock_t *newb = (memblock_t *)((char *) block +
                                                HEADER_SIZE + size);

              (newb->next = block->next)->prev = newb;
              (newb->prev = block)->next = newb;
              block->size = size;
              newb->size = extra - HEADER_SIZE;
              newb->tag = PU_FREE;
              newb->vm = 0;

            }

          rover = block->next;

allocated:
          block->tag = tag;
          block->user = user;
          block = (memblock_t *)((char *) block + HEADER_SIZE);
          if (user) *user = block;
          return block;
        }
    }
  while ((block = block->next) != start);

  while (!(block = (malloc)(size + HEADER_SIZE)))
    {
      if (!blockbytag[PU_CACHE])
        I_Error ("Z_Malloc: Fail to alloc %d bytes"
                 "\nSource: %s:%d",(unsigned long) size, file, line);
      Z_FreeTags(PU_CACHE,PU_CACHE);
    }

  if ((block->next = blockbytag[tag]))
    block->next->prev = (memblock_t *) &block->next;
  blockbytag[tag] = block;
  block->prev = (memblock_t *) &blockbytag[tag];
  block->vm = 1;
  block->size = size;
  goto allocated;
}

void (Z_Free)(void *p, const char *file, int line)
{
  if (p)
    {
      memblock_t *other, *block = (memblock_t *)((char *) p - HEADER_SIZE);

      if (block->user)  *block->user = NULL;

      if (block->vm)
        {
          if ((*(memblock_t **) block->prev = block->next))
            block->next->prev = block->prev;

          (free)(block);
        }
      else
        {
          block->tag = PU_FREE;
          if (block != zone)
            {
              other = block->prev;
              if (other->tag == PU_FREE)
                {
                  if (rover == block) rover = other;
                  (other->next = block->next)->prev = other;
                  other->size += block->size + HEADER_SIZE;
                  block = other;

                }
            }

          other = block->next;
          if (other->tag == PU_FREE && other != zone)
            {
              if (rover == other)
                rover = block;
              (block->next = other->next)->prev = block;
              block->size += other->size + HEADER_SIZE;
            }
        }
    }
}

void (Z_FreeTags)(int lowtag, int hightag, const char *file, int line)
{
  memblock_t *block = zone;

  if (lowtag <= PU_FREE)
    lowtag = PU_FREE+1;

  do
    if (block->tag >= lowtag && block->tag <= hightag)
      {
        memblock_t *prev = block->prev, *cur = block;
        (Z_Free)((char *) block + HEADER_SIZE, file, line);
        block = (prev->next == cur) ? cur : prev;
      }
  while ((block=block->next) != zone);

  if (hightag > PU_CACHE)
    hightag = PU_CACHE;

  for (;lowtag <= hightag; lowtag++)
    for (block = blockbytag[lowtag], blockbytag[lowtag] = NULL; block;)
      {
        memblock_t *next = block->next;


        if (block->user) *block->user = NULL;
        (free)(block);
        block = next;
      }
}

void (Z_ChangeTag)(void *ptr, int tag, const char *file, int line)
{
  memblock_t *block = (memblock_t *)((char *) ptr - HEADER_SIZE);

  if (block->vm)
    {
      if ((*(memblock_t **) block->prev = block->next))
        block->next->prev = block->prev;
      if ((block->next = blockbytag[tag]))
        block->next->prev = (memblock_t *) &block->next;
      block->prev = (memblock_t *) &blockbytag[tag];
      blockbytag[tag] = block;
    }
  block->tag = tag;
}

void *(Z_Realloc)(void *ptr, size_t n, int tag, void **user,
                  const char *file, int line)
{
  void *p = (Z_Malloc)(n, tag, user, file, line);
  if (ptr)
    {
      memblock_t *block = (memblock_t *)((char *) ptr - HEADER_SIZE);
      memcpy(p, ptr, n <= block->size ? n : block->size);
      (Z_Free)(ptr, file, line);
      if (user) *user=p;
    }
  return p;
}

void *(Z_Calloc)(size_t n1, size_t n2, int tag, void **user,
                 const char *file, int line)
{
  return
    (n1*=n2) ? memset((Z_Malloc)(n1, tag, user, file, line), 0, n1) : NULL;
}

char *(Z_Strdup)(const char *s, int tag, void **user,
                 const char *file, int line)
{
  return strcpy((Z_Malloc)(strlen(s)+1, tag, user, file, line), s);
}

void (Z_CheckHeap)(const char *file, int line)
{
  memblock_t *block = zone;
  do
    if ((block->next != zone &&
         (memblock_t *)((char *) block+HEADER_SIZE+block->size) != block->next)
        || block->next->prev != block || block->prev->next != block)
      I_Error("Z_CheckHeap: Block size don't touch next block\n"
              "Source: %s:%d", file, line);
  while ((block=block->next) != zone);
}