
//
// g_input.c : handle mouse/keyboard/joystick inputs,
//	       maps inputs to game controls (forward,use,open...)
//
// by Denis Fabrice for Doom LEGACY project based on doom sources.
//

#include "doomdef.h"
#include "doomstat.h"

#include "g_input.h"

#include "hu_stuff.h"	//need HUFONT start & end

// defined in m_misc.c
extern int usemouse;
extern int usejoystick;


boolean 	mousearray[4];
boolean*	mousebuttons = &mousearray[1];		// allow [-1]

// mouse values are used once
int		mouseSensitivity;
int		mlookSensitivity;

int		mousex;
int		mousey;
int		mlooky; 	//like mousey but with a custom sensitivity
				//for mlook
// joystick values are repeated
int		joyxmove;
int		joyymove;
boolean 	joyarray[5];
boolean*	joybuttons = &joyarray[1];		// allow [-1]

// old stuff from Doom2 setup...
//  kept for conversion to the new controls
int		mousebfire;
int		mousebstrafe;
int		mousebforward;

int		joybfire;
int		joybstrafe;
int		joybuse;
int		joybspeed;

// these keys are here for compatibility,
// most of them will disappear later... they're just to keep the
// keys of the original Doom2 config
int		key_right;
int		key_left;
int		key_up;
int		key_down;
int		key_strafeleft;
int		key_straferight;
int		key_fire;
int		key_use;
int		key_strafe;
int		key_speed;

// current state of the keys : true if pushed
byte	gamekeydown[NUMKEYS];

// each key/button maps to a control or 0 (nothing)
byte	controlmap[NUMINPUTS];

// current state for each control
byte	gamecontrol[num_gamecontrols];

typedef struct {
    int time;
    int state;
    int clicks;
} dclick_t;
static	dclick_t  mousedclicks[MOUSEBUTTONS];
static	dclick_t  joydclicks[JOYBUTTONS];



//
//  Convert mousebuttons and joybuttons from the old default.cfg to
//   'virtual' keys, capable of triggering any game control.
//
//  Compatibility with older config files : convert old key_xxxx to
//  game controls, only if gc_xxxx were not found.
//
//  THIS BIG TEMPORARY HACK is currently called at startup by M_LoadDefaults
//
void G_SetupDefaultGameControls (boolean gcfound)
{
    // map inputs to null control
    // memset to gc_null done by M_LoadDefaults... oooh i'm dirty

    // convert old rusty key_xxx to brand new shiny game controls!!
    if (!gcfound)
    {
	controlmap[key_up]     = gc_forward;
	controlmap[key_down]   = gc_backward;
	controlmap[key_left]   = gc_turnleft;
	controlmap[key_right]  = gc_turnright;
	controlmap[key_strafeleft] = gc_strafeleft;
	controlmap[key_straferight] = gc_straferight;
	controlmap[key_fire]   = gc_fire;
	controlmap[key_use]    = gc_use;
	controlmap[key_strafe] = gc_strafe;
	controlmap[key_speed]  = gc_speed;
    }

    // remap mouse
    if (usemouse && !gcfound)
    {
	if (mousebfire>=0)
	    controlmap[KEY_MOUSE1+mousebfire] = gc_fire;
	if (mousebstrafe>=0)
	    controlmap[KEY_MOUSE1+mousebstrafe] = gc_strafe;
	if (mousebforward>=0)
	    controlmap[KEY_MOUSE1+mousebforward] = gc_forward;
    }

    // remap joystick
    if (usejoystick && !gcfound)
    {
	if (joybfire>=0)
	    controlmap[KEY_JOY1+joybfire] = gc_fire;
	if (joybstrafe>=0)
	    controlmap[KEY_JOY1+joybstrafe] = gc_strafe;
	if (joybuse>=0)
	    controlmap[KEY_JOY1+joybuse] = gc_use;
	if (joybspeed>=0)
	    controlmap[KEY_JOY1+joybspeed] = gc_speed;
    }

    // map default weapon keys ... hehehe yes you can shortcut any
    // key to a weapon... but the menu to configure it has still to be done
    controlmap['1'] = gc_weapon1;
    controlmap['2'] = gc_weapon2;
    controlmap['3'] = gc_weapon3;
    controlmap['4'] = gc_weapon4;
    controlmap['5'] = gc_weapon5;
    controlmap['6'] = gc_weapon6;
    controlmap['7'] = gc_weapon7;
    controlmap['8'] = gc_weapon8;

    //controlmap['r'] = gc_weapon5;	  //rocket shortcut test...

}


// protos
static boolean G_CheckDoubleClick (int state, dclick_t *dt);


//
//  Remaps the inputs to game controls.
//
//  A game control can be triggered by one or more keys/buttons.
//
//  Each key/mousebutton/joybutton triggers ONLY ONE game control.
//
//
void  G_MapEventsToControls (event_t *ev)
{
    byte   control;
    int    i,flag;

    switch (ev->type)
    {
      case ev_keydown:
	if (ev->data1 <NUMKEYS)
	{
	    gamekeydown[ev->data1] = true;

	    // get control associated to the key
	    if ( (control = controlmap[ev->data1]) )
	       gamecontrol[control] = true;
	}
	break;

      case ev_keyup:
	if (ev->data1 <NUMKEYS)
	{
	    gamekeydown[ev->data1] = false;

	    // get control associated to the key
	    if ( (control = controlmap[ev->data1]) )
	       gamecontrol[control] = false;
	}
	break;

      case ev_mouse:
	mousebuttons[0] = ev->data1 & 1;
	mousebuttons[1] = ev->data1 & 2;
	mousebuttons[2] = ev->data1 & 4;
	mousex = ev->data2*(mouseSensitivity+1)/10;   // added 3-1-98
	mousey = ev->data3*(mouseSensitivity+1)/10;

	//added:10-02-98:
	// for now I use the mlook sensitivity just for mlook,
	// instead of having a general mouse y sensitivity.
	mlooky = ev->data3*(mlookSensitivity+1)/10;

	// for each mousebutton...
	for (i=0; i<MOUSEBUTTONS; i++)
	{
	   // if there is a control for this mousebutton...
	   if ( (control = controlmap[KEY_MOUSE1+i]) )
	   {
	      // set control true if mousebutton is pushed, false otherwise
	      if (mousebuttons[i])
		  gamecontrol[control] = true;
	      else
		  gamecontrol[control] = false;
	   }
	}
	break;

      case ev_joystick:
	joybuttons[0] = ev->data1 & 1;
	joybuttons[1] = ev->data1 & 2;
	joybuttons[2] = ev->data1 & 4;
	joybuttons[3] = ev->data1 & 8;
	joyxmove = ev->data2;
	joyymove = ev->data3;

	// for each joy button
	for (i=0; i<JOYBUTTONS; i++)
	{
	   // if a control is attached to the joy button...
	   if ( (control = controlmap[KEY_JOY1+i]) )
	   {
	      // set control true if joy button is pushed, false otherwise
	      if (joybuttons[i])
		  gamecontrol[control] = true;
	      else
		  gamecontrol[control] = false;
	   }
	}
	break;

      default:
	break;

    }

    // ALWAYS check for mouse & joystick double-clicks
    // even if no mouse event
    for (i=0;i<MOUSEBUTTONS;i++)
    {
	flag = G_CheckDoubleClick (mousebuttons[i], &mousedclicks[i]);
	if ( (control = controlmap[KEY_DBLMOUSE1+i]) )
	{
	  if (flag)
	      gamecontrol[control] = true;
	  else
	      gamecontrol[control] = false;
	}
    }

    for (i=0;i<JOYBUTTONS;i++)
    {
	flag = G_CheckDoubleClick (joybuttons[i], &joydclicks[i]);
	if( (control = controlmap[KEY_DBLJOY1+i]) )
	{
	  if (flag)
	      gamecontrol[control] = true;
	  else
	      gamecontrol[control] = false;
	}
    }

}


//
//  General double-click detection routine for any kind of input.
//
static boolean G_CheckDoubleClick (int state, dclick_t *dt)
{
    if (state != dt->state && dt->time > 1 )
    {
	dt->state = state;
	if (state)
	    dt->clicks++;
	if (dt->clicks == 2)
	{
	    dt->clicks = 0;
	    return true;
	}
	else
	    dt->time = 0;
    }
    else
    {
	dt->time += ticdup;
	if (dt->time > 20)
	{
	    dt->clicks = 0;
	    dt->state = 0;
	}
    }
    return false;
}


typedef struct {
    int  keynum;
    char name[15];
} keyname_t;

static keyname_t keynames[] = {

    {KEY_SPACE	   ,"SPACE"},
    {KEY_CAPSLOCK  ,"CAPS LOCK"},
    {KEY_ENTER	   ,"ENTER"},
    {KEY_TAB	   ,"TAB"},
    {KEY_ESCAPE    ,"ESCAPE"},
    {KEY_BACKSPACE ,"BACKSPACE"},

    {KEY_NUMLOCK   ,"NUMLOCK"},
    {KEY_SCROLLLOCK,"SCROLLLOCK"},

    // bill gates keys

    {KEY_LEFTWIN   ,"LEFTWIN"},
    {KEY_RIGHTWIN  ,"RIGHTWIN"},
    {KEY_MENU	   ,"MENU"},

    // shift,ctrl,alt are not distinguished between left & right

    {KEY_SHIFT	   ,"SHIFT"},
    {KEY_CTRL	   ,"CTRL"},
    {KEY_ALT	   ,"ALT"},

    // keypad keys

    {KEY_KPADSLASH,"KEYPAD /"},

    {KEY_KEYPAD7, "KEYPAD 7"},
    {KEY_KEYPAD8, "KEYPAD 8"},
    {KEY_KEYPAD9, "KEYPAD 9"},
    {KEY_MINUSPAD,"KEYPAD -"},
    {KEY_KEYPAD4, "KEYPAD 4"},
    {KEY_KEYPAD5, "KEYPAD 5"},
    {KEY_KEYPAD6, "KEYPAD 6"},
    {KEY_PLUSPAD, "KEYPAD +"},
    {KEY_KEYPAD1, "KEYPAD 1"},
    {KEY_KEYPAD2, "KEYPAD 2"},
    {KEY_KEYPAD3, "KEYPAD 3"},
    {KEY_KEYPAD0, "KEYPAD 0"},
    {KEY_KPADDEL, "KEYPAD ."},

    // extended keys (not keypad)

    {KEY_HOME,	    "HOME"},
    {KEY_UPARROW,   "UP ARROW"},
    {KEY_PGUP,	    "PGUP"},
    {KEY_LEFTARROW ,"LEFT ARROW"},
    {KEY_RIGHTARROW,"RIGHT ARROW"},
    {KEY_END,	    "END"},
    {KEY_DOWNARROW, "DOWN ARROW"},
    {KEY_PGDN,	    "PGDN"},
    {KEY_INS,	    "INS"},
    {KEY_DEL,	    "DEL"},

    // other keys

    {KEY_F1, "F1"},
    {KEY_F2, "F2"},
    {KEY_F3, "F3"},
    {KEY_F4, "F4"},
    {KEY_F5, "F5"},
    {KEY_F6, "F6"},
    {KEY_F7, "F7"},
    {KEY_F8, "F8"},
    {KEY_F9, "F9"},
    {KEY_F10,"F10"},
    {KEY_F11,"F11"},
    {KEY_F12,"F12"},

    // virtual keys for mouse buttons and joystick buttons

    {KEY_MOUSE1,  "MOUSE1"},
    {KEY_MOUSE1+1,"MOUSE2"},
    {KEY_MOUSE1+2,"MOUSE3"},

    {KEY_JOY1,	"JOY1"},
    {KEY_JOY1+1,"JOY2"},
    {KEY_JOY1+2,"JOY3"},
    {KEY_JOY1+3,"JOY4"},

    {KEY_DBLMOUSE1,  "DBLMOUSE1"},
    {KEY_DBLMOUSE1+1,"DBLMOUSE2"},
    {KEY_DBLMOUSE1+2,"DBLMOUSE3"},

    {KEY_DBLJOY1,  "DBLJOY1"},
    {KEY_DBLJOY1+1,"DBLJOY2"},
    {KEY_DBLJOY1+2,"DBLJOY3"},
    {KEY_DBLJOY1+3,"DBLJOY4"},

};

#define NUMKEYNAMES (sizeof(keynames)/sizeof(keyname_t))

//
//  Returns the name of a key (or virtual key for mouse and joy)
//  the input value being an index into controlmap[]
//
char* G_KeynumToString (int keynum)
{
static char keynamestr[2];

    int    j;

    // return a string with the ascii char if displayable
    if (keynum>' ' && keynum<='z')
    {
	keynamestr[0] = keynum;
	keynamestr[1] = '\0';
	return keynamestr;
    }

    // find a description for special keys
    for (j=0;j<NUMKEYNAMES;j++)
	if (keynames[j].keynum==keynum)
	    return keynames[j].name;

    // create a name for Unknown key
    sprintf (keynamestr,"KEY%d",keynum);
    return keynamestr;
}


//
//  Detach any keys associated to the given game control
//
//added:08-02-98:
void  G_ClearControlKeys (int control)
{
    int   i;

    for (i=0; i<NUMINPUTS; i++)
	if (controlmap[i]==control)
	    controlmap[i] = gc_null;
}

//
//  Returns two keycodes associated to a control,
//  the second is shift left 9 bits... this is a hack to save two keys
//  in the default.cfg... this is temporary.
//
int  G_TwokeysForControl (int control)
{
    int   i;
    int   keys;

    keys = -1;
    for (i=0; i<NUMINPUTS; i++)
	if (controlmap[i]==control)
	{
	    keys = i;
	    break;
	}

    // search for a second key
    for ( ++i; i<NUMINPUTS; i++)
	if (controlmap[i]==control)
	    keys |= i<<9;	    //yuk!

    return keys;
}
