// opcode.c
// (c) Luc Cluitmans 1995

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "wadentry.h"
#include "opcode.h"
#include "instruct.h"
#include "statment.h"
#include "specials.h"
#include "goutput.h"
#include "script.h"
#include "prntstat.h"
#include "things.h"

// -------------------------------------------------------


// -------------------------------------------------------

static void VarName(int type, int idx, int argcount);
    // type: (0=local/arg) (1=map) (2=world)
    // adds to goutput string

static void PrintPrint(ScriptInfo *psi, WORD wInst);
    // Prints 'print()' or 'printbold()' expression
    // 'wInst' points to the 'endprint' or 'endprintbold' instruction

// -------------------------------------------------------

static char *keynames[] =
{
    "",
    "/* steel key */",
    "/* cave key */",
    "/* axe key */",
    "/* fire key */",
    "/* emerald key */",
    "/* dungeon key */",
    "/* silver key */",
    "/* rusted key */",
    "/* horn key */",
    "/* swamp key */",
    "/* castle key */"
};
#define KEYNAMECOUNT (sizeof(keynames)/sizeof(keynames[0]))

static char *sectorsounds[] =
{
    "",
    "/* Heavy */",
    "/* Metal */",
    "/* Creak */",
    "/* Silence */",
    "/* Lava */",
    "/* Water */",
    "/* Ice */",
    "/* Earth */",
    "/* Metal2 */"
};
#define SECTSNDCOUNT (sizeof(sectorsounds)/sizeof(sectorsounds[0]))

static char *puzzlenames[] =
{
    "/* ZZ_Skull */",
    "/* ZZ_BigGem */",
    "/* ZZ_GemRed */",
    "/* ZZ_GemGreen1 */",
    "/* ZZ_GemGreen2 */",
    "/* ZZ_GemBlue1 */",
    "/* ZZ_GemBlue2 */",
    "/* ZZ_Book1 */",
    "/* ZZ_Book2 */",
    "/* ZZ_Skull2 */",
    "/* ZZ_FWeapon */",
    "/* ZZ_CWeapon */",
    "/* ZZ_MWeapon */",
    "/* ZZ_Gear */",
    "/* ZZ_Gear2 */",
    "/* ZZ_Gear3 */",
    "/* ZZ_Gear4 */"
};
#define PUZZLECOUNT (sizeof(puzzlenames)/sizeof(puzzlenames[0]))

// -------------------------------------------------------


BOOL PrintScript(ScriptInfo *psi, int iIdx)
    // returns FALSE on success or TRUE on failure
    // use GetStringBuffer to retrieve error string
{
    WORD    v, wL;
    char    *lead;
    int argcount;
    
    if(GetInstructions(psi, iIdx))
    {
        return(TRUE);
    }

    if(GetStatements(psi))
    {
        return(TRUE);
    }
    argcount = psi->iArgCount;

    AccumulateString("script %ld",
        psi->bhv->pscd->sp[iIdx].dwId%1000);
    if(psi->bhv->pscd->sp[iIdx].dwId>1000)
    {
        AccumulateString(" OPEN");
    }
    else
    {
        AccumulateString(" (");
        if(argcount==0)
        {
            AccumulateString("void");
        }
        else
        {
            for(v=0, lead="int "; v<(WORD)argcount; v++, lead=", int ")
            {
                AccumulateString("%sarg%d", lead, v);
            }
        }
        AccumulateString(")  ");
    }
    FinishLine();
    AccumulateString("{");
    FinishLine();

    // print variable list;
    wL = psi->wLVarCnt;
    for(v=0;
        v+argcount < wL;  // Watch out: unsigned comparison!
        v++)
    {
        if(psi->pwLVarType[v+argcount] & VAR_USEDASSTRING)
        {
            AccumulateString("    str str%d;", v);
        }
        else
        {
            AccumulateString("    int var%d;", v);
        }
        FinishLine();
    }
    if(v>0)
    {
        FinishLine();
    }
    
    // skip superfluous 'terminate' at end of script
    wL = psi->wCntStat;
    if(psi->lpis[psi->lpstat[wL-1].wLead].lpdw[0] == 1) // terminate
    {
        wL--;
    }
    if(PrintStatementRange(psi, 0, wL, 1))
    {
        return(TRUE);
    }
    
    AccumulateString("}");
    FinishLine();

    return(FALSE);
}

BOOL PrintBehavior(ScriptInfo *scr)
{
    WORD w;
    
    ClearStringBuffer();
    AccumulateString("// " VERSIONSTRING " by Luc Cluitman.");
    FinishLine();
    FinishLine();
    
    AccumulateString("#include \"common.acs\"");
    FinishLine();
    FinishLine();
    
    AccumulateString("// BEHAVIOR %ld scripts, %ld strings",
                    scr->bhv->pscd->dwCnt, scr->bhv->pstd->dwCnt);
    FinishLine();
    FinishLine();
    
    AccumulateString("// ----------------------------------------");
    FinishLine();

    if(PrintGlobalVars(scr))
    {
        return(TRUE);
    }

    AccumulateString("// ----------------------------------------");
    FinishLine();
    AccumulateString("// Scripts:");
    FinishLine();
    FinishLine();
    for(w=0; w<scr->bhv->pscd->dwCnt; w++)
    {
        if(PrintScript(scr, w))
        {
            return(TRUE);
        }
        FinishLine();
    }
    FinishLine();
    
    AccumulateString("// ----------------------------------------");
    FinishLine();
    AccumulateString("// Strings:");
    FinishLine();
    for(w=0; w<scr->bhv->pstd->dwCnt; w++)
    {
        AccumulateString("// [%3d] '%s'",
            w, scr->bhv->pRaw + scr->bhv->pstd->dwOff[w]);
        FinishLine();
    }
    FinishLine();

    return(FALSE);
}

WORD LocalVarCnt(Instruction *lpis, WORD wInstCnt)
    // returns maximum local variable number
{
    WORD w;
    DWORD dwLocalCnt; // highest local var index + 1
    
    for(w=0, dwLocalCnt=0; w<wInstCnt; w++)
    {
        switch(lpis[w].wInterpretation)
        {
        case WINT_LOCALVAR:
        case WINT_ASSIGNLOCAL:
        case WINT_MODIFYLOCAL:
            if(lpis[w].lpdw[1]+1>dwLocalCnt)
            {
                dwLocalCnt = lpis[w].lpdw[1]+1;
            }
            break;
        }
    }
    return((WORD)dwLocalCnt);
}

WORD PrintArgs(ScriptInfo *psi, WORD wExpFrom, WORD wExpMax,
                      WORD wNumArgs, char *lead, BOOL bNeedParen)
    // prints wNumArgs arguments of the expression at index wExpMax.
    // Searching for arguments starts at wExpFrom and can continue
    // to at most wExpMax. 'lead' is printed before the first argument,
    // the interargument separator is fixed to ", ".
    // Returns the number after the last instruction printed (pass to
    // next call of PrintArgs as wExpFrom parameter).
{
    WORD v, wArgs;
    Instruction *ptris;
    Opcode *popc2;
    Opcode *popc;
    Instruction *lpis = psi->lpis;
    
    if(wNumArgs>1) bNeedParen = FALSE;
    ptris = lpis+wExpMax;
    popc = opcodes + lpis[wExpMax].lpdw[0];
    for(v=wExpFrom, wArgs=0; v<wExpMax && wArgs<wNumArgs; v++)
    {
        if(lpis[v].wXLevel == ptris->wXLevel+1)
        {
            popc2 = opcodes + lpis[v].lpdw[0];
            if(popc2->wPushCnt>0) // not a postfix inc/dec
            {
                AccumulateString("%s", lead);
                if(popc2->wInterpretation == WINT_CONSTANT)
                {
                    // Bypass PrintExpression to use type info
                    if(popc->cArgs[0])
                    {
                        PrintConstant(psi, lpis[v].lpdw[1], popc->cArgs[wArgs]);
                    }
                    else if(lpis[v].cType)
                    {
                        PrintConstant(psi, lpis[v].lpdw[1], lpis[v].cType);
                    }
                    else
                    {
                        PrintConstant(psi, lpis[v].lpdw[1], '0');
                    }
                }
                else
                {
                    PrintExpression(psi, v,
                        (bNeedParen &&
                         (popc2->wInterpretation==WINT_INFIXB ||
                          popc2->wInterpretation==WINT_COMPARE ||
                          (
                            popc2->wInterpretation==WINT_INFIXA &&
                            popc2!=popc
                          ))));
                }
                lead = ", ";
                wArgs++;
            }
            else
            {
                PrintExpression(psi, v, 0);
            }
        }
    }
    // print loose ends belonging to last argument
    if(v<wExpMax &&
       v!=wExpFrom && // something was printed at all
       lpis[v].wInterpretation>=WINT_MODIFYLOCAL &&
       lpis[v].wInterpretation<=WINT_MODIFYWORLD &&
       lpis[v-1].wInterpretation-WINT_LOCALVAR ==
            lpis[v].wInterpretation-WINT_MODIFYLOCAL &&
       lpis[v-1].lpdw[1] == lpis[v].lpdw[1])
    {
        PrintExpression(psi, v, 0);
        v++;
    }
    return(v);
} 

void PrintExpression(ScriptInfo *psi, WORD wExpIdx, BOOL bNeedParen)
{
    WORD v;
    char *lead, *spec, *oldarg;
    Instruction *ptris;
    Opcode *popc;
    WORD wCntDw;
    DWORD *pdw, dw;
    
    ptris = psi->lpis+wExpIdx;
    pdw = ptris->lpdw;
    popc = opcodes + *pdw++;
    wCntDw = popc->wCnt;
    
    switch(ptris->wInterpretation)
    {
    case WINT_SPECIAL:
        wCntDw--;
        spec = SpecialName((WORD)(*pdw));
        
        // actually CHANGE arguments...
        oldarg = popc->cArgs;
        popc->cArgs = SpecialArgs((WORD)(*pdw));
        pdw++;
        
        AccumulateString("%s(", spec);
        if(popc->wArgCnt==0)
        {
            if(wCntDw>0) AccumulateString("const:");
            for(v=0, lead=""; v<wCntDw; v++, lead=", ")
            {
                AccumulateString("%s", lead);
                dw = *pdw++;
                if(popc->cArgs && popc->cArgs[0])
                {
                    PrintConstant(psi, dw, popc->cArgs[v]);
                }
            }
        }
        else
        {
            PrintArgs(psi, (WORD)ptris->iFirstInst, wExpIdx, popc->wArgCnt, "", 0);
        }
        AccumulateString(")");
        
        // restore argument types
        break;
    case WINT_JUMP:
        AccumulateString("%s L%04d", popc->szName, psi->lpis[ptris->wJump].iLabel);
        break;    
    case WINT_CONDJUMPTRUE:
    case WINT_CONDJUMPFALSE:
        AccumulateString("%s (", popc->szName);
        PrintArgs(psi, (WORD)ptris->iFirstInst, wExpIdx, popc->wArgCnt, "", 0);
        AccumulateString(") L%04d", psi->lpis[ptris->wJump].iLabel);
        break;    
    case WINT_CASE:
        AccumulateString("case %d jump L%04d", (WORD)pdw[0],
                        psi->lpis[ptris->wJump].iLabel);
        break;    
    case WINT_BUILTIN:
        AccumulateString("%s", popc->szName);
        AccumulateString("(");
        if(popc->wArgCnt==0)
        {
            if(wCntDw>0) AccumulateString("const:");
            for(v=0, lead=""; v<wCntDw; v++, lead=", ")
            {
                AccumulateString("%s", lead);
                if(popc->cArgs[0])
                {
                    PrintConstant(psi, *pdw++, popc->cArgs[v]);
                }
                else
                {
                    AccumulateString("%ld", *pdw++);
                }
            }
        }
        else
        {
            PrintArgs(psi, (WORD)ptris->iFirstInst, wExpIdx, popc->wArgCnt, "", 0);
        }
        AccumulateString(")");
        break;
    case WINT_BEGINPRINT:
        AccumulateString("// *%s* ", popc->szName);
        break;
    case WINT_PRINTSTR:
    case WINT_PRINTNUM:
    case WINT_PRINTCHR:
        AccumulateString("// *%s* ", popc->szName);
        AccumulateString("(");
        PrintArgs(psi, (WORD)ptris->iFirstInst, wExpIdx, popc->wArgCnt, "", 0);
        AccumulateString(")");
        break;
    case WINT_ENDPRINT:
    case WINT_ENDPRINTBOLD:
        PrintPrint(psi, wExpIdx);
        break;
    case WINT_KEYWORD:
        AccumulateString("%s", popc->szName);
        break;
    case WINT_INFIXA: // always two stack arguments and one stack result.
    case WINT_INFIXB:
    case WINT_COMPARE:
        if(bNeedParen) AccumulateString("(");
        v = PrintArgs(psi, (WORD)ptris->iFirstInst, wExpIdx, 1, "", TRUE);
        AccumulateString("%s", popc->szName);
        PrintArgs(psi, v, wExpIdx, 1, "", TRUE);
        if(bNeedParen) AccumulateString(")");
        break;
    case WINT_CONSTANT:
        AccumulateString("%ld", *pdw++); // usually bypassed!
        break;
    case WINT_DROP:
        AccumulateString("// end of switch (drop)");
        break;
    case WINT_LOCALVAR:
    case WINT_MAPVAR:
    case WINT_WORLDVAR:
        VarName(ptris->wInterpretation - WINT_LOCALVAR, (int)*pdw++, psi->iArgCount);
        break;
    case WINT_ASSIGNLOCAL:
    case WINT_ASSIGNMAP:
    case WINT_ASSIGNWORLD:
        // = += *= etc.
        VarName(ptris->wInterpretation - WINT_ASSIGNLOCAL,
                                        (int)*pdw++, psi->iArgCount);
        AccumulateString(" %s ", popc->szName);
        PrintArgs(psi, (WORD)ptris->iFirstInst, wExpIdx, 1, "", 0);
        break;
    case WINT_MODIFYLOCAL:
    case WINT_MODIFYMAP:
    case WINT_MODIFYWORLD:
        // ++ / --
        if(ptris->wXLevel==0)
        {
            // only print variable name if used as level 0 command!
            VarName(ptris->wInterpretation - WINT_MODIFYLOCAL,
                                        (int)*pdw++, psi->iArgCount);
        }
        else
        {
            // don't print variable name at all...
        }
        AccumulateString("%s", popc->szName);
        break;
    case WINT_SWITCH:
        AccumulateString("switch (");
        // find argument:
        PrintArgs(psi, (WORD)ptris->iFirstInst, wExpIdx, 1, "", 0);
        AccumulateString(") L%04d", psi->lpis[ptris->wJump].iLabel);
        break;    
    default:
        AccumulateString("@(%s", popc->szName);
        for(v=0, lead=""; v<wCntDw; v++, lead=", ")
        {
            AccumulateString("%s%ld", lead, *pdw++);
        }
        AccumulateString(")@");
        break;
    }
}

static void VarName(int type, int idx, int argcount)
    // type: (0=local/arg) (1=map) (2=world)
    // adds to goutput string
{
    // Note: current scriptinfo available via gScr...
    
    switch(type)
    {
    case 0:
        if((WORD)idx<gScr->wLVarCnt && (gScr->pwLVarType[idx]&VAR_USEDASSTRING))
        {
            if(idx>=argcount)
            {
                AccumulateString("str%d", idx-argcount);
            }
            else
            {
                AccumulateString("arg%d", idx); // invalid???
            }
        }
        else
        {
            if(idx>=argcount)
            {
                AccumulateString("var%d", idx-argcount);
            }
            else
            {
                AccumulateString("arg%d", idx);
            }
        }
        break;
    case 1:
        if((WORD)idx<gScr->wMVarCnt && (gScr->pwMVarType[idx]&VAR_USEDASSTRING))
        {
            AccumulateString("mapstr%d", idx);
        }
        else
        {
            AccumulateString("mapvar%d", idx);
        }
        break;
    case 2:
        AccumulateString("world%d", idx);
        break;
    }
}

void PrintConstant(ScriptInfo *psi, DWORD dwValue, char cType)
{
    BOOL bPrinted=FALSE;
    char *s;
    WORD w;
    
    switch(cType)
    {
    case 's':
        if(dwValue>=0 && dwValue<psi->bhv->pstd->dwCnt)
        {
            bPrinted = TRUE;
            AccumulateString("\"%s\"",
                psi->bhv->pRaw + psi->bhv->pstd->dwOff[(WORD)dwValue]);
        }
        else
        {
            AccumulateString("/*INVALID?*/");
        }
        break;
    case 'm':
        // nothing special
        break;
    case 't':
        s = ThingName((WORD)dwValue);
        if(s)
        {
            bPrinted = TRUE;
            AccumulateString("%s", s);
        }
        break;
    case 'T':
        // nothing special
        break;
    case 'k':
        w = (WORD)dwValue;
        if(w>=0 && w<KEYNAMECOUNT)
        {
            bPrinted = TRUE;
            AccumulateString("%d %s", w, keynames[w]);
        }
        break;
    case 'p':
        w = (WORD)dwValue;
        if(w>=0 && w<PUZZLECOUNT)
        {
            bPrinted = TRUE;
            AccumulateString("%d %s", w, puzzlenames[w]);
        }
        break;
    case 'n':
        w = (WORD)dwValue;
        if(w>=0 && w<SECTSNDCOUNT)
        {
            bPrinted = TRUE;
            AccumulateString("%d %s", w, sectorsounds[w]);
        }
        break;
    case 'S':
        s = SpecialName((WORD)dwValue);
        if(s)
        {
            bPrinted = TRUE;
            AccumulateString("%ld /*%s*/", dwValue, s);
        }
        break;
    case 'F':
        switch((WORD)dwValue)
        {
        case 0:
            bPrinted = TRUE;
            AccumulateString("LINE_FRONT");
            break;
        case 1:
            bPrinted = TRUE;
            AccumulateString("LINE_BACK");
            break;
        }
        break;
    case 'f':
        switch((WORD)dwValue)
        {
        case 0:
            bPrinted = TRUE;
            AccumulateString("SIDE_FRONT");
            break;
        case 1:
            bPrinted = TRUE;
            AccumulateString("SIDE_BACK");
            break;
        }
        break;
    case 'g':
        switch((WORD)dwValue)
        {
        case 0:
            bPrinted = TRUE;
            AccumulateString("GAME_SINGLE_PLAYER");
            break;
        case 1:
            bPrinted = TRUE;
            AccumulateString("GAME_NET_COOPERATIVE");
            break;
        case 2:
            bPrinted = TRUE;
            AccumulateString("GAME_NET_DEATHMATCH");
            break;
        }
        break;
    case 'G':
        switch((WORD)dwValue)
        {
        case 0:
            bPrinted = TRUE;
            AccumulateString("SKILL_VERY_EASY");
            break;
        case 1:
            bPrinted = TRUE;
            AccumulateString("SKILL_EASY");
            break;
        case 2:
            bPrinted = TRUE;
            AccumulateString("SKILL_NORMAL");
            break;
        case 3:
            bPrinted = TRUE;
            AccumulateString("SKILL_HARD");
            break;
        case 4:
            bPrinted = TRUE;
            AccumulateString("SKILL_VERY_HARD");
            break;
        }
        break;
    case 'b':
        switch((WORD)dwValue)
        {
        case 0:
            bPrinted = TRUE;
            AccumulateString("OFF");
            break;
        case 1:
            bPrinted = TRUE;
            AccumulateString("ON");
            break;
        }
        break;
    case 'B':
        bPrinted = TRUE;
        if(dwValue)
        {
            AccumulateString("TRUE");
        }
        else
        {
            AccumulateString("FALSE");
        }
        break;
    case 'x':
        switch((WORD)dwValue)
        {
        case 0:
            bPrinted = TRUE;
            AccumulateString("TEXTURE_TOP");
            break;
        case 1:
            bPrinted = TRUE;
            AccumulateString("TEXTURE_MIDDLE");
            break;
        case 2:
            bPrinted = TRUE;
            AccumulateString("TEXTURE_BOTTOM");
            break;
        }
        break;
    }
    
    if(!bPrinted)
    {
        AccumulateString("%ld", dwValue);
    }
}

BOOL PrintGlobalVars(ScriptInfo *psi)
{
    // Prints a list of all world and map variables used
    // in all scripts of the given behavior.
    // (leaves *psi in LS_BEHAVIOR mode)

    // Also sets the number of mapvariables in psi,
    // allocates space for map variable type info and tests
    // map variables for use as string variable.
    
    WORD w, v, vi, wMapMax, wWorldMax;
    char cWorldVar[256]; // boolean array
    
    // clear arrays
    for(w=0; w<256; w++)
    {
        cWorldVar[w] = 0;
    }
    
    // process all scripts
    for(w=0, wMapMax=0, wWorldMax=0; w<psi->bhv->pscd->dwCnt; w++)
    {
        // load Instructions
        if(GetInstructions(psi, w))
        {
            return(TRUE);
        }
        
        // scan variable usage
        for(v=0; v<psi->wCntInst; v++)
        {
            switch(psi->lpis[v].wInterpretation)
            {
            case WINT_MAPVAR:
            case WINT_ASSIGNMAP:
            case WINT_MODIFYMAP:
                vi = (WORD)psi->lpis[v].lpdw[1];
                if(vi+1 > wMapMax)
                {
                    wMapMax = vi+1;
                }
                break;
            case WINT_WORLDVAR:
            case WINT_ASSIGNWORLD:
            case WINT_MODIFYWORLD:
                vi = (WORD)psi->lpis[v].lpdw[1];
                if(vi >= 256)
                {
                    AccumulateString(
                        "Sorry, didn't expect more than 256 world variables");
                    return(TRUE);
                }
                cWorldVar[vi] = 1;
                if(vi+1 > wWorldMax)
                {
                    wWorldMax = vi+1;
                }
                break;
            }
        }
    }
    
    // update map variable state
    psi->wMVarCnt = wMapMax;
    if(psi->pwMVarType)
    {
        free(psi->pwMVarType);
    }
    if(psi->pwLVarType)
    {
        free(psi->pwLVarType);
        psi->pwLVarType = NULL;
        psi->wLVarCnt = 0;
    }
    if(wMapMax>0)
    {
        psi->pwMVarType = calloc(wMapMax, sizeof(WORD));
        if(!psi->pwMVarType)
        {
            AccumulateString("Out of memory");
            return(TRUE);
        }
        
        // scan for map variable types
        for(w=0; w<psi->bhv->pscd->dwCnt; w++)
        {
            // load Instructions
            if(GetInstructions(psi, w))
            {
                return(TRUE);
            }
            
            if(ScanStringVars(psi, OFFSET_MAP))
            {
                return(TRUE);
            }
            
            UnloadInstructions(psi);
        }
    }

    // print list of variables
    if(wMapMax>0)
    {
        AccumulateString("// List of MAP variables:");
        FinishLine();
        for(v=0; v<wMapMax; v++)
        {
            if(psi->pwMVarType[v] & VAR_USEDASSTRING)
            {
                AccumulateString("str mapstr%d;", v);
            }
            else
            {
                AccumulateString("int mapvar%d;", v);
            }
            FinishLine();
        }
        FinishLine();
    }

    if(wWorldMax>0)
    {
        AccumulateString("// List of WORLD variables:");
        FinishLine();
        for(v=0; v<256; v++)
        {
            if(cWorldVar[v])
            {
                AccumulateString("world int %d:world%d;", v, v);
                FinishLine();
            }
        }
        FinishLine();
    }

    return(FALSE);
}

static void PrintPrint(ScriptInfo *psi, WORD wInst)
    // Prints 'print()' or 'printbold()' expression
    // 'wInst' points to the 'endprint' or 'endprintbold' instruction
{
    WORD w, v;
    char *lead;
    
    AccumulateString("%s", (psi->lpis[wInst].wInterpretation==WINT_ENDPRINT)
                           ? "print"
                           : "printbold");
    AccumulateString("(");

    // search for matching 'beginprint'
    for(v=wInst;
        v>0 && psi->lpis[v].wInterpretation!=WINT_BEGINPRINT;
        v--);
    if(psi->lpis[v].wInterpretation!=WINT_BEGINPRINT)
    {
        // HELP!
        return;
    }
    
    // print following toplevel instructions (should be PRINTNUM, etc.)
    for(w=v+1, lead=""; w<wInst; w++)
    {
        if(psi->lpis[w].wXLevel!=0) continue;
        switch(psi->lpis[w].wInterpretation)
        {
        case WINT_PRINTSTR:
            AccumulateString("%ss:", lead);
            PrintArgs(psi, (WORD)psi->lpis[w].iFirstInst,
                      w, 1, "", FALSE);
            lead = ", ";
            break;
        case WINT_PRINTNUM:
            AccumulateString("%sd:", lead);
            PrintArgs(psi, (WORD)psi->lpis[w].iFirstInst,
                      w, 1, "", FALSE);
            lead = ", ";
            break;
        case WINT_PRINTCHR:
            AccumulateString("%sc:", lead);
            PrintArgs(psi, (WORD)psi->lpis[w].iFirstInst,
                      w, 1, "", FALSE);
            lead = ", ";
            break;
        }
    }
        
    AccumulateString(")");
}

BOOL ScanStringVars(ScriptInfo *psi, WORD wVarType)
    // Scans for variables that are used as string variables.
    // Note: World variables can never be used as string
    // variables.
    // 'wVarType' should be OFFSET_MAP or OFFSET_LOCAL.
{
    WORD wArgs[7];
    WORD wI, w, wV;
    Instruction *lpis;
    char *psz;

    if(psi->iLoadLevel<LS_INSTRUCTIONS ||
       (psi->iLoadLevel<LS_STATEMENTS && wVarType==OFFSET_LOCAL))
    {
        AccumulateString("ScanStringVars: Phase error");
        return(TRUE);
    }
    
    lpis = psi->lpis;
    for(wI=0; wI<psi->wCntInst; wI++)
    {
        // find end of statement
        while(wI<psi->wCntInst && lpis[wI].wXLevel>0) wI++;
        if(wI>=psi->wCntInst)
        {
            AccumulateString("FATAL: Unexpected end of instruction list");
            return(TRUE);
        }
        
        // Checking for string arguments only makes sense if the statement
        // leader (instruction wI) takes stack arguments that can be a string.
        // Note that 'specials' never take string arguments.
        psz = opcodes[lpis[wI].lpdw[0]].cArgs;
        if(opcodes[lpis[wI].lpdw[0]].wArgCnt>0 && // takes stack arguments
           psz &&               // argument types are defined at all
           strchr(psz, 's'))    // at least one argument is a string
        {
            // grab subexpression leaders
            ArgumentArray(psi, wI, wArgs);
            for(w=0; w<7 && psz[w]!='\0' && wArgs[w]!=0xFFFF; w++)
            {
                if(psz[w]!='s') continue;
                wV = lpis[wArgs[w]].wInterpretation - wVarType;
                    // result should be one of the WINT_xxLOCALxx values
                if(wV==WINT_LOCALVAR ||
                   wV==WINT_ASSIGNLOCAL ||
                   wV==WINT_MODIFYLOCAL)
                {
                    // Gotcha!
                    wV = (WORD)lpis[wArgs[w]].lpdw[1]; // var number
                    
//fprintf(stderr, "Variable %d / class %d used for string\n",
//        wV, wVarType);
                    
                    switch(wVarType)
                    {
                    case OFFSET_LOCAL:
                        if(!psi->pwLVarType || psi->wLVarCnt<=wV)
                        {
                            AccumulateString("FATAL: inconsistent local variable\n");
                            return(TRUE);
                        }
                        psi->pwLVarType[wV] |= VAR_USEDASSTRING;
                        break;
                    case OFFSET_MAP:
                        if(!psi->pwMVarType || psi->wMVarCnt<=wV)
                        {
                            AccumulateString("FATAL: inconsistent map variable\n");
                            return(TRUE);
                        }
                        psi->pwMVarType[wV] |= VAR_USEDASSTRING;
                        break;
                    }
                }
            }
        }
    }
    

    return(FALSE);
}

// EOF
