// prntstat.c

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bastypes.h"
#include "wadentry.h"
#include "opcode.h"
#include "instruct.h"
#include "statment.h"
#include "specials.h"
#include "goutput.h"
#include "script.h"
#include "prntstat.h"

// -----------------------------------------------------------

static void PrintStatement(ScriptInfo *psi, WORD wStat);
                    // prints one 'simple' (nonjump) statement

static WORD PrintDo(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent);
                    // Prints 'do {} while();' or 'do{}until();'
                    // Returns index of last statement printed
                    // The matching 'while' or 'until' should appear
                    // *before* index 'wMax' (implies: not *at* 'wMax')
                    // Returns 0xFFFF on failure.

static WORD PrintIf(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent,
                    BOOL bNoIndent);
                    // Prints 'if(){}' (not 'if(){}else{}') constructs.
                    // Returns index of last statement printed
                    // The last statement printed should occur *before*
                    // 'wMax'.
                    // If 'bNoIndent' is true, the first line is not indented.
                    // (used in printing if {} else if {} constructs
                    // Returns 0xFFFF on failure.

static WORD PrintIfElse(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent,
                    BOOL bNoIndent);
                    // Prints 'if(){}else{}' (not 'if(){}') constructs.
                    // Returns index of last statement printed
                    // The last statement printed should occur *before*
                    // 'wMax'.
                    // If 'bNoIndent' is true, the first line is not indented.
                    // (used in printing if {} else if {} constructs
                    // Returns 0xFFFF on failure.

static WORD PrintLoop(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent);
                    // Prints 'while(){}' or 'until(){}' constructs.
                    // Returns index of last statement handled.
                    // The last statement should occur *before*
                    // 'wMax'.
                    // Returns 0xFFFF on failure.

static WORD PrintSwitch(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent);
                    // Prints 'switch(){}' constructs.
                    // Returns index of last statement handled.
                    // The last statement should occur *before*
                    // 'wMax'.
                    // Returns 0xFFFF on failure.

// -----------------------------------------------------------

BOOL PrintStatementRange(ScriptInfo *psi, WORD wFrom, WORD wTo, WORD wIndent)
    // print statements wFrom up to but not including wTo
    // Returns TRUE on failure
{
    WORD wS, w, v;
    
    for(wS=wFrom; wS<wTo; wS++)
    {
#if 0
        // debug: print labels
        if(psi->lpstat[wS].iLabel>0)
        {
            AccumulateString("L%04d:", psi->lpstat[wS].iLabel);
            FinishLine();
        }
#endif

        if(psi->lpstat[wS].wDoCnt>0)
        {
            v = PrintDo(psi, wS, wTo, wIndent);
            if(v==0xFFFF) return(TRUE);
            wS = v;
        }
        else
        {
            switch(psi->lpstat[wS].wJumpType)
            {
            case JT_NONE:
                // skip print leaders
                switch(psi->lpis[psi->lpstat[wS].wLead].wInterpretation)
                {
                case WINT_BEGINPRINT:
                case WINT_PRINTSTR:
                case WINT_PRINTNUM:
                case WINT_PRINTCHR:
                    break;
                default:
                    for(w=0; w<wIndent; w++) AccumulateString("    ");
                    PrintStatement(psi, wS);
                    AccumulateString(";");
                    FinishLine();
                }
                break;

            case JT_IF:
                v = PrintIf(psi, wS, wTo, wIndent, FALSE);
                if(v==0xFFFF)
                {
                    return(TRUE);
                }
                wS = v;
                break;
            case JT_IFELSE:
                v = PrintIfElse(psi, wS, wTo, wIndent, FALSE);
                if(v==0xFFFF)
                {
                    return(TRUE);
                }
                wS = v;
                break;
            case JT_WHILE:
                v = PrintLoop(psi, wS, wTo, wIndent);
                if(v==0xFFFF)
                {
                    return(TRUE);
                }
                wS = v;
                break;
            case JT_UNTIL:
                v = PrintLoop(psi, wS, wTo, wIndent);
                if(v==0xFFFF)
                {
                    return(TRUE);
                }
                wS = v;
                break;
            case JT_SWITCHBREAK:
                for(w=0; w<wIndent; w++) AccumulateString("    ");
                // PrintStatement(psi, wS);
                AccumulateString("break;");
                FinishLine();
                break;
            case JT_TERMINATOR:
                // Print Nothing
#if 0
                for(w=0; w<wIndent; w++) AccumulateString("    ");
                // PrintStatement(psi, wS);
                AccumulateString("// end-of-switch");
                FinishLine();
#endif
                break;
            case JT_BREAK:
                for(w=0; w<wIndent; w++) AccumulateString("    ");
                // PrintStatement(psi, wS);
                AccumulateString("break;");
                FinishLine();
                break;
            case JT_CONTINUE:
                for(w=0; w<wIndent; w++) AccumulateString("    ");
                // PrintStatement(psi, wS);
                AccumulateString("continue;");
                FinishLine();
                break;

            case JT_SWITCH:
#if 0
                for(w=0; w<wIndent; w++) AccumulateString("    ");
                PrintStatement(psi, wS);
                AccumulateString("; // SWITCH");
                FinishLine();
#endif
                v = PrintSwitch(psi, wS, wTo, wIndent);
                if(v==0xFFFF)
                {
                    return(TRUE);
                }
                wS = v;
                break;
            
            // jump types that are handled elsewhere:
            case JT_CASE:
            case JT_DEFAULT:
            case JT_ELSE:
            case JT_DOWHILE:
            case JT_DOUNTIL:
            case JT_ENDWHILE:
            case JT_ENDUNTIL:
                ClearStringBuffer();
                AccumulateString("FATAL: Jump Type %d should not have occurred here",
                                  psi->lpstat[wS].wJumpType);
                return(TRUE);
					 /*break;*/

				default:
					 ClearStringBuffer();
					 AccumulateString("FATAL: invalid Jump Type");
					 return(TRUE);
				}
		  }
	 }

    return(FALSE);
}

// ----------------------------------------------------------------

static void PrintStatement(ScriptInfo *psi, WORD wStat)
                    // prints one 'simple' (nonjump) statement
{
    PrintExpression(psi, psi->lpstat[wStat].wLead, FALSE);
}

static WORD PrintDo(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent)
                    // Prints 'do {} while();' or 'do{}until();'
                    // Returns index of last statement printed
                    // The matching 'while' or 'until' should appear
                    // *before* index 'wMax' (implies: not *at* 'wMax')
                    // Returns 0xFFFF on failure.
{
    WORD wTgt; // index of matching while/until
    WORD w;
    
    if(psi->lpstat[wFrom].wDoCnt==0)
    {
        ClearStringBuffer();
        AccumulateString("'PrintDo' error: wDoCnt==0");
        return(0xFFFF);
    }
    
    // find match
    for(wTgt=wFrom, w=0; wTgt<wMax; wTgt++)
    {
        if(((psi->lpstat[wTgt].wJumpType == JT_DOWHILE) ||
            (psi->lpstat[wTgt].wJumpType == JT_DOUNTIL)) &&
           (psi->lpstat[wTgt].wJump == wFrom))
        {
            w++;
            if(w>=psi->lpstat[wFrom].wDoCnt) break;
        }
    }
    if(wTgt>=wMax)
    {
        ClearStringBuffer();
        AccumulateString("'PrintDo' error: missing DOWHILE or DOUNTIL");
        return(0xFFFF);
    }
    
    // temporarily decrement wDoCnt to enable recursion
    psi->lpstat[wFrom].wDoCnt--;

    // print 'do' header
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("do");
    FinishLine();
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("{");
    FinishLine();
    
    // print statement block
    if(PrintStatementRange(psi, wFrom, wTgt, wIndent+1))
    {
        // restore wDoCnt value
        psi->lpstat[wFrom].wDoCnt++;
        return(0xFFFF);
    }
    
    // print trailer
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    if(psi->lpstat[wTgt].wJumpType == JT_DOWHILE)
    {
        AccumulateString("} while(");
    }
    else
    {
        AccumulateString("} until(");
    }
    
    w = psi->lpstat[wTgt].wLead;
    PrintArgs(psi, (WORD)(psi->lpis[w].iFirstInst), w, 1, "", 0);
    
    AccumulateString(");");
    FinishLine();
    

    // restore wDoCnt value
    psi->lpstat[wFrom].wDoCnt++;
    return(wTgt);
}

static WORD PrintIf(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent,
                    BOOL bNoIndent)
                    // Prints 'if(){}' (not 'if(){}else{}') constructs.
                    // Returns index of last statement printed
                    // The last statement printed should occur *before*
                    // 'wMax'.
                    // If 'bNoIndent' is true, the first line is not indented.
                    // (used in printing if {} else if {} constructs
                    // Returns 0xFFFF on failure.
{
    WORD wTgt; // statement after end-of-if
    WORD w;
    
    wTgt = psi->lpstat[wFrom].wJump;
    if(wTgt > wMax)
    {
        ClearStringBuffer();
        AccumulateString("FATAL: Sync error when printing 'if(){}' (Tgt=%d, Max=%d)",
                         wTgt, wMax);
        return(0xFFFF);
    }
    
    // print header
    if(!bNoIndent)
    {
        for(w=0; w<wIndent; w++) AccumulateString("    ");
    }
    AccumulateString("if(");
    w = psi->lpstat[wFrom].wLead;
    PrintArgs(psi, (WORD)(psi->lpis[w].iFirstInst), w, 1, "", FALSE);
    AccumulateString(")");
    FinishLine();
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("{");
    FinishLine();
    
    // print statement block
    if(PrintStatementRange(psi, wFrom+1, wTgt, wIndent+1))
    {
        return(0xFFFF);
    }
    
    // print trailer
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("}");
    FinishLine();
    
    return(wTgt-1);
}

static WORD PrintLoop(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent)
                    // Prints 'while(){}' or 'until(){}' constructs.
                    // Returns index of last statement handled.
                    // The last statement should occur *before*
                    // 'wMax'.
                    // Returns 0xFFFF on failure.
{
    WORD wTgt; // statement after end-of-if
    WORD w;
    
    wTgt = psi->lpstat[wFrom].wJump;
    if(wTgt>wMax || wTgt-1<=wFrom)
    {
        ClearStringBuffer();
        AccumulateString(
            "FATAL: Sync error when printing 'while/until(){}' (Tgt=%d, Max=%d)",
            wTgt, wMax);
        return(0xFFFF);
    }
    
    // print header
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("%s(", (psi->lpstat[wFrom].wJumpType==JT_WHILE)?"while":"until");
    w = psi->lpstat[wFrom].wLead;
    PrintArgs(psi, (WORD)(psi->lpis[w].iFirstInst), w, 1, "", FALSE);
    AccumulateString(")");
    FinishLine();
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("{");
    FinishLine();
    
    // print statement block
    if(PrintStatementRange(psi, wFrom+1, wTgt-1, wIndent+1))
    {
        return(0xFFFF);
    }
    
    // print trailer
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("}");
    FinishLine();
    
    return(wTgt-1);
}

static WORD PrintIfElse(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent,
                    BOOL bNoIndent)
                    // Prints 'if(){}else{}' (not 'if(){}') constructs.
                    // Returns index of last statement printed
                    // The last statement printed should occur *before*
                    // 'wMax'.
                    // If 'bNoIndent' is true, the first line is not indented.
                    // (used in printing if {} else if {} constructs
                    // Returns 0xFFFF on failure.
{
    WORD wTgt; // statement after end-of-if
    WORD wElse; // start of 'else' block
    WORD w;
    BOOL bSequence;
    
    wElse = psi->lpstat[wFrom].wJump;
    if(wElse>wMax || wElse-1<=wFrom ||
       psi->lpstat[wElse-1].wJumpType!=JT_ELSE)
    {
        ClearStringBuffer();
        AccumulateString(
            "FATAL: Sync error when printing 'if(){}else{}' (Else=%d, Max=%d)",
            wElse, wMax);
        return(0xFFFF);
    }
    wTgt = psi->lpstat[wElse-1].wJump;
    if(wTgt>wMax || wTgt<wElse)
    {
        ClearStringBuffer();
        AccumulateString(
            "FATAL: Sync error 2 when printing 'if(){}else{}' (Tgt=%d, Else=%d, Max=%d)",
            wTgt, wElse, wMax);
        return(0xFFFF);
    }
    
    
    // print header
    if(!bNoIndent)
    {
        for(w=0; w<wIndent; w++) AccumulateString("    ");
    }
    AccumulateString("if(");
    w = psi->lpstat[wFrom].wLead;
    PrintArgs(psi, (WORD)(psi->lpis[w].iFirstInst), w, 1, "", FALSE);
    AccumulateString(")");
    FinishLine();
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("{");
    FinishLine();
    
    // print 'if' statement block
    if(PrintStatementRange(psi, wFrom+1, wElse-1, wIndent+1))
    {
        return(0xFFFF);
    }
    
    // print 'if' trailer
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("}");
    FinishLine();

    // Check if the else block fully consists of an if{} construct or
    // an if{}else{} construct: print sequentially in that case.
    bSequence = FALSE;
    if(psi->lpstat[wElse].wJumpType==JT_IF)
    {
        // Need to check if the end of the nested block is the same as the
        // end of the current block.
        if(psi->lpstat[wElse].wJump == wTgt)
        {
            bSequence = TRUE;
        }
    }
    else if(psi->lpstat[wElse].wJumpType==JT_IFELSE)
    {
        // Need to check if the end of the nested block is the same as the
        // end of the current block.
        w = psi->lpstat[wElse].wJump;
        if(psi->lpstat[w-1].wJump == wTgt)
        {
            bSequence = TRUE;
        }
    }
    
    if(bSequence)
    {
        for(w=0; w<wIndent; w++) AccumulateString("    ");
        AccumulateString("else ");
        if(psi->lpstat[wElse].wJumpType==JT_IF)
        {
            w = PrintIf(psi, wElse, wTgt, wIndent, TRUE); 
            if(w==0xFFFF)
            {
                return(0xFFFF);
            }
        }
        else
        {
            w = PrintIfElse(psi, wElse, wTgt, wIndent, TRUE);
            if(w==0xFFFF)
            {
                return(0xFFFF);
            }
        }
    }
    else
    {
        for(w=0; w<wIndent; w++) AccumulateString("    ");
        AccumulateString("else");
        FinishLine();
        for(w=0; w<wIndent; w++) AccumulateString("    ");
        AccumulateString("{");
        FinishLine();
        
        // print 'else' statement block
        if(PrintStatementRange(psi, wElse, wTgt, wIndent+1))
        {
            return(0xFFFF);
        }
        
        for(w=0; w<wIndent; w++) AccumulateString("    ");
        AccumulateString("}");
        FinishLine();
    }
    
    return(wTgt-1);
}

#pragma argsused
static WORD PrintSwitch(ScriptInfo *psi, WORD wFrom, WORD wMax, WORD wIndent)
                    // Prints 'switch(){}' constructs.
                    // Returns index of last statement handled.
                    // The last statement should occur *before*
                    // 'wMax'.
                    // Returns 0xFFFF on failure.
{
    WORD wTgt; // statement after switch
    WORD wCase; // start of 'case' table
    WORD wDefault; // default target (0xFFFF for none)
    WORD wCaseCnt; // number of cases
    WORD w, v;
    DWORD dw;
    char cType; // switch argument type
    WORD wStartBlock, wEndBlock;
                   // start/end of block for current 'case'/'default' part
    
    // (1) find basic switch info
    wTgt = psi->lpstat[wFrom].wBreak;
    wCase = psi->lpstat[wFrom].wJump;
    for(wCaseCnt=0, w=wCase; w<wTgt; w++)
    {
        if(psi->lpstat[w].wJumpType==JT_CASE)
        {
            wCaseCnt++;
        }
    }
    if(psi->lpstat[wTgt-1].wJumpType==JT_DEFAULT)
    {
        wDefault = psi->lpstat[wTgt-1].wJump;
    }
    else
    {
        wDefault = 0xFFFF;
    }
    cType = psi->lpis[psi->lpstat[wFrom].wLead].cType;
    if(cType == '\0') cType = '0';
    
    // (2) print 'switch' header
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("switch(");
    w = psi->lpstat[wFrom].wLead;
    PrintArgs(psi, (WORD)(psi->lpis[w].iFirstInst), w, 1, "", FALSE);
    AccumulateString(")");
    FinishLine();
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("{");
    FinishLine();
    
    // (3) repeatedly print blocks until end-of-switch is reached
    for(wStartBlock=wFrom+1; wStartBlock<wCase; wStartBlock=wEndBlock)
    {
        // print all cases pointing to this block
        for(v=wCase; v<wCase+wCaseCnt; v++)
        {
            if(psi->lpstat[v].wJumpType==JT_CASE &&
               psi->lpstat[v].wJump == wStartBlock)
            {
                for(w=0; w<wIndent; w++) AccumulateString("    ");
                // grab case value in 'dw'
                dw = psi->lpis[psi->lpstat[v].wLead].lpdw[1];
                AccumulateString("case ");
                PrintConstant(psi, dw, cType);
                AccumulateString(":");
                FinishLine();
            }
        }
        if(wDefault == wStartBlock)
        {
            for(w=0; w<wIndent; w++) AccumulateString("    ");
            AccumulateString("default:");
            FinishLine();
        }
    
        // find end-of-block
        for(wEndBlock=wStartBlock+1; wEndBlock<wCase; wEndBlock++)
        {
            if(psi->lpstat[wEndBlock].iLabel==0)
            {
                continue;
            }
            // check if wEndBlock is targeted by any 'case' or by 'default'
            if(wEndBlock == wDefault) break;
            for(v=wCase; v<wCase+wCaseCnt; v++)
            {
                if(psi->lpstat[v].wJumpType==JT_CASE &&
                   psi->lpstat[v].wJump == wEndBlock)
                {
                    break;
                }
            }
            if(psi->lpstat[v].wJumpType==JT_CASE &&
               psi->lpstat[v].wJump == wEndBlock)
            {
                break;
            }
        }
        
        // print block
        if(PrintStatementRange(psi, wStartBlock, wEndBlock, wIndent+1))
        {
            return(0xFFFF);
        }
    }
    
    // print trailer
    for(w=0; w<wIndent; w++) AccumulateString("    ");
    AccumulateString("}");
    FinishLine();

    return(wTgt-1);
}






// EOF

