/************************************************\
* WinTex, Copyright (c) 1995 Olivier Montanuy
*         (montanuy@lannion.cnet.fr)
* With Technical help from M.Mathews and R.Paquay.
*
* All rights reserved. Any commercial  usage is
* prohibited. Parts of this code can be used in
* freeware programs, provided WinTex is credited.
* This code comes with no guaranty whatsoever.
\************************************************/

#define WINTEXMODULE 'c'

/****************************************************\
*
*
*  ALL THE FUNCTIONS HERE ARE INDEPENDENT
*  call only once at the time. not in parallel
*
\****************************************************/

#include "lbwintex.h"
/*include windows.h after lbwintex.h*/
#if defined __WINDOWS__
#include <windows.h>
#endif
#include "lbcommon.h"
#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <ctype.h>
#include <string.h>
#include <sys/stat.h>
/*time stamp*/
#include <utime.h>
/*setsize, getsize*/
#include <io.h>

Int8 Buff[MAXBUFFSZ];
/****************************************************\
*
*
*  Error Handling
*
*
\****************************************************/
static Int32 ERRcurrentWnd;
static Int16 ERRok=TRUE;
void ERRsetWnd(Int32 Wnd)
{
  ERRcurrentWnd=Wnd;
}
Int16 ERRdeclare(Int8 **Txt, Int16 type, Int16 line, Int8 module)
{
  Int8 ErrorTxt[66];
  Int16 n;
  if(ERRok==TRUE)
  { /* Error [<module><line>]*/
	 Strcpy(Buff,"An error occured [code ");
	 ErrorTxt[0]=module;ErrorTxt[1]=0;
	 Strcat(Buff,ErrorTxt);
	 ltoa(line,ErrorTxt,10);
	 Strcat(Buff,ErrorTxt);
	 Strcat(Buff,"]:\n");
	 ErrorTxt[0]='\0';
	 for(n=0;Txt[n]!=NULL;n++)
	 { if(n==(-type))
		{ Strncpy(ErrorTxt,Txt[n],64-1);
		  break;
		}
	 }
	 strcat(Buff,ErrorTxt);
	 strcat(Buff,".");
#if defined __WINDOWS__
	 MessageBox((HWND)ERRcurrentWnd,Buff,(LPCSTR)"Error",MB_ICONSTOP|MB_OK/*|MB_SYSTEMMODAL*/);
#else
  fprintf(stderr,"\n** %s **\n",Buff);
#endif
  }
  return ERR_SURE;
}
/* TRUE = disable error report
** FALSE = enable
*/
void ERRquiet(Int16 ok)
{ ERRok = (ok)? FALSE: TRUE;
}


static Int8 *ERRmsgCommon[] =
{ "",
  "The End",  /*-1*/
  "Error",
  "Internal bug",
  "Out of memory",
  "Invalid object ref.",
  "Bad parameter",
  "Bad size",
  "Bad file",
  "Bad header",
  "Bad entry",
  "Incorrect data",
  "Windows error",
  "VB picture bug.",/*"Pal create", */
  "VB picture bug.",/*"Pal select", */
  "Bad Bitmap size",
  "Bitmap is not 256 colors (8 bits)",
  "Bad Bitmap header",
  "No DIB",
  "No Palette",
  "Not a RGB bitmap",
  "No editor defined",
  "Bad WAV format",
  "Sound should be 8-bit per sample",
  "Sound should be MONO",
  "Sound should not be compressed",
  "Bad sound format",
  "Bad VOC format",
  "Bad structure",
  "This version is not supported",
  "Unknown",
	NULL
};
Int16 ERRfaultC(Int16 line,Int16 type,Int8 Module)
{
  return ERRdeclare(ERRmsgCommon,type,line,Module);
}

#define ERR_CORR   -1
#define ERR_OPENW  -2
#define ERR_OPENR  -3
#define ERR_FREAD  -4
#define ERR_FSEEK  -5
static Int8 *ERRmsgInternal[] =
{ "",
  "Corrupt Memory",
  "Can't write file",
  "Can't read file",
  "File Read error",
  "File Seek Error",
  "Unknown",
	NULL
};
	 /*Error declaration MACRO*/
#define ERRfaultInt(a) ERRfaultI(__LINE__,a,WINTEXMODULE)
Int16 ERRfaultI(Int16 Line,Int16 Type, Int8 Module)
{
  return ERRdeclare(ERRmsgInternal,Type,Line,Module);
}




/****************************************************\
*
*
*  Memory Management. For Windoze.
*
*
\****************************************************/
/*
** The MEM structure is a header with memory info
** and a checkword, that is used to store the global
** handle to memory.
** THIS STRUCTURE MUST BE PADDED 16 FOR SECURITY
** (May help with a bug of Borland C++ related to
**  tables. Foo[n] is not made into a huge pointer)
*/
#if defined __WINDOWS__
struct HMEM
{
  GLOBALHANDLE hMem;
  GLOBALHANDLE hChk;
  Int32 Size;
  Int8 Padding[16-sizeof(Int32)-2*sizeof(GLOBALHANDLE)];
};
typedef struct HMEM PTR *pHMEM;
#define Hchk(a) (GLOBALHANDLE)(~((Int32)a)+0x666L)
static pVoid MemOut(GLOBALHANDLE hmem,Int32 Size)
{ pHMEM mem;
  if(hmem==NULL) {ERRfault(ERR_MEM);return NULL;}
  mem=(pHMEM)GlobalLock(hmem);
  if(mem==NULL) {ERRfault(ERR_MEM);return NULL;}
  mem->Size= Size;
  mem->hMem= hmem;
  mem->hChk= Hchk(hmem);  	/*Security check*/
  return (pVoid )(mem+1);	/*plus sizeof(struct HMEM)*/
}
static pHMEM MemIn(pVoid Data)
{ pHMEM mem=(pHMEM)Data;
  if(mem==NULL)
  { ERRfault(BAD_PARM);return NULL;} /*Protection*/
  mem = mem -1;                   /*minus sizeof(struct HMEM)*/
  if(mem->hChk != Hchk(mem->hMem))
  { ERRfaultInt(ERR_CORR);return NULL;} /*Check fails*/
  return mem;
}
#endif /*__WINDOWS__*/

/* Emulation of Malloc*/
pVoid Malloc(Int32 Size)
{
#if defined __WINDOWS__
  GLOBALHANDLE hmem;
  if(Size<=0) return NULL;
  /*memory must be fixed*/
  hmem=GlobalAlloc(GMEM_FIXED,Size+sizeof(struct HMEM));
  return MemOut(hmem,Size);
#elif defined __DOS__
  return (pVoid)farmalloc(Size);
#else
  return (pVoid)malloc(Size);
#endif /*__WINDOWS__*/
}
/* Emulation of Malloc. MOVEABLE*/
pVoid MallocSpecial(Int32 Size)
{
#if defined __WINDOWS__
  GLOBALHANDLE hmem;
  if(Size<=0) return NULL;
  /*memory must be MOVEABLE for sndPlaySound*/
  hmem=GlobalAlloc(GMEM_MOVEABLE,Size+sizeof(struct HMEM));
  return MemOut(hmem,Size);
#elif defined __DOS__
  return (pVoid)farmalloc(Size);
#else
  return (pVoid)malloc(Size);
#endif /*__WINDOWS__*/
}

/* Emulation of Calloc*/
pVoid Calloc(Int32 Nb, Int32 Size)
{
#if defined __WINDOWS__
  pVoid ptr;
  Size= Size * Nb;
  if(Size<0) return NULL;
  ptr=Malloc(Size);
  if(ptr==NULL) return NULL;
  Memset(ptr,0,Size);
  return ptr;
#elif defined __DOS__
  return (pVoid)farcalloc(Nb,Size);
#else
  return (pVoid)calloc(Nb,Size);
#endif
}
/* Emulation of Realloc*/
pVoid Realloc(pVoid Data,Int32 Size)
{
#if defined __WINDOWS__
  GLOBALHANDLE hmem;
  pHMEM mem;
  if(Size<=0) return NULL;
  mem=MemIn(Data);
  if(mem==NULL)return NULL;
  hmem = mem->hMem;
  GlobalUnlock(hmem);
  /*memory is fixed, but can it be moved, due to GMEM_MOVEABLE*/
  hmem = GlobalReAlloc(hmem,Size+sizeof(struct HMEM),GMEM_MOVEABLE);
  return MemOut(hmem,Size);
#elif defined __DOS__
  return (pVoid)farrealloc(Data,Size);
#else
  return (pVoid)realloc(Data,Size);
#endif
}
/* Emulation of Free*/
void Free(pVoid Data)
{
#if defined __WINDOWS__
  GLOBALHANDLE hmem;
  pHMEM mem;
  /*if(Data==NULL)return;*/
  mem=MemIn(Data);
  if(mem==NULL)return;
  hmem=mem->hMem;
  GlobalUnlock(hmem);
  GlobalFree(hmem);
#elif defined __DOS__
  farfree(Data);
#else
  free(Data);
#endif
}
/*
** Size of memory object
*/
Int32 Memsize(pVoid Data)
{
#if defined __WINDOWS__
  pHMEM mem;
  mem=MemIn(Data);
  if(mem==NULL)return NULL;
  return mem->Size;
#else
  return 0;
#endif
}
/*
** Copy memory
*/
Int16 Memcpy(pVoid Data,pVoid Src,Int32 Size)
{ /*return (_fmemcpy(Data,Src,Size)!=NULL);*/
  pInt8 d=(pInt8 )Data;
  pInt8 s=(pInt8 )Src;
  Int32 i;
  if(Size<=0)return 0;
  for(i=0;i<Size;i++, d+=1, s+=1) *d=*s;
  return 1;
}
/*
** Set memory
*/
Int16 Memset(pVoid Data,Int8 Src,Int32 Size)
{ /*return (_fmemset(Data,Src,Size)!=NULL);*/
  pInt8 d=(pInt8 )Data;
  Int32 i;
  if(Size<=0)return 0;
  for(i=0;i<Size;i++,d+=1) *d=Src;
  return 1;
}
/*
** Set memory by double word
*/
Int16 MemsetL(pVoid Data,Int8 Src,Int32 Size)
{
  pInt32 d=(pInt32)Data;
  Int32 i,size;
  Int32 src;
  i=((Int32)Src)&0xFF;
  src=(((((i<<8)|i)<<8)|i)<<8)|i;
  if(Size<=0)return 0;
  size= Size/sizeof(Int32);
  for(i=0;i<size;i++,d+=1) *d=src;
  return 1;
}
/****************************************************\
*
*
*  Single-File management.
*  Only one file at the time. Avoids troubles.
*
\****************************************************/

   /*return >0 if file exists*/
static struct stat FILEst;
Int16 FILEexist(pInt8 File, Bool Dir)
{ if(File==NULL) return ERRfault(BAD_PARM);
  Strncpy(Buff, File,MAXBUFFSZ-1);
  FILEst.st_mode=0;
  if(stat(Buff,&FILEst)!=0)return FALSE;
  if((Dir==TRUE)&&(FILEst.st_mode & S_IFDIR)) return TRUE;
  if((Dir==FALSE)&&(FILEst.st_mode & S_IFREG)) return TRUE;
  return FALSE;
}
/*
** This module handle only one single file at
** the time. This is voluntary. Avoids troubles.
*/
static FILE *FILEfp;
static Int16 FILEfpOk=FALSE;
#define FMODORCREATE (4)
#define FMODORCREATETXT (5)
#define FREADTXT (6)
Int16 FILEopen(pInt8 File,Int16 Mode)
{ Int8 *mode;
  switch(Mode)
  { case FCREATE:
		mode="wb"; break;  /*create*/
	 case FMODORCREATE:               /*modify, or create*/
	 case FMODIFY:
		 mode="rb+";break;  /*modify*/
	 case FMODORCREATETXT:
		 mode="rt+";break;  /*modify*/
	 case FREADTXT:
		 mode="rt";break;
	 default:
	 case FREAD:
		 mode="rb"; break; /*read*/
  }
  if(FILEfpOk!=FALSE) return ERRfault(ERR_BUG);
  Strncpy(Buff,File,MAXBUFFSZ-1);
  FILEfp=fopen(Buff,mode);
  /*second chance, if modify or create*/
  if(FILEfp==NULL)
  { switch(Mode)
	 { case FMODORCREATE:
		 FILEfp=fopen(Buff,"wb");break;
		case FMODORCREATETXT:
		  FILEfp=fopen(Buff,"wt");break;
	 }
  }
  if(FILEfp==NULL)
  { switch(Mode)
	 { case FREAD:
		case FREADTXT:
		  return ERRfaultInt(ERR_OPENR);
		default:
		  return ERRfaultInt(ERR_OPENW);
	 }
  }
  FILEfpOk=TRUE;
  return 1;
}
Int16 FILEclose(void)
{ if(FILEfpOk!=TRUE) return ERRfault(ERR_BUG);
  fclose(FILEfp);
  FILEfpOk=FALSE;
  return 1;
}
Int32 FILEread(pInt8 Data,Int32 Start,Int32 Size)
{ static Int32 wsize,sz=0;
  static Int32 res;
  if(FILEfpOk!=TRUE) return ERRfault(ERR_BUG);
  if(Start>=0)
  { if(fseek(FILEfp,Start,SEEK_SET)!=0)
		return ERRfaultInt(ERR_FSEEK);
  }
  if(Data==NULL) return 0;
  if(Size==0) return 0;
  if(Size<0) return ERRfault(BAD_PARM);
  if(Start>0x4000000L) return ERRfault(BAD_PARM);/*64 meg*/
  if(Size>0x1000000L) return ERRfault(BAD_PARM); /*16 meg*/
  for(wsize=sz=0;wsize<Size;wsize+=sz)
  { sz = ((Size-wsize)>0x4000L)? 0x4000L:Size-wsize;
	 res=fread(&Data[wsize],1,(size_t)sz,FILEfp);
	 if(res<(size_t)sz){ break; /*no more to read*/}
  }
  if(res<0) { return ERRfaultInt(ERR_FREAD);}
  return wsize;
}
Int32 FILEwrite(pInt8 Data,Int32 Start,Int32 Size)
{ static Int32 wsize,sz=0;
  static Int32 res;
  if(FILEfpOk!=TRUE) return ERRfault(ERR_BUG);
  if(Start>=0)
  { if(fseek(FILEfp,Start,SEEK_SET)!=0)
	 { return ERRfaultInt(ERR_FSEEK); }
  }
  if(Data==NULL) return 0;
  if(Size==0) return 0;
  if((Size<0)||(Start>0x4000000L)||(Size>0x1000000L))
  { return ERRfault(BAD_PARM);} /*16 meg*/
  for(wsize=sz=0;wsize<Size;wsize+=sz)
  {
	 sz = ((Size-wsize)>0x4000L)? 0x4000L:Size-wsize;
	 res=fwrite(&Data[wsize],1,(size_t)sz,FILEfp);
	 if(res<(size_t)sz){ break; /*no more to write*/}
  }
  return wsize;
}
/*
** File: load a huge lump of data, of size Size,
** from file File, at position Start.
** into a pre-allocated memory zone, Data
** returns size read
*/
Int32 FILEreadData(pInt8 Data,Int32 Start,Int32 Size, pInt8  File)
{ Int32 res;
  res=FILEopen(File,FREAD);
  if(res<0)return res;
  res=FILEread(Data,Start,Size);
  FILEclose();
  return res; /*size read*/
}
/*
** File: load a huge lump of text, of size Size,
** from file File, at position Start.
** into a pre-allocated memory zone, Data
** returns size read
*/
Int32 FILEreadText(pInt8 Data,Int32 Start,Int32 Size, pInt8  File)
{ Int32 res;
  res=FILEopen(File,FREADTXT);
  if(res<0)return res;
  res=FILEread(Data,Start,Size);
  FILEclose();
  return res; /*size read*/
}
/*
** File: save a huge lump of data, of size Size,
** to file File, at position Start.
** from a pre-allocated memory zone, Data
** returns size written
*/                        /*pInt8 */
Int32 FILEwriteData(pInt8 Data,Int32 Start,Int32 Size, pInt8  File)
{ Int32 res;
  res=FILEopen(File,FMODORCREATE);  /*modify or create file*/
  if(res<0)return res;
  res=FILEwrite(Data,Start,Size);
  FILEclose();
  return res; /*size read*/
}
/*
** File: save a huge lump of data, of size Size,
** to file File, at position Start.
** from a pre-allocated memory zone, Data
** returns size written
*/                        /*pInt8 */
Int32 FILEwriteText(pInt8 Data,Int32 Start,Int32 Size, pInt8  File)
{ Int32 res;
  res=FILEopen(File,FMODORCREATETXT);  /*modify or create text file*/
  if(res<0)return res;
  res=FILEwrite(Data,Start,Size);
  FILEclose();
  return res; /*size read*/
}
/*
** Get a file time stamp. (date of last modification)
*/
Int32 FILEgetTime(pInt8  File)
{ struct stat statbuf;
  Strncpy(Buff,File,MAXBUFFSZ-1);
  if(stat(Buff,&statbuf)<0) return 0;
  return (Int32)statbuf.st_ctime;
}
/*
** Set a file time stamp.
*/
void FILEsetTime(pInt8  File, Int32 Time)
{ struct utimbuf stime;
  Strncpy(Buff,File,MAXBUFFSZ-1);
  stime.modtime=stime.actime=Time;
  utime(Buff, &stime);
}
/*
** get file size
*/
Int32 FILEgetSize(pInt8  File)
{ struct stat statbuf;
  Strncpy(Buff,File,MAXBUFFSZ-1);
  if(stat(Buff,&statbuf)<0) return 0L;
  return (Int32)statbuf.st_size;
}
/*
**  change size of a file
*/
Int16 FILEsetSize(pInt8  File, Int32 Size)
{ FILE *fp;
  Strncpy(Buff,File,MAXBUFFSZ-1);
  fp = fopen(Buff,"rb+");
  if(fp==NULL) return -1;
  /*ftruncate(fileno(fp), Size);*/
  chsize(fileno(fp),Size);
  fclose(fp);
  return 1;
}








/****************************************************\
*
*
*  String Handling.
*
*
\****************************************************/


/*
** string matching, case insensitive, early stop
** returns 1 if matches, 0 if fails
*/
#if 0
Int16 Strmatch(pInt8 Data,pInt8 Match,Int16 Size)
{ Int16 n;
  pInt8 m=Match;
  pInt8 d=Data;
  for(n=0;n<Size;n++, m+=1,d+=1)
  { if((*m)=='\0') break;
	 if((*d)!=(*m)) /*check if different*/
	 { if(toupper(((Int16)*d)&0xFF)!=toupper(((Int16)*m)&0xFF))
	return 0; /*fails*/
	 }
  }
  return 1; /*matches*/
}
#endif
/*
** String comparison, case insensitive
** returns 1 if match, 0 if fails.
*/
#if 1
Int16 Strncmpi(pInt8 Data,pInt8 Match,Int16 Size)
{ Int16 n;
  pInt8 m=Match;
  pInt8 d=Data;
  for(n=0;n<Size;n++,m+=1,d+=1)
  { if((*d)!=(*m)) /*check if different*/
	 { if(toupper(((Int16)*d)&0xFF)!=toupper(((Int16)*m)&0xFF))
	return 0; /*fails*/
	 }
	 if((*m)=='\0') break;
  }
  return 1; /*matches*/
}
#endif
/*
** String comparison, case sensitive
** returns 1 if match, 0 if fails.
*/
Int16 Strncmp(pInt8 Data,pInt8 Match,Int16 Size)
{ Int16 n;
  pInt8 m=Match;
  pInt8 d=Data;
  for(n=0;n<Size;n++,m+=1,d+=1)
  { if((*d)!=(*m)) /*check if different*/
	 { return 0; /*fails*/ }
	 if((*m)=='\0') break;
  }
  return 1; /*matches*/
}
/*
** Length of string
*/
Int16  Strlen(pInt8 Src, Int16 Size)
{ Int16 i;
  pInt8 s=Src;
  for(i=0;i<Size;i++,s+=1)
  { if((*s)=='\0') break;
  }
  return i;/*needs +1*/
}
pInt8 Strncpy(pInt8 Dest, pInt8 Src, Int16 Size)
{ Int16 i;
  pInt8 d=Dest;
  pInt8 s=Src;
  for(i=0;i<Size;i++,d+=1,s+=1)
  { if((*s)=='\0') break;
    *d=*s;
  }
  *d='\0';
  return Dest;
}
pInt8 Strcpy(pInt8 Dest,pInt8 Src)
{ return Strncpy(Dest,Src,0x4000-1);
}
pInt8 Strcat(pInt8 Dest, pInt8 Src)
{ Int16 i;
  pInt8 d=Dest;
  for(i=0;i<0x4000;i++,d+=1)
  { if(*d=='\0')break;
  }
  return Strncpy(d,Src,0x4000-1);
}
/*
** Append Number
*/
pInt8 StrcatNum(pInt8 Dest, Int32 Nb, Int16 Base)
{ static Int8 Asc[32+4];
  ltoa(Nb,Asc,Base);
  Asc[32]='\0';
  return Strcat(Dest,Asc);
}
/****************************************************\
*
*  Entry name Normalisation
* (remove spaces, pad with zeroes)
*
\****************************************************/

/*
** Normalize, dest of size NORMALISELEN
*/
Int16 Normalise(pInt8 Dest,pInt8 Src)
{
  Int16 n,r;
  Bool pad=FALSE;
  Int8 c='A';
  if(Dest==NULL) return BAD_PARM;
  if(Src==NULL)
  { Dest[0]='\0'; return 0;}
  for(r=n=0;n<NORMALISELEN;n++)
  { if(pad!=TRUE)
	 {
		c=Src[n]; r++;
	 }
	 if((c=='\0')||(c<=' '))
	 {
		pad=TRUE;
	 }
	 else if(isprint(c))
	 {
	  c=toupper(c);
	 }
	 else
	 { pad=TRUE;
	 }
	 c= (pad==TRUE)? '\0': c;
	 Dest[n] = c;
  }
  return r;
}
/*
** Entry names normalising
*/
Int16 Normalize(pInt8 Dest)
{
  return Normalise(Dest,Dest);
}
/*
** normalise with space. in Dest[NORMALISELEN+2]
*/
Int16 NormalizeS(pInt8 Dest,pInt8 Src)
{
  Int16 n;
  n=Normalise(Dest,Src);
  if(n<0)return n;
  for(n=0;n<NORMALISELEN;n++) if(Dest[n]=='\0') Dest[n]=' ';
  Dest[NORMALISELEN]='\0';
  return NORMALISELEN;
}
/*
** print a 4-digit number, space padded in Dest[4+1]
*/
Int16 NormalizeN(pInt8 Dest,Int16 nb)
{
  Int16 n;
  Int8 Number[16+2];
  if(Dest==NULL) return BAD_PARM;
  if(nb<-999) n=-999;else if(nb>9999) n=9999; else n=nb;
  itoa(n,Number,10);
  for(n=0;n<4;n++)
  { if(Number[n]=='\0')break;
	 Dest[n]=Number[n];
  }
  for(;n<4;n++)
  { Dest[n]=' '; }
  return 4;
}

/****************************************************\
*
*  Swap, Big Endian <-> little endian
*
\****************************************************/
static union
{ Int32 l;
  float f;
  Int16 s;
  Int8 x[4];
} SwapS;
Int32 SwapInt32(Int32 Val)
{ Int8 x;
  SwapS.l=Val;
  x=SwapS.x[3];SwapS.x[3]=SwapS.x[0];SwapS.x[0]=x;
  x=SwapS.x[2];SwapS.x[2]=SwapS.x[1];SwapS.x[1]=x;
  return SwapS.l;
}
float SwapFloat(float Val)
{ Int8 x;
  SwapS.f=Val;
  x=SwapS.x[3];SwapS.x[3]=SwapS.x[0];SwapS.x[0]=x;
  x=SwapS.x[2];SwapS.x[2]=SwapS.x[1];SwapS.x[1]=x;
  return SwapS.f;
}
Int16 SwapInt16(Int16 Val)
{ Int8 x;
  SwapS.s=Val;
  x=SwapS.x[1];SwapS.x[1]=SwapS.x[0];SwapS.x[0]=x;
  return SwapS.s;
}

