/************************************************\
* WinTex, Copyright (c) 1995 Olivier Montanuy
*         (montanuy@lannion.cnet.fr)
* With Technical help from M.Mathews and R.Paquay.
*
* All rights reserved. Any commercial  usage is
* prohibited. Parts of this code can be used in
* freeware programs, provided WinTex is credited.
* This code comes with no guaranty whatsoever.
\************************************************/

#define WINTEXMODULE 'd'

#include "lbwintex.h"
/*include windows.h after lbwintex.h*/
#if defined __WINDOWS__
#include <windows.h>
#endif
#include "lbcommon.h"
#include "lbwindoze.h"
#include "lbwad.h"
#include <stdio.h>
#include <stdlib.h>
#include "lbdraw.h"
#include "lbdispl.h"
#include "lbdoom.h"

/*
** This module can only handle up to 16 BMP at the time
** as separate objects. But registration (object creation)
** is necessary before anything. unregister, at the end.
*/

/*
** Performance can be boosted by converting the
** palette to the system palette.
**
** Wish list:
**  - normal Y order (this depends on Windoze!)
**
*/


/*
** conversion function
*/
static void Windoze2Display(pWINDOZE Win, pDISPLAY Display)
{
  if((Win==NULL)||(Win->Type!=WINDOZE_PICTBOX))
  { ERRfault(ERR_BUG);
	 Display->hWnd=NULL;
	 Display->hDC=NULL;
	 return;
  }
  Display->hWnd=(HWND)Win->hWnd; /*NULL= none*/
  Display->hDC =(HDC)Win->hDC;
  Display->hBitmap=(HBITMAP)Win->hBitmap;
  Display->hPal=(HPALETTE)Win->hPal;
  Display->SzX =Win->SzX;
  Display->SzY =Win->SzY;
  Display->OfsX=Win->OfsX;
  Display->OfsY=Win->OfsY;
}
static void Display2Windoze(pWINDOZE Win, pDISPLAY Display)
{
  if((Win==NULL)||(Win->Type!=WINDOZE_PICTBOX))
  {
    ERRfault(ERR_BUG);
	 return;
  }
  Win->hWnd=Display->hWnd;
  Win->hBitmap=Display->hBitmap;
  Win->hPal=Display->hPal;
  Win->SzX =Display->SzX;
  Win->SzY =Display->SzY;
  Win->OfsX=Display->OfsX;
  Win->OfsY=Display->OfsY;
}


/*
** structs for the BMP object
*/
#define BMP_PAL_NONE 0 /*no palette*/
#define BMP_PAL_RGB  1 /*use RGB palette from BMP object*/
#define BMP_PAL_SYS  2 /*use system palette*/
struct BMPDEF
{ pInt8   Ptr;    /*pointer to pixels*/
  Int16   SzX;	  /*width*/
  Int16   SzY;	  /*height*/
  Int32    LineSz; /*size of lines*/
  Int16   NBits;  /*number of bits*/
  Int16   Scale;  /*scale, in 8:8 fixed point*/
  /*
  ** Display
  */
  DISPLAY Display;
  /*
  ** global reference color palette
  */
  Int16  Transparent;       /*transparent index*/
  Int16  NonTransp;         /*replacement for index = transparent*/
  Int16  PalType;           /*system or this palette?*/
  Int16  PalOkay;           /*palette ready?*/
  RGB    Palette[0x100];    /*RGB, for picture display*/
  RGB    RefPal[0x100];     /*RGB, for quantise*/
};
typedef struct BMPDEF PTR *pBMPDEF;
/*
** BMPOBJ is just a mask for BMPDEF
*/
#if sizeof(struct BMPOBJ) != sizeof(struct BMPDEF)
#error size of BMPOBJ must be the same as size of BMPOBJ
#endif
/******************************************************************/
/*
**
** BMP Object registration
**
*/
Int16 BMPinitI(pBMPOBJ Bmp)
{
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) { return ERRfault(ERR_BUG);}
  /**/
  This->Scale=1<<8;    /*scale 1*/
  This->PalType=BMP_PAL_NONE;
  This->PalOkay=FALSE;
  This->Ptr=NULL;
  This->Transparent=0xF7;
  This->NonTransp=This->Transparent; /*replacement*/
  return 1;
}
/*
** last operation on Bmp object
*/
Int16 BMPfreeI(pBMPOBJ Bmp)
{
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) { return ERRfault(ERR_BUG);}
  if(This->Ptr!=NULL)
  { ERRfault(ERR_BUG);
    Free(This->Ptr);
  }
  return 1;
}
/*
** Inquire about the status of Bmp
**
** bit0: Bmp Exist
** bit1: Bmp has palette
** bit2: Bmp has data
** nit3: Bmp sizeX is reasonable
** bit4: Bmp sizeY is reasonable
** bit5: Bmp Nb of bits is correct
*/
Int16 BMPcheck(pBMPOBJ Bmp)
{
  Int16 res;
  Int32 linsz;
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) return 0;
  res=1;
  if(This->PalOkay==TRUE) res|=1<<1;
  if(This->Ptr!=NULL)
  { res|=1<<2;
	 if((This->SzX>0)&&(This->SzX<512)) res|=1<<3;
	 if((This->SzY>0)&&(This->SzY<256)) res|=1<<4;
	 switch(This->NBits)
	 { case 8:  linsz=This->SzX;break;
		case 24: linsz=This->SzX*3;break;
		case 4:  linsz=This->SzX/2;break;
		default: linsz=-1;break;
	 }
	 linsz=(linsz+3L)&(~3L);
	 if(This->LineSz==linsz) res|=1<<5;
  }
  return res;
}





/****************************************************\
*
*
*  Color Palette
*
*
\****************************************************/
/*
** Set Reference Palette and Current Palette.
*/
Int16 BMPsetPalette(pBMPOBJ Bmp,pInt8 Pal)
{ Int16 p;
  pRGB pal=(pRGB)Pal;
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) { return ERRfault(ERR_BUG);}
  /**/
  for(p=0;p<256;p++)
  { This->RefPal[p].R =pal[p].R;
	 This->RefPal[p].G =pal[p].G;
	 This->RefPal[p].B =pal[p].B;
  }
  Memcpy(This->Palette,This->RefPal,256*sizeof(RGB));
  This->PalOkay=TRUE;
  return 1;
}
/*
** Transpose Palette, from an array of 256 integer
** that say what index a given index becomes
** (as found in COLORMAP or FOGMAP)
*/
#if (DLLFORDOOM)
Int16 BMPtransposePalette(pBMPOBJ Bmp,pInt8 Trn/*[0x100]*/)
{ Int16 i,t;
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) { return ERRfault(ERR_BUG);}
  /**/
  for(i=0;i<256;i++)
  { if(i==This->Transparent)continue;
	 t=((Int16)Trn[i])&0xFF;/*transpose*/
	 Memcpy(&(This->Palette[i]),&(This->RefPal[t]),sizeof(RGB));
  }
  return 1;
}
#endif
/*
** Set Current Palette to Reference Palette
*/
Int16 BMPrestorePalette(pBMPOBJ Bmp)
{ pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) { return ERRfault(ERR_BUG);}
  /**/
  Memcpy(This->Palette,This->RefPal,256*sizeof(RGB));
  return 1;
}
/*
** Transparent color
*/
Int16 BMPgetTransparentColor(pBMPOBJ Bmp)
{
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->PalOkay==TRUE))
  { return ERRfault(ERR_BUG);}
  return ((Int16)This->Transparent)&0xFF;
}
/*
** set This->Transparent and modify the palette accordingly
** Transparent = RGB(r,g,b)
*/

Int16 BMPsetTransparentColor(pBMPOBJ Bmp,Int16 Index,Int32 Transparent)
{ static Int16 R,G,B;
  static Int16 i; Int16 inv;
  /* DOOM or HERETIC palette? */
  static Int16 Inv[] ={0xF7,0xFF,0xF7,0xFF};
  RGB Old;
  static RGB Col[] =
  { {(Int8)0,  (Int8)0,  (Int8)0, },
	 {(Int8)255,(Int8)255,(Int8)255},
	 {(Int8)0,  (Int8)255,(Int8)255},
	 {(Int8)0,  (Int8)255,(Int8)255}
  };
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) { return ERRfault(ERR_BUG);}
  if(This->PalOkay!=TRUE)
  { return ERRfault(ERR_BUG);}
  /**/
  R=(Int16)((Transparent)&0xFFL);
  G=(Int16)((Transparent>>8)&0xFFL);
  B=(Int16)((Transparent>>16)&0xFFL);
  /**/
  if(Index>=0)
  {
	 inv= Index&0xFF;
  }
  else
  {
	 for(i=0;i<4;i++)
	 {
		inv=Inv[i]&0xFF;
		if(This->RefPal[inv].R !=Col[i].R)continue;/*R*/
		if(This->RefPal[inv].G !=Col[i].G)continue;/*G*/
		if(This->RefPal[inv].B !=Col[i].B)continue;/*B*/
		break;
	 }
	 if(i>=4) i=0; /*default to DOOM*/
    inv=Inv[i]&0xFF;
  }
  /* the index that correponds to the transparent color
  ** is sometime used in pictures. it should not be.
  ** we must find an equivalent for this color
  */
  Memcpy(&Old,&(This->RefPal[inv]),sizeof(RGB));
  /*change palette*/
  This->RefPal[inv].R= (Int8)(R);
  This->RefPal[inv].G= (Int8)(G);
  This->RefPal[inv].B= (Int8)(B);
  This->Transparent=(inv&0xFF);
  This->NonTransp  =BMP8getColor(This->RefPal,inv,Old.R,Old.G,Old.B);
  Memcpy(This->Palette,This->RefPal,256*sizeof(RGB));
  return inv;
}


/****************************************************\
*
*
*  Allocate BMP, possibly from bytes
*
*
\****************************************************/


Int16 BMPinit(pBMPOBJ Bmp,Int16 SzX,Int16 SzY,Int16 NBits,pInt8 Lmp, Int16 PalTyp)
{ Int32 size;
  pBMPDEF This=(pBMPDEF)Bmp;
  /*Bmp object must exist and be void*/
  if((This==NULL)||(This->Ptr!=NULL)) { return ERRfault(ERR_BUG);}
  /**/
  (This->SzX)=SzX;
  (This->SzY)=SzY;
  (This->NBits)=NBits;
  if(NBits!=8) return ERRfault(BAD_NBIT);
  This->LineSz = (Int32)SzX;
  /*align lines on Int32 word*/
  (This->LineSz)= ((This->LineSz) + 3L) & (~3L); /*align 4*/
  /*calulate size*/
  size= (This->LineSz) * ((Int32)(This->SzY));  /*size*/
  if(Lmp!=NULL)
	 (This->Ptr)=Lmp;
  else
  { (This->Ptr)=(pInt8 )Malloc(size);
	 if(This->Ptr==NULL) return ERR_MEM;
  }
  This->PalType = (PalTyp<=0)? BMP_PAL_SYS: BMP_PAL_RGB;
  return 1;
}
/*
** Clear bitmap. if colIndex<0, use transp color
*/
Int16 BMPclear(pBMPOBJ Bmp, Int16 colIndex)
{
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->Ptr==NULL))
  { return ERRfault(ERR_BUG);}
  if(This->NBits!=8) return ERRfault(ERR_BUG);
  if(colIndex<0) colIndex=This->Transparent;
  MemsetL(This->Ptr,(Int8)(colIndex&0xFF),(This->LineSz)*((Int32)This->SzY));
  return 1;
}
/*
** Transfer control of BMP pixels to another module.
** forget it. the other module is responsible for
** releasing the memory
*/
pInt8 BMPfreeGift(pBMPOBJ Bmp,pInt16 SzX,pInt16 SzY, pInt32 LineSz)
{ pInt8 Lmp;
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->Ptr==NULL)) { ERRfault(ERR_BUG); return NULL;}
  /**/
  *LineSz=(This->LineSz);
  *SzX=(This->SzX);
  *SzY=(This->SzY);
  Lmp=(This->Ptr);
  This->Ptr=NULL;
  return Lmp;
}
/*free*/
Int16 BMPfree(pBMPOBJ Bmp)
{
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL)
  { return ERRfault(ERR_BUG);}
  if(This->Ptr==NULL)
  { return ERR_BUG;}
  /**/
  Free(This->Ptr);
  This->Ptr=NULL;
  return 1;
}
Int16 BMPsetScale(pBMPOBJ Bmp,Int16 Scale)
{
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->Ptr==NULL)) { return ERRfault(ERR_BUG);}
  /**/
  This->Scale=Scale;
  return 1;
}







/****************************************************\
*
*
*  Quantize colors
*
*
\****************************************************/

/*
** Get the index of the color closest to RGB
*/

Int8 BMPmatchColor(pBMPOBJ Bmp,Int16 R, Int16 G, Int16 B)
{
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) { return ERRfault(ERR_BUG);}
  /**/
  /*match color*/
  return BMP8getColor(This->RefPal,0,R,G,B);
}
/*
** Static ystem palette
*/
static RGB BMPsysPalette[]=
{
  {0x00,0x00,0x00}, /*BLACK*/
  {0x80,0x80,0x80}, /*DARKGREY*/
  {0xC0,0xC0,0xC0}, /*GREY*/
  {0xFF,0xFF,0xFF}, /*WHITE*/
  {0x80,0x00,0x00}, /*DARKRED*/
  {0x00,0x80,0x00}, /*DARKGREEN*/
  {0x00,0x00,0x80}, /*DARKBLUE*/
  {0x80,0x00,0x80}, /*DARKPURPLE*/
  {0x80,0x80,0x00}, /*DARKYELLOW*/
  {0x00,0x80,0x80}, /*DARKCYAN*/
  {0xFF,0x00,0x00}, /*RED*/
  {0x00,0xFF,0x00}, /*BLUE*/
  {0x00,0x00,0xFF}, /*GREEN*/
  {0xFF,0x00,0xFF}, /*PURPLE*/
  {0xFF,0xFF,0x00}, /*YELLOW*/
  {0x00,0xFF,0xFF}, /*CYAN*/
};

/****************************************************\
*
*
*  BMP read or obtain
*
*
\****************************************************/


/*
** Read Bitmap File into Bmp structure
** return 1 if ok
*/
Int16 BMPinitBitmap(pBMPOBJ Bmp, pInt8 File)
{
  Int16 SzX,SzY;
  pInt8 Ptr;
  pBMPDEF This=(pBMPDEF)Bmp;
  /*Bmp must exist and be void*/
  if((This==NULL)||(This->Ptr!=NULL)) { return ERRfault(ERR_BUG);}
  /*
  ** Read the Bmp
  */
  Ptr=BMP8readBMP(&SzX, &SzY, This->RefPal, File);
  if(Ptr==NULL) return BAD_HEAD;
  /*
  ** Init the Bmp object
  */
  return BMPinit(Bmp, SzX, SzY, 8, Ptr,1);
}



/*
** Read Playpal into BMP, Alloc BMP, Don't free pic
** Addapt to current palette, and return the offsets.
*/
#if 0
Int16 BMPinitPlaypal(pBMPOBJ Bmp,pInt8 Pal, Int32 PalSz)
{ /*entry pic buffer*/
  Int32 res;
  Int16 SzY,y,x;
  Int32 yl,ypal;
  pInt8 pbmp,ppal;
  pBMPDEF This=(pBMPDEF)Bmp;
  /*Bmp must exist and be void*/
  if((This==NULL)||(This->Ptr!=NULL))
  { return ERRfault(ERR_BUG);}
  /*security*/
  if((Pal==NULL)||(PalSz<0x300))
  { return BAD_PARM;}
  /**/
  SzY=(Int16)((PalSz/0x300)&0xFF);
  /*allocate a Bmp*/
  res=BMPinit(Bmp,0x100,SzY,8,NULL); /*8 bit*/
  if(res<0)
  { return (Int16)res;}
  /*get that Bmp*/
  /*convert entry to BMP. normal Y order*/
  for(y=0,yl=(This->LineSz)*(Int32)((This->SzY)-1),ypal=0;
		y<(This->SzY); y++,yl-=(This->LineSz),ypal+=0x300)
  {
	 /*copy palette*/
	 for(x=0,pbmp=&(This->Ptr)[yl],ppal=&Pal[ypal];
		  x<0x100; x++,pbmp+=1,ppal+=3)
	 { /*convert transparent index*/
		*pbmp = BMPmatchColor(Bmp,ppal[0],ppal[1],ppal[2]);
	 }
  }
  return 1;
}
#endif /*0*/
/*
** Read a Flat into a BMP
** Don't free pic
*/
Int16 BMPinitFlat(pBMPOBJ Bmp,pInt8 Flat,Int16 SzX, Int16 SzY)
{ /*entry pic buffer*/
  Int32 res,yl,yfl;
  Int16 y,x;
  pInt8 pbmp,pflat;
  Int8 trans,notrans;
  pBMPDEF This=(pBMPDEF)Bmp;
  /*Bmp must exist and be void*/
  if((This==NULL)||(This->Ptr!=NULL))
  { return ERRfault(ERR_BUG);}
  /**/
  /*security*/
  if(Flat==NULL)
  { return BAD_PARM;}
  /*init old transp index conversion*/
  trans=(Int8)This->Transparent;
  notrans=(Int8)This->NonTransp;
  /*get szX,szY, from size*/
  /*allocate a Bmp*/
  res=BMPinit(Bmp,SzX,SzY,8,NULL,1); /*8 bit*/
  if(res<0)
  { return (Int16)res;}
  /*get that Bmp*/
  /*convert entry to BMP. normal Y order*/
  for(y=0,yfl=0,yl=(This->LineSz)*(Int32)((This->SzY)-1);
		y<(This->SzY);y++,yl-=(This->LineSz),yfl+=SzX)
  {
	 /*copy, taking into account transparent color*/
	 for(x=0,pbmp=&(This->Ptr)[yl],pflat=&Flat[yfl];
		  x<SzX; x++,pbmp+=1,pflat+=1)
	 { /*convert transparent index*/
		*pbmp = ((*pflat)==trans)? notrans : (*pflat);
	 }
  }
  return 1;
}


/*
** Read a Duke Flat into a BMP
** Don't free pic
*/
#if (DLLFORDUKE)
Int16 BMPinitDukePic(pBMPOBJ Bmp,pInt8 Flat,Int16 SzX, Int16 SzY)
{ /*entry pic buffer*/
  Int32 res,yl,yfl;
  Int16 y,x;
  pInt8 pbmp;
  pBMPDEF This=(pBMPDEF)Bmp;
  /*Bmp must exist and be void*/
  if((This==NULL)||(This->Ptr!=NULL))
  { return ERRfault(ERR_BUG);}
  /**/
  /*security*/
  if(Flat==NULL)
  { return BAD_PARM;}
  /*get szX,szY, from size*/
  /*allocate a Bmp*/
  res=BMPinit(Bmp,SzX,SzY,8,NULL,1); /*8 bit*/
  if(res<0)
  { return (Int16)res;}
  /*get that Bmp*/
  /*convert entry to BMP. normal Y order*/
  for(y=0,yl=(This->LineSz)*(Int32)((This->SzY)-1);
		y<(This->SzY);
		y++,yl-=(This->LineSz))
  {
	 /*copy, taking into account transparent color*/
	 for(x=0,pbmp=&(This->Ptr)[yl], yfl=0;
		  x<SzX; x++,pbmp+=1,yfl+=SzY)
	 { /*convert transparent index*/
		*pbmp=Flat[y+yfl];
	 }
  }
  return 1;
}
#endif /*DLLFORDUKE*/


/*
** Read DoomPIC into BMP, Alloc BMP
** Don't free pic
** Don't report errors (because flat comes after...)
*/
#if (DLLFORDOOM)
Int16 BMPinitDoomPic(pBMPOBJ Bmp,pInt8 Pic, Int32 Size, pInt16 Ofsx, pInt16 Ofsy)
{ /*doom pic buffer*/
  pDMPICHEAD Head;
  /*doom pic scanning*/
  Int16 szX,szY, setc,setcount, x,y;
  pInt8 Set;
  Int32  setpos,yl,res;
  static Int8 col,trans,notrans;
  pBMPDEF This=(pBMPDEF)Bmp;
  /*Bmp must exist and be void*/
  if((This==NULL)||(This->Ptr!=NULL)) { return ERRfault(ERR_BUG);}
  /**/
  /*security*/
  if(Pic==NULL)return BAD_PARM;
  /*init old transp index conversion*/
  trans=(Int8)This->Transparent;
  notrans=(Int8)This->NonTransp;
  /*check DOOM pic*/
  Head=(pDMPICHEAD)Pic;
  szX=Head->SzX;
  szY=Head->SzY;
  *Ofsx= Head->OfsX;
  *Ofsy= Head->OfsY;
  if(szX<=0)  {return BAD_HEAD;}
  if(szX>512) {return BAD_HEAD;}
  if(szY<=0)  {return BAD_HEAD;}
  if(szY>256){return  BAD_HEAD;}
  /*Alloc BMP */
  res = BMPinit(Bmp,szX,szY,8,NULL,1); /*8 bit Bmp*/
  if(res<0)
  { return (Int16)res;}
  /* Clear Bmp with transp. color, load default palette*/
  BMPclear(Bmp,This->Transparent);
  /* Convert DOOM pic into BMP*/
  setpos=sizeof(struct DMPICHEAD)-sizeof(Int32)+szX*sizeof(Int32);
  if(setpos>Size)   /*beyond limits. error*/
  { BMPfree(Bmp); return BAD_ENTRY;}
  /*read each column*/
  for(x=0;x<szX;x++)
  { setpos=Head->Column[x];
	 /*for each set in column*/
	 while(1)
	 { if(setpos>Size)   /*beyond limits. error*/
		{ BMPfree(Bmp); return BAD_ENTRY;}
		Set=(pInt8 )(&Pic[setpos]);
		if(Set[0]==(Int8)0xFF) break; /*last code*/
		setcount=((Int16)Set[1])&0xFF;
		if(setpos+3+setcount+1>=Size)  /*beyond limits. error*/
		{ BMPfree(Bmp); return BAD_ENTRY;}
		y=((Int16)Set[0])&0xFF;
		yl=(This->LineSz)*((Int32)((This->SzY)-y-1));
		for(setc=0;setc<setcount;setc++,y++,yl-=(This->LineSz))
		{ if(y>=szY) /*beyond pic limits. error*/
	{ BMPfree(Bmp); return BAD_ENTRY;}
	col=Set[3+setc];
	/*convert transparent index*/
	if(col==trans) col=notrans;
	(This->Ptr)[((Int32)x)+yl]=col;
		}
		setpos+=(3+setcount+1);
	 }
  }
  return 1;
}
#endif




/*
** Read a MIP Tex into a BMP
** Don't free Lmp
*/
#if (DLLFORQUAK)
Int16 BMPinitMipTex(pBMPOBJ Bmp, pInt8 Lmp, Int32 LmpSz, Int16 SzX, Int16 SzY, pInt32 Offs)
{ /*entry pic buffer*/
  Int16 n,y,x0;
  Int32 res,yl,yfl;
  pBMPDEF This=(pBMPDEF)Bmp;
  /*Bmp must exist and be void*/
  if((This==NULL)||(This->Ptr!=NULL)||(Lmp==NULL)||(Offs==NULL))
  { return ERRfault(ERR_BUG);}
  /*
  ** Allocate empty Bmp
  */
  res=BMPinit(Bmp,SzX+(SzX>>1)+(SzX>>2)+(SzX>>3),SzY,8,NULL,1); /*8 bit*/
  if(res<0)
  { return (Int16)res;}
  BMPclear(Bmp,0xFF);
  /*
  ** Copy all Mips
  */
  for(n=0, x0=0; n<4; n++)
  {
	 /*
	 ** Check bmp start is correct
	 */
	 if((Offs[n]<0)||(Offs[n]+((Int32)SzX)*((Int32)SzY)>LmpSz))
	 { continue; }
	 /*
	 ** Convert picture to BMP.
	 */
	 for(y=0,yl=x0+(This->LineSz)*(Int32)((This->SzY)-1),yfl=Offs[n];
		  y<SzY;  y++,yl-=(This->LineSz),yfl+=SzX)
	 { Memcpy(&(This->Ptr)[yl],&Lmp[yfl],SzX); }
	 x0+=SzX;
	 SzX>>=1;
	 SzY>>=1;
  }
  /*load default palette*/
  BMPrestorePalette(Bmp);
  return 1;
}
#endif /*DLLFORQUAKE*/
















/****************************************************\
*
*
*  Modify BMP
*
*
\****************************************************/


/* Assuming the color palette is the same, this pastes some
** bitmap onto the current bitmap, at offsets OfsX, OfsY.
** The transparent color must be the same.
*/
#if (DLLFORDOOM)
Int16 BMPputBmp(pBMPOBJ Bmp,Int16 OfsX,Int16 OfsY,pInt8 Pat,Int16 PatSzX,Int16 PatSzY)
{ Int32 PatLineSz;
  Int16 xp,yp,xt,yt; /*coordinate*/
  /*Int32 linp,lint;*/ /*line position. y*PatLineSz  yt*This->LineSz*/
  Int8 col,transp;
  pInt8 Ptr;
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->Ptr==NULL)) { return ERRfault(ERR_BUG);}
  /**/
  /*must be 8-bit*/
  if(This->NBits!=8) ERRfault(ERR_BUG);
  /*get transparent color*/
  transp=((Int16)This->Transparent)&0xFF;
  /*restore line size, assuming 8-bit*/
  PatLineSz=((Int32)PatSzX+3) & (~3L);
  /*Paste Bmp on current Bmp*/
  /* (xp,yp) = [xp+linp] = position in Bmp space */
  /* (xt,yt) = [xt+lint] = position in texture space*/
  Ptr=&(This->Ptr[(This->SzY-1-OfsY)*This->LineSz]);
  Pat=&(Pat[PatLineSz*(PatSzY-1)]);
  for( yp =0,   /*linp =PatLineSz*(PatSzY-1),*/
		 yt =OfsY/*,lint =(This->SzY-1-OfsY)*This->LineSz*/;
		 yp<PatSzY;
		 yp++,   /*  linp-=PatLineSz,*/    Pat-=PatLineSz,
		 yt++,   /*  lint-=This->LineSz*/ Ptr-=This->LineSz )
  { if(yt>=This->SzY)continue;
	 if(yt<0)continue;
	 for(xp=0,xt=OfsX; xp<PatSzX; xp++,xt++)
	 { if(xt>=This->SzX)continue;
		if(xt<0)continue;
		col=Pat[(Int32)xp/*+linp*/];
		/*if transparent, don't paste*/
		if(col!=transp)/*This->*/Ptr[(Int32)xt/*+lint*/]=col;
	 }
  }
  return 1;
}
#endif /*DLLFORDOOM*/

/*
** Draw a line on a Bmp (Lame version. Aliasing...)
**
** Warning: origin is at the lower-left corner, not upper left
** Line starts from sX,sY and goes to eX,eY
**
*/
Int16 BMPline(pBMPOBJ Bmp,Int16 sX,Int16 sY,Int16 eX,Int16 eY,Int16 Color)
{
  pBMPDEF This=(pBMPDEF)Bmp;
#if 0
  if((This==NULL)||(This->Ptr==NULL)||(This->NBits!=8))
  { return ERRfault(ERR_BUG);}
#endif
  return DRWline(This->Ptr,This->LineSz, This->SzX,This->SzY, sX, sY, eX, eY, Color);
}










/****************************************************\
*
*
*  BMP data retrieval
*
*
\****************************************************/

/*
** write a BMP to a file, free BMP
*/
Int16 BMPwrite(pBMPOBJ Bmp,pInt8  File)
{ Int16 res;
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->Ptr==NULL)) { return ERRfault(ERR_BUG);}
  /**/
  res=BMP8writeBMP(This->Ptr,This->SzX,This->SzY, This->RefPal, File);
  /*
  ** return error if needed
  */
  if(res<0)
  { return (Int16)res;}
  return 1;
}
/*
**
*/
#if (DLLFORDOOM)
Int16 BMPisFlat(pBMPOBJ Bmp,Int16 Type)
{
  pBMPDEF This=(pBMPDEF)Bmp;
  if(This==NULL) { return ERRfault(ERR_BUG);}
  /**/
  if(Type>0)  /*DOOM flats: 64x64 or 64x128*/
  {
	 if(This->SzX==64)
	 { if((This->SzX==64)||(This->SzY==128))
		{ return 1;}
	 }
  }
  else        /*ordinary flats: 320x200*/
  {
	 if((This->SzX==320)&&(This->SzY==200))
	 { return 1;}
  }
  return -1;   /*fail*/
}
#endif
/*
** Get a Flat from a BMP
*/
pInt8 BMPgetFlat(pBMPOBJ Bmp,pInt32 pSize)
{ /*entry pic buffer*/
  static pInt8 flat;
  Int16 y;
  Int32 Size,yl,yfl;
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->Ptr==NULL)) { ERRfault(ERR_BUG); return NULL;}
  /**/
  /*Get size of BMP*/
  Size=((Int32)(This->SzX))*(Int32)(This->SzY);
  if(Size<0) {ERRfault(BAD_PARM);return NULL;}
  /*Get entry memory*/
  flat=Malloc(Size);
  if(flat==NULL)
  { return NULL;}
  /*convert BMP to entry, normal Y order*/
  for(yfl=0,y=0,yl=(This->LineSz)*((This->SzY)-1),yfl=0;
		y<(This->SzY); y++,yl-=(This->LineSz),yfl+=(This->SzX))
  { Memcpy(&flat[yfl],&(This->Ptr)[yl],(This->SzX));
  }
  /*return flat and size*/
  *pSize=Size;
  return flat;
}


/*
** Get a DOOM pic from a BMP
*/
#if (DLLFORDOOM)
pInt8 BMPgetDoomPic(pBMPOBJ Bmp,pInt32 pSize,Int16 OfsX,Int16 OfsY)
{ /*entry pic buffer*/
  pInt8 Pic;
  pDMPICHEAD Head;
  pInt8 Set;
  Int32 Size,colnpos,setpos,yl;
  Int16 x,y,setcount;
  Int8 pix,lastpix;
  Int8 transp;
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->Ptr==NULL)) { ERRfault(ERR_BUG); return NULL;}
  /**/
  transp = This->Transparent;
  /*security*/
  if(This->Ptr==NULL) { ERRfault(ERR_BUG);return NULL;}
  /*expansion*/
  colnpos= sizeof(struct DMPICHEAD)-sizeof(Int32)+((Int32)(This->SzX))*sizeof(Int32);
  /* worst expansion when converting from PIXEL column to
  ** list of sets: (5*Ysize/2)+1, corresponding to a dotted vertical
  ** transparent line. Ysize/2 dots, 4overhead+1pix, 1 last FF code.
  ** but if too big, allow only a 10 split mean per line
  */
  Size=colnpos + ((Int32)(This->SzX)) * (1+5*(((Int32)(This->SzY)+1)/2));
  if(Size<=0)
  { ERRfault(BAD_PARM);return NULL;}
  /*Get entry memory*/
  Pic=Malloc(Size);
  if(Pic==NULL) { return NULL;}
  Head=(pDMPICHEAD)Pic;
  Head->SzX=(This->SzX);
  Head->SzY=(This->SzY);
  Head->OfsX=(OfsX!=INVALIDINT)? OfsX: (This->SzX/2);
  Head->OfsY=(OfsY!=INVALIDINT)? OfsY: (This->SzY-5);
  /*convert BMP to DOOM PIC, normal Y order*/
  for(x=0;x<(This->SzX);x++)
  { Head->Column[x]=colnpos;
	 setpos=0;
	 lastpix=transp;
	 for(y=0,yl=(This->LineSz)*((This->SzY)-1); y<(This->SzY); y++,yl-=(This->LineSz))
	 { /*get column pixel on Bmp*/
		pix=(This->Ptr)[yl+((Int32)x)];
		/*check if begining of set*/
		if(pix!=transp)
		{ if(lastpix==transp)  /*begining of set*/
	{ Set=(pInt8 )&(Pic[colnpos+setpos]);
	  setcount=0;
	  Set[0]=(y&0xFF);      /*y position*/
	  Set[1]=0;                 /*count (updated later)*/
	  Set[2]=pix;       /*unused*/
	}
	Set[3+setcount]=pix;/*non transparent pixel*/
	setcount++;          /*update count of pixel in set*/
		}
		else /*pix is transparent*/
		{ if(lastpix!=transp)/*finish the current set*/
	{ Set[1]=(setcount&0xFF);
	  Set[3+setcount]=lastpix;
	  setpos+= 3+setcount+1;/*1pos,1count,1dummy,setcount pixels,1dummy*/
	}
	/*else: not in set but in transparent area*/
		}
		lastpix=pix;
	 }
	 if(lastpix!=transp)      /*finish current set, if any*/
	 { Set[1]=(setcount&0xFF);
		Set[3+setcount]=lastpix;
		setpos+= 3+setcount+1;        /*1pos,1count,1dummy,setcount pixels,1dummy*/
	 }
	 Pic[colnpos+setpos]=(Int8)0xFF; /*end of all sets*/
	 colnpos+=(Int32)(setpos+1);          /*position of next column*/
  }
  /*Get actual Pic size, and adjust*/
  Size=colnpos;
  Pic=Realloc(Pic,Size);
  /*Return Pic*/
  *pSize=Size;
  return Pic;
}
#endif /*DLLFORDOOM*/



/****************************************************\
*
*
*  Display BMP
*
*
\****************************************************/
/*
** Show Bmp
*/
Int16 BMPshowWindoze(pWINDOZE Pic,Int16 Ops)
{ DISPLAY Display;
  Windoze2Display(Pic,&Display);
  return BMP8display(&Display,Ops);
}
/*
** Display, using Palette, not reference Palette
*/
Int16 BMPassignToWindoze(pBMPOBJ Bmp, pWINDOZE Pic)
{ Int16 res;
  pBMPDEF This=(pBMPDEF)Bmp;
  if((This==NULL)||(This->Ptr==NULL))
  { return ERR_BUG;}
  /**/
  /*
  ** Display
  */
  Windoze2Display(Pic,&(This->Display));
  switch(This->PalType)
  { case BMP_PAL_RGB:
		res=BMP8create(&(This->Display),This->Ptr,This->SzX,This->SzY,This->Palette,256);
		break;
	 case BMP_PAL_SYS:
		res=BMP8create(&(This->Display),This->Ptr,This->SzX,This->SzY,BMPsysPalette,sizeof(BMPsysPalette)/sizeof(RGB));
		break;
  }
  Display2Windoze(Pic,&(This->Display));
  /*
  ** return if error
  */
  if(res<0) return res;
  return 1;
}

/****************************************************\
*
*
* DISPLAY OBJECT INTERNAL LIST API
*
\****************************************************/
#if 0
Int16 LstClear(pWINDOZE Win)
{
#if defined __WINDOWS__
  if((Win==NULL)||(Win->hWnd==NULL))
  { return ERRfault(BAD_PARM);}
  switch(Win->Type)
  {
	 case WINDOZE_LISTBOX:
		return LISTclear((HWND)Win->hWnd);
	 case WINDOZE_DROPBOX:
		return CMBXclear((HWND)Win->hWnd);
  }
#endif
  return ERRfault(BAD_PARM);;
}

Int16 LstSet(pWINDOZE Win,Int16 Index,Int32 Code,pInt8 Text)
{
#if defined __WINDOWS__
  if((Win==NULL)||(Win->hWnd==NULL))
  { return ERRfault(BAD_PARM);}
  switch(Win->Type)
  {
	 case WINDOZE_LISTBOX:
		return LISTset((HWND)Win->hWnd,Index,Code,Text);
	 case WINDOZE_DROPBOX:
		return CMBXset((HWND)Win->hWnd,Index,Text);
  }
#endif
  return ERRfault(BAD_PARM);
}

#if 1
Int16 LstCount(pWINDOZE Win)
{
#if defined __WINDOWS__
  if((Win==NULL)||(Win->hWnd==NULL))
  { return ERRfault(BAD_PARM);}
  switch(Win->Type)
  {
	 case WINDOZE_LISTBOX:
		return LISTcount((HWND)Win->hWnd);
	 case WINDOZE_DROPBOX:
		return CMBXcount((HWND)Win->hWnd);
  }
#endif /*__WINDOWS__*/
  return ERRfault(BAD_PARM);
}
#endif /**/

#if 1
Int16 LstGet(pWINDOZE Win,Int16 Index,pInt8 Text,Int16 TextSz)
{
#if defined __WINDOWS__
  if((Win==NULL)||(Win->hWnd==NULL))
  { return ERRfault(BAD_PARM);}
  switch(Win->Type)
  {
	 case WINDOZE_LISTBOX:
		return LISTget((HWND)Win->hWnd,Index,Text,TextSz);
	 case WINDOZE_DROPBOX:
		return CMBXget((HWND)Win->hWnd,Index,Text,TextSz);
  }
#endif /*__WINDOWS__*/
  return ERRfault(BAD_PARM);
}
#endif /*0*/


/********************************************************/
/*
** List Box Handling
*/
/* Add a NORMALISELEN-Int8 text entry to a list*/
static Int8 LBnam[NORMALISELEN+24];
Int16 LstAdd8(pWINDOZE Wnd,Int32 Code,pInt8 Name)
{
  Normalise(LBnam,Name);
  LBnam[NORMALISELEN]='\0';
  return LstSet(Wnd,-1,Code,LBnam); /* -1 = add */
}
/* Add a NORMALISELEN-Int8 text entry to a list*/
Int16 LstAdd8XY(pWINDOZE Wnd,Int32 Code,pInt8 Name, Int16 X, Int16 Y)
{
  NormalizeS(LBnam,Name);
  LBnam[NORMALISELEN]=' ';
  NormalizeN(&LBnam[NORMALISELEN+1],X);
  LBnam[NORMALISELEN+1+4]=' ';
  NormalizeN(&LBnam[NORMALISELEN+1+4+1],Y);
  LBnam[NORMALISELEN+1+4+1+4]='\0';
  return LstSet(Wnd,-1,Code,LBnam); /* -1 = add */
}
#endif



