/************************************************\
* WinTex, Copyright (c) 1995 Olivier Montanuy
*         (montanuy@lannion.cnet.fr)
* With Technical help from M.Mathews and R.Paquay.
*
* All rights reserved. Any commercial  usage is
* prohibited. Parts of this code can be used in
* freeware programs, provided WinTex is credited.
* This code comes with no guaranty whatsoever.
\************************************************/
  
#define WINTEXMODULE 'D'


#include "lbwintex.h"
#include "lbcommon.h"
#include <stdlib.h>
#include "lbwad.h"
#include "lbdispl.h"

#include "lbwaddef.h"
#include "lbdoom.h"
#include "lbwaddir.h"
#include "lbwadir.h"
#include "lbwadid.h"
#include "lbtexu.h"
#include "lblevel.h"
#include "lbmusic.h"
#include "lbqklump.h"
#include "lbqkbsp.h"


/****************************************************\
*
*
*  WAD OBJECT API
*
*
\****************************************************/


  /*
  ** register, unregister WAD object
  */
static Int16 WADregister(Int16 Main);
static void WADunregister(Int16 Self);
  /*
  ** Add data into a WAD file.
  ** (only this function is allowed to add data)
  */
static Int32 WADaddDataI(pWADDEF This, pInt8 Lmp, Int32 LmpSz, Int16 Entry);
  /*
  ** Initialisations
  */
static Int16 WADprepareIwadI(pWADDEF This, Int32 Transparent);
static Int16 WADpreparePwadI(pWADDEF This, Int32 Transparent);
#if (DLLFORQUAK)
static void WADprepareWad2I(pWADDEF This, Int32 Transparent);
static void WADpreparePackI(pWADDEF This,Int32 Transparent);
#endif
static void WADprepareKenI(pWADDEF This);









/****************************************************\
*
*
*  Registration of WAD objects
*
*
\****************************************************/


#define MAXWADDEF  (24)    /*nb max of objects*/
static pWADDEF WadReg[]=
{ NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
  NULL
};
/*
** Get the reference of an object
*/
pWADDEF WADgetThat(Int16 Self)
{ pWADDEF This;
  if((Self<0)||(Self>MAXWADDEF))
  {ERRfault(BAD_PARM);return NULL;}
  This=WadReg[Self];
  if(This==NULL)
  {ERRfault(ERR_OBJREF);return NULL;}
  return WadReg[Self];
}
pWADDEF WADgetThis(Int16 Self)
{ pWADDEF This;
  if((Self<0)||(Self>MAXWADDEF))
  {ERRfault(BAD_PARM);return NULL;}
  This=WadReg[Self];
  if(This==NULL)
  {ERRfault(ERR_OBJREF);return NULL;}
  ERRsetWnd(This->ErrWnd); /*error*/
  return WadReg[Self];
}
/*
** Register a WAD
** Main = reference of main WAD.
*/
static Int16 WADregister(Int16 Main)
{  Int16 reg;
	pWADDEF This;
  /*Find a free WAD DEF*/
  for(reg=0;reg<MAXWADDEF;reg++)
	 if(WadReg[reg]==NULL) break;
  if(reg>=MAXWADDEF) return ERRfaultWad(ERR_WADNB);
  This=(pWADDEF )Malloc(sizeof(struct WADDEF));
  if(This==NULL) return ERR_MEM;
  This->Lst=NULL;
  This->DirDirty=FALSE;
  /*main wad*/
  if((Main>=0)&&(Main<MAXWADDEF))
	 This->IwadSelf=Main;
  else
	 This->IwadSelf=-1;
  /*request a BMP object*/
  if(BMPinitI(&(This->Bmp))<0) return ERR_BUG;
  /*everything ok? then declare object*/
  WadReg[reg]=This;  /*WADsetThis(Self,This)*/
  return reg;
}
/*
** Unregister. called when releasing WAD dir.
*/
static void WADunregister(Int16 Self)
{  pWADDEF This;
  if((Self<0)||(Self>MAXWADDEF)) {ERRfault(BAD_PARM);}
  /*do not warn if already unregistered*/
  This=WadReg[Self];
  if(This==NULL)  {return;}
  BMPfreeI(&(This->Bmp));
  Free(This);
  WadReg[Self]=NULL;/*WADsetThis(Self,NULL)*/
}


/****************************************************\
*
*
*  Checks
*
*
\****************************************************/
Int16 EXPORT WADsetCustoms(Int16 Self, Int32 ErrWnd)
{ pWADDEF This;
  This=WADgetThis(Self);
  /**/
  if(This==NULL)
  { return BAD_PARM;}
  /*
  ** ErrWnd Handle
  */
  if(ErrWnd!=NULL)
  { This->ErrWnd = ErrWnd;
  }
  return 1;
}

/*
** Check WAD object
**  bit0= WAD exist
**  bit1= MainWAD exists
**  bit2= has directory
**  bit3= size is correct
**  bit4= game is valid
**
**  bit8= Bmp exist
**  bit9= Bmp has data
*/
Int16 EXPORT WADcheck(Int16 Self)
{ Int16 res;
  pWADDEF This=WADgetThis(Self);
  if((Self<0)||(Self>MAXWADDEF)) return 0;
  This=WadReg[Self];
  if(This==NULL) return 0;
  res=1;
  if(This->IwadSelf>=0)
  { if(WADgetThat(This->IwadSelf)!=NULL)
	  res|=1<<1;
  }
  if(This->Lst!=NULL) res|=1<<2;
  if((This->LstNb>0)&&(This->LstNb<0x4000)) res|=1<<3;
  if((This->Game&GAM_MASK)!=GAM_NONE) res|=1<<4;
  res|= (BMPcheck(&(This->Bmp)))<<8;
  return res;
}
/*
** Tell if a WAD can be modified
*/
Int16 WADmodifiableI(pWADDEF This)
{ /*
  ** do not modify IWAD
  */
  if((This->WadType==IS_IWAD)||(This->WadType==IS_KENS))
  { return ERRfaultWad(ERR_MDF_IWAD);}
  /*
  ** do not modify locked WAD
  */
  if(This->Locked!=FALSE)
  { return ERRfaultWad(ERR_WADLOCK);}
  return 1;
}


/****************************************************\
*
*
*  Special handling of IWAD, PWAD and WAD2
*
*
\****************************************************/
/*
** Init various data by reading IWAD. necessary.
**   0x10=DOOM 0x11=DOOM1 0x12=DOOM2
**   0x20=HERETIC 0x21=HERETIC1 0x22=HERETIC2
** init PNAMES, TEXTURES, ....
*/
#if (DLLFORDOOM)
static Int16 WADprepareIwadI(pWADDEF This, Int32 Transparent)
{
	Int16 game;
	pInt8 Pal;
	Int32 PalSz;
	if(This->IwadSelf!=-1) return ERRfault(ERR_BUG);
	if(This->Lst==NULL) return ERRfault(ERR_BUG);
	This->Game=GAM_NONE;
	game=GAM_NONE|GAM_SHAR;
	if(WADDfindEntryI(This,0,"ENDOOM",-1)>0)
	{
	  game = GAM_DOOM;
	  if(WADDfindEntryI(This,0,"MAP01",-1)>0)
	  { game |= GAM_CMRC; }
	  else if(WADDfindEntryI(This,0,"E1M1",-1)>0)
	  { game |= GAM_REGS; }
	}
	else if(WADDfindEntryI(This,0,"FOGMAP",-1)>0)
	{ game = GAM_HEXN | GAM_CMRC;
	}
	else if(WADDfindEntryI(This,0,"ENDTEXT",-1)>0)
	{
	  game = GAM_HTIC;
	  if(WADDfindEntryI(This,0,"E1M1",-1)>0)
	  { game |= GAM_REGS; }
	}
	else if(WADDfindEntryI(This,0,"ENDSTRF",-1)>0)
	{
	  game = GAM_STRF | GAM_CMRC;
	}
  /*determine DOOM,DOOM2,HERETIC*/
  if((game & GAM_MASK) == GAM_NONE)
  {
	 return ERRfaultWad(ERR_BADIWAD);
  }
/*
  if((game & (~GAM_MASK)) == GAM_SHAR)
  {
	 ERRfaultWad(SHR_IWAD);
  }
*/
  This->Game=game;
  /*
  ** Set palette
  */
  Pal=WADreadObligEntryI(This,&PalSz,"PLAYPAL");
  if(Pal==NULL)
  { return ERRfaultWad(ERR_PALNONE);}
  if(PalSz>=0x300)
  { BMPsetPalette(&(This->Bmp),Pal);}
  Free(Pal);
  /*transparent color, modif color pal*/
  BMPsetTransparentColor(&(This->Bmp),-1,Transparent);
  return game;
}
/*
** Prepare the PWAD, copy the palette from mainWAD
*/
static Int16 WADpreparePwadI(pWADDEF This, Int32 Transparent)
{  pWADDEF That;
	pInt8 Pal;
	Int32 PalSz;
	Int16 res;
  /*Get the Main WAD object reference.*/
  That=WADgetThat(This->IwadSelf);
  if(That!=NULL)
  { This->Game = That->Game; }
  Pal=WADreadObligEntryI(This,&PalSz,"PLAYPAL");
  if(Pal==NULL)
  { return ERRfaultWad(ERR_PALNONE);}
  if(PalSz>=0x300)
  { res=BMPsetPalette(&(This->Bmp),Pal);}
  Free(Pal);
  /*transparent color, modif color pal*/
  BMPsetTransparentColor(&(This->Bmp),-1,Transparent);
  return res;
}
#endif
/*
** Prepare WAD2
*/
#if (DLLFORQUAK)
static void WADprepareWad2I(pWADDEF This, Int32 Transparent)
{ pInt8 Pal;
  Int32 PalSz;
  Pal=WADreadObligEntryI(This, &PalSz,"PALETTE");
  /*set default grey palette if needed*/
  if((Pal==NULL)||(PalSz<0x300))
  { Pal=WADQpalGetLmp(0x300);}
  if(Pal==NULL)
  { return;}
  BMPsetPalette(&(This->Bmp),Pal);
  Free(Pal);
  BMPsetTransparentColor(&(This->Bmp),0xFF,Transparent);
}
static void WADpreparePackI(pWADDEF This,Int32 Transparent)
{ pInt8 Pal;
  Int16 p;
  p= WADDfindEntryI(This,0,"PALETTE",EPK_PAL);
  if(p>=0)
  { Pal=WADreadEntryPartI(This, p, 0L, 0x300L);}
  else
  { Pal=WADQpalGetLmp(0x300);}
  if(Pal==NULL)
  { return;}
  BMPsetPalette(&(This->Bmp),Pal);
  Free(Pal);
  BMPsetTransparentColor(&(This->Bmp),0xFF,Transparent);
}
#endif  /*DLLFORQUAK*/


#if (DLLFORDUKE)
static void WADprepareKenI(pWADDEF This)
{ pInt8 Pal;
  Int16 p,col;
  p= WADDfindEntryI(This,0,"PALETTE",EDK_DAT);
  if(p<0)
  { return;}
  Pal=WADreadEntryPartI(This, p, 0L, 0x300L);
  /*set default grey palette if needed*/
  if(Pal==NULL)
  { Pal=Malloc(0x300);
	 if(Pal==NULL)
	 { return;}
	 for(p=0;p<0x300;p++)
	 { Pal[p]=(Int8)(p/3);}
  }
  else
  { /* Triple light level*/
	 for(p=0;p<0x300;p++)
	 { col = ((Int16)Pal[p])&0xFF;
		Pal[p] = (Int8)((col<<=1)+col);
	 }
  }
  BMPsetPalette(&(This->Bmp),Pal);
  Free(Pal);
}
#endif  /*DLLFORDUKE*/




/****************************************************\
*
*
*  WAD FILE
*
*
\****************************************************/

/*
** Init WAD object
*/
/*
** Load Wad Dir from WAD file, into memory
**  Main = reference of main WAD, if WAD is a PWAD
**       = -1 id WAD is an IWAD
**  Transparent = RGB(r,g,b) = BMP Transparent color
**  Lock = if >0, then wad is locked
*/
Int16 WADinitI(pWADDEF This, pInt8 File, Int16 Main, Int16 Lock, Int32 Transparent, Int32 ErrWnd)
{ Int32 res;
  /**/
  if(This==NULL)
  {return BAD_PARM;}
  if(This->Lst!=NULL)
  { return ERRfault(ERR_BUG);}
  This->ErrWnd = ErrWnd;
  /*
  ** prepare the WAD object
  */
  /*say if WAD is locked*/
  This->Locked=(Lock>=WAD_LOCK_YES)? TRUE:FALSE;
  /*sound*/
  This->Sound=TRUE;
  /* init0 directory*/
  This->Lst=NULL;This->LstNb=0;
  This->DirDirty=FALSE;
#if (DLLFORDOOM)
  /* init texture object*/
  WADTinitI(&(This->Txp));
  /* init level object */
  WADLinitI(&(This->Dml));
#endif /*DLLFORDOOM*/
#if (DLLFORQUAK)
  /* init Bsp*/
  BSPinitI(&(This->Qkl));
#endif /*DLLFORQUAK*/
  /*
  ** Preserve file name
  */
  Strncpy(This->File,File,sizeof(This->File)-1);
  /*
  ** Check if file exists. if not, create a fake
  */
  if(FILEexist(This->File, FALSE)==FALSE)
  {
	 res=WADHmakeFakeWadI(This->File);
	 if(res<0) return (Int16)res;
  }
  /*
  ** Preserve time stamp and size the WAD
  */
  This->OrigTime=FILEgetTime(This->File);
  This->OrigFileSz=FILEgetSize(This->File);
  /*
  ** Set initial write pointer
  */
  This->WritePtr=(This->OrigFileSz+0xFL)&(~0xFL);
  /*
  ** Read WAD header
  */
  This->WadType=WADHheaderReadI(&This->OrigDirPos,&This->OrigDirNb,This->File,This->OrigFileSz);
  if(This->WadType<0) { return This->WadType;}
  /*
  ** Read IWAD or PWAD directory
  */
  switch(This->WadType)
  {
#if (DLLFORDOOM)
	 case IS_IWAD:
	 case IS_PWAD:
	 res= WADDinitFromWadI(This, (Int16)(This->OrigDirNb&0x7FFF));
	 if(res<0)
	 { return (Int16)res;}
	 res=-1;
	 /*
	 ** Detect special void PWAD made from IWAD
	 */
	 if((This->WadType==IS_PWAD)&&(This->IwadSelf<0))
	 { /*a dangerous hack*/
		return 0;
	 }
	 /*
	 ** Init various data for PWAD or IWAD
	 */
	 if((This->WadType==IS_IWAD)||(This->IwadSelf<0))
	 { /*this is a main WAD*/
		This->IwadSelf=-1;
		/*prevent modification of the WAD*/
		This->Locked=TRUE;
		/*check WAD is correct*/
		res=WADprepareIwadI(This,Transparent);
	 }
	 else /*PWAD, not bad IWAD*/
	 {
		res=WADpreparePwadI(This,Transparent);
	 }
	 if(res<0)
	 {
		return (Int16)res;
	 }
	 /*identify entries*/
	 if(This->LstNb>0)
	 {
		/*identify entries, by name*/
		WADidentDirFastI(This);
		/*identify all entries, by reading file*/
		if((This->WadType==IS_PWAD) || ((Main>=0)&&(This->Game&(GAM_HEXN|GAM_HTIC|GAM_STRF))))
		{ WADidentDirSlowI(This);}
		/*unidentified entries become data LUMPs*/
		WADidentDirDefaultI(This);
	 }
	 break;
#endif /*DLLFORDOOM*/
  /*
  ** Read WAD2 directory
  */
#if (DLLFORQUAK)
	 case IS_WAD2:
		This->Game=GAM_QUAK;
		res=WADDinitFromWad2I(This,(Int16)(This->OrigDirNb&0x7FFF));
		if(res<0)
		{ return (Int16)res;}
		WADprepareWad2I(This,Transparent);
		/*prevent modification of the WAD*/
		This->Locked=TRUE;
	 break;
	 case IS_PACK:
		This->Game=GAM_PACK;
		res=WADDinitFromPackI(This,(Int16)(This->OrigDirNb&0x7FFF));
		if(res<0)
		{ return (Int16)res;}
      WADpreparePackI(This,Transparent);
		/*prevent modification of the WAD*/
		This->Locked=FALSE;
	 break;
#endif  /*DLLFORQUAK*/

#if (DLLFORDUKE)
  case IS_KENS:
		This->Game=GAM_DUKE;
		res=WADDinitFromKenI(This,(Int16)(This->OrigDirNb&0x7FFF));
		if(res<0)
		{ return (Int16)res;}
		WADprepareKenI(This);
		/*prevent modification of the WAD*/
		This->Locked=TRUE;
		break;
#endif  /*DLLFORDUKE*/
	 default:
		return ERR_BUG;
  }
  /*WAD directory is not dirty*/
  This->DirDirty=FALSE;
  return 1;
}
/*
** Free the directory
** if Save>0  save directory if modified
** else if Save<0 restore the file
*/
Int16 WADfreeI(pWADDEF This, Int16 Save)
{
  if(This==NULL){return BAD_PARM;}
  if(This->Lst==NULL) return ERR_BUG; /*non explicit error*/
  /*
  ** Save WAD directory, if modified
  */
  if(This->DirDirty==TRUE)
  { if(Save>=WAD_SAVE_OK)
	 { if(WADHdirSaveI(This,1)<0)
		{ /*
		  ** WAD directory save failed, warn and restore
		  */
		  ERRfaultWad(ERR_DIRTY);
		  Save=WAD_SAVE_RS;
		}
	 }
	 /*
	 ** Restore file if needed
	 ** This is DANGEROUS: if another tool modifies the WAD,
	 ** then it's not valid anymore
	 */
	 if(Save<=WAD_SAVE_RS)
	 { /*
		** Check that it's needed
		*/
		if(FILEgetSize(This->File) >= This->OrigFileSz)
		{ /* restore header*/
		  if(WADHheaderWriteI(This->File,This->OrigDirPos,This->OrigDirNb)>0)
		  {  /* restore size*/
			  FILEsetSize(This->File,This->OrigFileSz);
			  /* restore time stamp*/
			  FILEsetTime(This->File,This->OrigTime);
		  }
		}
	 }
  }
#if (DLLFORDOOM)
  /*
  ** Free patch, texture list
  */
  WADTfreeI(&(This->Txp));
  /*
  ** Free level stuff
  */
  WADLfreeI(&(This->Dml));
#endif /*DLLFORDOOM*/
#if (DLLFORQUAK)
  /* init Bsp*/
  BSPfreeI(&(This->Qkl));
#endif /*DLLFORQUAK*/
  /*
  ** Free directory list
  */
  WADDfreeI(This);
  return 0;
}
/*
** Preserve WAD file original size and directory
**  Restore = if <0 preserve, if >0 restore file
*/
Int16 WADpreserveI(pWADDEF This, Int16 Restore)
{
  if(Restore<=0)
  { /*
	 ** Prepare to restore
	 */
	 WADHdirSaveI(This,1);
	 This->OrigFileSz= FILEgetSize(This->File);
	 This->OrigTime  = FILEgetTime(This->File);
  }
  else
  { /*
	 ** Restore file
	 */
	 FILEsetSize(This->File,This->OrigFileSz);
	 FILEsetTime(This->File,This->OrigTime);
	 /*
	 ** read directory from scratch
	 *
	 WADfreeI(This);
	 WADinitI(This,This->File,0,0,This->Transp);
	 */
	 This->Locked=TRUE; /*security*/
  }
  return 1;
}


/****************************************************\
*
*
* Reading and writing entry data
*
*
\****************************************************/

/*
** Read a lump, to memory
*/
pInt8 WADreadEntryI(pWADDEF This,pInt32 pLmpSz,Int16 Entry)
{ pInt8 Lmp;
  Int32 Start,LmpSz;
  Int32 res;
  /*default to 0*/
  *pLmpSz=0;
  if(This==NULL){return NULL;}
  /*get entry start,size*/
  res=WADDgetEntryI(This,&Start,&LmpSz,Entry);
  if((res<0)||(LmpSz<=0))
  { return NULL; }
  /*get memory*/
  Lmp =Malloc( LmpSz);
  if(Lmp==NULL)
  { return NULL;}
  /*read data*/
  res= FILEreadData(Lmp,Start,LmpSz,This->File);
  if(res<0) { Free(Lmp);return NULL;}
  *pLmpSz=LmpSz;
  return Lmp;
}
/*
** Allocate memory and read part of an entry from PWAD
**  Entry = reference of the entry
**  Start = start of the entry part (from start of entry)
**  Size = size of the entry part (in the entry)
** Returns a pointer to the lump (NULL if no entry/no part)
*/
#if (DLLFORDUKE)
pInt8 WADreadEntryPartI(pWADDEF This, Int16 Entry, Int32 Start, Int32 Size)
{ pInt8 Lmp;
  Int32 start,LmpSz;
  Int32 res;
  /**/
  if(This==NULL){return NULL;}
  if((Start<0)||(Size<0))
  { ERRfault(ERR_BUG); return NULL;}
  /*get entry start,size*/
  res=WADDgetEntryI(This,&start,&LmpSz,Entry);
  if((res<0)||(LmpSz<=0))
  { return NULL; }
  /*
  ** check entry part is contained inside entry
  */
  LmpSz=LmpSz-Start;  /*maximum size of lump part*/
  if(Size>LmpSz)
  { ERRfaultWad(ERR_ESIZE); return NULL; }
  /**/
  start += Start;     /*actual start of lump part*/
  /*
  ** get memory
  */
  Lmp =Malloc(Size);
  if(Lmp==NULL)
  { return NULL;}
  /*
  ** Read data
  */
  res= FILEreadData(Lmp,start,Size,This->File);
  if(res<0) { Free(Lmp);return NULL;}
  return Lmp;
}
#endif /*DLLFORDUKE*/
/*
** Read an obligatory entry, that could be in WAD,
** but will be in IWAD otherwise.
** Returns NULL if not found.
** Free the returned pointer, if not NULL.
*/
pInt8 WADreadObligEntryI(pWADDEF This, pInt32 pSize, pInt8 Name)
{  pWADDEF That;
   pInt8 Lmp;
   Int32 LmpSz;
  Int16 entry;
  Lmp=NULL;
  /*try in PWAD*/
  entry=WADDfindEntryI(This,0,Name,-1);
  if(entry>=0)
  { Lmp=WADreadEntryI(This,&LmpSz,entry);}
  /*try in main WAD*/
  if(Lmp==NULL)
  { if(This->IwadSelf<0) return NULL;
    That=WADgetThat(This->IwadSelf);
    if(That==NULL) return NULL;
	 entry=WADDfindEntryI(That,0,Name,-1);
    if(entry<0)return NULL;
    Lmp=WADreadEntryI(That,&LmpSz,entry);
  }
  if(Lmp==NULL) return NULL;
  /*okay, found it*/
  *pSize=LmpSz;
  return Lmp;
}
/*
** put data in WAD, return Start
*/
static Int32 WADaddDataI(pWADDEF This, pInt8 Lmp, Int32 LmpSz, Int16 Entry)
{ Int32 Start,Size;
  Int16 Overwrite=FALSE; /*overwrite old entry*/
  if(This==NULL) return ERRfault(ERR_BUG);
  /*check locked*/
  if(This->Locked!=FALSE)
  { return ERRfaultWad(ERR_WADLOCK);}
  /*security: write pointer always after file start*/
  if(This->WritePtr < This->OrigFileSz)
  { This->WritePtr=This->OrigFileSz;}
  /*align on Int32 bytes*/
  This->WritePtr=(This->WritePtr+3L)&~3L;
  /*
  ** Check if entry already exists and can be overwritten
  ** Rule: overwite only is entry is beyond OrigFileSz (new entry)
  **       and if there is enough room there (Size>LmpSz)
  */
  if(Entry>=0)
  {
	 WADDgetEntryI(This, &Start, &Size, Entry);
	 if((Start > This->OrigFileSz)&&(Size >= LmpSz))
	 { Overwrite=TRUE; }
  }
  /*
  ** By Default: position on WritePtr
  ** (should be last used data)
  */
  if(Overwrite!=TRUE)
  { Start=This->WritePtr; }
  if((Lmp!=NULL)&&(LmpSz>0))
  {
	 Size=FILEwriteData(Lmp,Start,LmpSz,This->File);
	 if(Size<0){ return (Int16)Size;}
	 if(Overwrite!=TRUE)
	 { This->WritePtr +=LmpSz; }
  }
  return Start;
}
/*
** High level modification service
** replace or inserts an entry of the given Type and name
** if Lmp not NULL, then also upload the data.
** doesn't update the WAD directory
*/
Int16 WADmodifyEntryI(pWADDEF This, Int16 Entry, IDENT Id, pInt8 Name, pInt8 Lmp, Int32 LmpSz)
{ Int32 Start;
  if(This==NULL)
  { return BAD_PARM;}
  if(Lmp==NULL)
  { LmpSz=0; }
  if(LmpSz<0)
  { return BAD_PARM; }
  /**/
  if(WADmodifiableI(This)<=0) return ERR_SURE;
  /*
  ** We might be overwriting a temp WAD directory
  ** so write the original WAD header for security
  */
  if(This->DirDirty!=TRUE)
  {
	 if( WADHheaderWriteI(This->File,This->OrigDirPos,This->OrigDirNb)> 0)
	 {
		This->DirDirty=TRUE;
	 }
  }
  /*
  ** Find where to insert entry (and insert it if needed)
  ** (level parts shall be inserted on level header)
  */
  Entry=WADwhereEntryI(This, Entry, Id, Name);
  if((Entry<0)||(Entry>=This->LstNb))
  { return ERRfaultWad(ERR_MDF_NTRY); }
  /*
  ** Write data in WAD
  */
  Start=WADaddDataI(This, Lmp,LmpSz,Entry);
  if(Start<0) return (int)Start;
  /*
  ** Modify directory, returns position
  */
  return WADDsetEntryI(This,Entry,-1,NULL,Start,LmpSz);
}
/*
**
** Insert one entry
**
*/
Int16 WADinsertEntryI(pWADDEF This, Int16 Entry, IDENT Id, pInt8 Name, pInt8 Lmp, Int32 LmpSz)
{
  Int32 Start;
  if(Lmp==NULL)
  { Start = 0; }
  else
  { Start= WADaddDataI(This, Lmp, LmpSz,-1);
	 if(Start<0)
	 { return (Int16)Start; }
  }
  return WADDinsEntryI(This, Entry, Id, Name, Start, LmpSz);
}

/****************************************************\
*
*
*  Level handling
*
*
\****************************************************/

/*
** save a level WAD into New WAD File
*/
Int16 WADsaveLevelI(pWADDEF This, Int16 Entry,pInt8 File)
{ Int16 e,New;
  pInt8 Lmp;
  Int32 LmpSz;
  pWADDIR TLst;
  pWADDEF That;
  if((This==NULL)||(This->Lst==NULL))
  { return ERRfault(ERR_BUG);}
  if((Entry<0)||(Entry>This->LstNb))
  { return ERRfault(BAD_PARM);}
  /*check Entry is a valid level header*/
  if(WADDgetEntryI(This,NULL,NULL,Entry) !=ELVLHDR)
  {
	 return ERRfault(ERR_BADLEVEL);
  }
  /*create fake WAD*/
  if(WADHmakeFakeWadI(File)<0)
  { return ERR_BADWAD;}
  /*
  ** open new WAD
  */
  New = WADregister(This->IwadSelf);
  if(New<0) { return New;}
  That=WADgetThat(New);
  if(That==NULL) { return BAD_PARM;}
  e=WADinitI(That,File,This->IwadSelf,WAD_LOCK_NO,0x0000FFFF, This->ErrWnd);
  if(e<0)
  { WADunregister(New); return BAD_PARM; }

  /*
  ** write level data
  ** This->Lst is not changing
  */
  e=Entry;
  for(TLst=&(This->Lst[e]); e<This->LstNb; e++, TLst+=1)
  { /*
	 ** exit on next level header or on non-level data
	 */
	 if((e>Entry)&&(TLst->Id==ELVLHDR)) break;
	 if((TLst->Id&EMASK)!=ELEVEL) break;
	 /*
	 ** It's a level part, read entry
	 */
	 Lmp=WADreadEntryI(This,&LmpSz,e);
	 /*
	 ** Add in new WAD
	 */
	 WADinsertEntryI(That, -1, TLst->Id, TLst->Name, Lmp, LmpSz);
	 if(Lmp!=NULL) Free(Lmp);
  }
  /*
  ** save directory of new WAD
  */
  WADHdirSaveI(That,-1);
  /*
  ** exit, do not restore  new WAD
  */
  WADfreeI(That,WAD_SAVE_NO);
  WADunregister(New);
  return Entry;
}

/*
** load a level WAD into PWAD
** get first level header, and upload
**
** what = 0   load all level
** what = 1   load only nodes,blockmap,segs,vertex,ssect
** what = 2   load only reject
*/

Int16 WADloadLevelI(pWADDEF This, Int16 Entry, pInt8 Name, pInt8 File, Int16 What)
{
  Int16 res,e,New;
  pInt8 Lmp;
  Int32 LmpSz;
  pWADDEF That;
  pWADDIR TLstA;
  /**/
  if(This==NULL) return BAD_PARM;
  if(This->Lst==NULL)
  { return ERRfault(ERR_BUG);}
  if(Entry>This->LstNb)
  { Entry=-1;}
  if((Entry>=0)&&(This->Lst[Entry].Id!=ELVLHDR))
  { return ERRfault(ERR_BADLEVEL);
  }
  /*
  ** cancel current level
  */
  WADLfreeI(&(This->Dml));
  /*
  ** open new WAD, locked
  */
  New = WADregister(This->IwadSelf);
  if(New<0) { return New;}
  That=WADgetThat(New);
  if(That==NULL) { return BAD_PARM;}
  res=WADinitI(That,File,This->IwadSelf,WAD_LOCK_YES,0x0000FFFF,This->ErrWnd);
  if(res<0)
  { WADunregister(New); return BAD_PARM; }
  /*
  ** look for level header in new WAD
  */
  for(e=0,TLstA=That->Lst; e<That->LstNb; e++,TLstA+=1)
  { /*TLstA = That->Lst[e]*/
	 if(TLstA->Id==ELVLHDR) break;
  }
  if(e<That->LstNb)
  { /*
	 ** Read level header and put it in PWAD
	 ** Find where the level shall be inserted in PWAD.
	 */
	 Lmp = WADreadEntryI(That,&LmpSz,e);
	 Entry=WADmodifyEntryI(This, Entry, ELVLHDR, Name, Lmp, LmpSz);
	 if(Lmp!=NULL)
	 { Free(Lmp); }
	 if(Entry>=0)
	 { /*
		** load level entries, starting just after the header
		*/
		for(e+=1,TLstA+=1; e<That->LstNb; e++, TLstA+=1)
		{ /*TLstA = That->Lst[e]*/
		  /*
		  ** exit if another level header or on non-level data is found
		  */
		  if(TLstA->Id==ELVLHDR) break;
		  if((TLstA->Id&EMASK)!=ELEVEL) break;
		  /* Only nodes (EVERTX ESEGS ESSECT ENODE EREJCT EBLOCK)*/
		  if(What==WAD_LDNOD)
		  {
			 switch(TLstA->Id)
			 { case ETHING: case ELINEDEF: case ESIDEDEF: case ESECTOR:
				  continue;
			 }
		  }
		  /* Only REJECT */
		  else if(What==WAD_LDREJ)
		  {
			 if(TLstA->Id != EREJECT)  continue;
		  }
		  /*
		  ** Read entry from new WAD, modify PWAD entry.
		  */
		  Lmp=WADreadEntryI(That,&LmpSz,e);
		  res=WADmodifyEntryI(This,Entry,TLstA->Id,TLstA->Name,Lmp,LmpSz);
		  if(Lmp!=NULL)
		  { Free(Lmp); }
		  if(res<0) break;
		}
	 }
  }
  /*
  ** exit, do not restore  new WAD
  */
  WADfreeI(That,WAD_SAVE_NO);
  WADunregister(New);
  return Entry;
}

/****************************************************\
*
*
*  Sprite handling
*
*
\****************************************************/
/*
** remove a phase from sprite name
*/
static Int16 WADspriteView(Int8 View)
{
  switch(View)
  { case '0':
		return 0;
	 case '1': case '2': case '3': case '4':
	 case '5': case '6': case '7': case '8':
		return 1;
	 default:
		return -1;
  }
}
static Int16 WADspritePhase(pInt8 Name,Int8 Phase,Int8 View)
{
  Int16 view; /*code for view*/
  if(Phase=='\0') return 0;
  view=WADspriteView(View);
  if(view<0) return 0;
  /*remove inverted view*/
  if(Name[6]==Phase)
  {
	 if((View==Name[7])||(view== 1-WADspriteView(Name[7])))
	 { Name[6]='\0'; Name[7]='\0';
	 }
  }
  /*remove main view*/
  if(Name[4]==Phase)
  {
	 if((View==Name[5])||(view== 1-WADspriteView(Name[5])))
	 { Name[4]=Name[6]=Name[5]=Name[7]='\0';
	 }
  }
  return 1;
}
/*
** tell if a sprite is needed
** modify Name if part of the sprite is canceled
** return <0 if sprite not needed, >0 if needed
*/
Int16 WADspriteNeeded(pWADDEF This, Int16 Top, pInt8 Name)
{
  pWADDIR TLst;
  Int16 Entry;
  if(This==NULL) return ERR_BUG;
  if(Top>This->LstNb) Top=This->LstNb;
  /**/
  for(Entry=0,TLst=This->Lst; Entry<Top; Entry++,TLst+=1)
  {
	 /*eliminate non-sprite entry*/
	 if((TLst->Id & EMASK)!=ESPRITE) continue;
	 /*eliminate sprite with name different*/
	 if(Strncmp(Name,TLst->Name,4)<=0) continue;
	 /*check phase 1*/
	 WADspritePhase(Name,TLst->Name[4],TLst->Name[5]);
	 /*check phase 2*/
	 WADspritePhase(Name,TLst->Name[6],TLst->Name[7]);
	 /*check sprite is still valid*/
	 if(Name[4]=='\0') return 0;
  }
  return 1;
}

/****************************************************\
*
*
*  Exported functions
*
*
\****************************************************/
/*
** Init WAD object
** If lock>0, then wad is locked
*/
Int16 EXPORT WADinit(pInt8  File,Int16 Main,Int16 Lock,Int32 Transparent)
{ Int16 Self,res;
  Self=WADregister(Main);
  if(Self<0) { return Self;} /*Error*/
  res=WADinitI(WADgetThis(Self),File, Main, Lock, Transparent,NULL);
  if(res<0)
  { WADunregister(Self); return res;} /*Error*/
  if(res==0) return ERRfaultWad(ERR_BADIWAD);
  /*
  ** Init music
  */
  MUSinit();
  /**/
  return Self;
}
/*
** Free the directory
** if Save>0  save directory if modified
** else if Save<0 restore the file
*/
Int16 EXPORT WADfree(Int16 Self, Int16 Save)
{
  WADfreeI(WADgetThis(Self),Save);
	/*
  ** Unregister WAD object
  */
  WADunregister(Self);
  /*
  ** ShutDown music
  */
  MUSstop();
  /*
  ** a new value for Self
  */
  return -1;
}
/*
** Export find entry
*/
Int16 EXPORT WADfindEntry(Int16 Self,Int16 Here,pInt8 Name)
{
  return WADDfindEntryI(WADgetThis(Self),Here,Name,-1);
}
/*
** Export get entry
** returns Id
*/
Int16 EXPORT WADgetEntry(Int16 Self, pInt32 pStart, pInt32 pSize, Int16 Entry)
{  IDENT Id;
	Int32 Start,Size;
  Id= WADDgetEntryI(WADgetThis(Self),&Start,&Size,Entry);
  *pStart=Start;
  *pSize=Size;
  if(Id==EWHAT) Id=-1;
  return Id;
}
/* Set dir Entry, with the provided Name, Start, Size
** with Entry<0 to add an entry
**	 Entry>0 to modify the entry
**	 Name="" to delete the entry
**the WAD directory is not saved on file, it is kept dirty
*/

/****** SHOULD INSERT ACCORDING TO TYPE *****/
Int16 EXPORT WADsetEntry(Int16 Self, Int16 Here, pInt8 Name, Int32 Start, Int32 Size, Int16 Id, Int16 Modif)
{  pWADDEF This;
  This=WADgetThis(Self);
  if(This==NULL)
  { return BAD_PARM;}
  if((This->Lst==NULL)||(This->LstNb<0))
  { return ERRfault(ERR_BUG);}
  if(This->Locked!=FALSE)
  { return ERRfaultWad(ERR_WADLOCK);}
  switch(Modif)
  {
	 /*delete entry*/
	 case MODIF_DEL:
		return WADDdelEntryI(This, Here);
	 /*add entry*/
	 case MODIF_ADD:
		return WADDinsEntryI(This, -1, Id, Name, Start, Size);
	 /*insert entry*/
	 case MODIF_INS:
		return WADDinsEntryI(This, Here, Id, Name, Start, Size);
	 /*set entry*/
	 case MODIF_SET:
		 return WADDsetEntryI(This, Here, Id, Name, Start, Size);
  }
  return ERRfault(BAD_PARM);
}
/*
** Get informations
*/
#define VbTRUE -1
#define VbFALSE 0
Int16 EXPORT WADgetInfos(Int16 Self, pInt16 pGame, pInt16 pIwadSelf, pInt16 pDirDirty, pInt8 File)
{  pWADDEF This;
  This=WADgetThis(Self);
  if(This==NULL)
  { return BAD_PARM;}
  if((This->Lst==NULL)||(This->LstNb<0))
  { return ERRfault(ERR_BUG);}
  if((This->Game & GAM_MASK )==GAM_NONE)
  { return -1;}
  if(pGame!=NULL)
  { *pGame    = (This->Game&0x7FFF);}
  if(pIwadSelf!=NULL)
  { *pIwadSelf = (This->IwadSelf);}
  if(pDirDirty!=NULL)
  { *pDirDirty= (This->DirDirty)? VbTRUE:VbFALSE;}
  if(File!=NULL)
  { Strncpy(File,This->File,sizeof(This->File)-1);
  }
  return Strlen(This->File,sizeof(This->File));
}
/*
** Save Directory of WAD, so that WAD becomes usable
** by other WAD tools, like level editors
*/
Int16 EXPORT WADsaveDir(Int16 Self)
{  pWADDEF This;
  This=WADgetThis(Self);
  if((This==NULL)||(This->Lst==NULL))
  { return ERRfault(ERR_BUG);}
  if(This->DirDirty!=TRUE)
  { return 0; }
  if(This->Locked!=FALSE)
  { return ERRfaultWad(ERR_WADLOCK);}
  return WADHdirSaveI(This,1);
}







/****************************************************\
*
*
*  Cleanup WAD (locks the system)
*
*
\****************************************************/

/*
** Cleanup PWAD into New WAD File
*/
Int16 EXPORT WADcleanup(Int16 Self, pInt8 File)
{ Int16 e,New;
  pInt8 Lmp;
  Int32 LmpSz,Start;
  pWADDIR TLst;
  pWADDEF This;
  pWADDEF That;
  This=WADgetThis(Self);
  if((This==NULL)||(This->Lst==NULL))
  { return ERRfault(ERR_BUG);}
  /*create fake WAD*/
  if(WADHmakeFakeWadI(File)<0)
  { return ERR_BADWAD;}
  /*open new WAD*/
  New = WADregister(This->IwadSelf);
  if(New<0) { return New;}
  That=WADgetThat(New);
  if(That==NULL) { return BAD_PARM;}
  e=WADinitI(That,File,This->IwadSelf,WAD_LOCK_NO,0x0000FFFF,This->ErrWnd);
  if(e<0)
  { WADunregister(New); return BAD_PARM; }
  /*copy all entries from PWAD into new WAD*/
  TLst=This->Lst;
  for(e=0;e<This->LstNb;e++,TLst+=1)
  { /*TLst=This->Lst[e]*/
	 Start=-1;
	 /*
	 ** Put entry into new WAD
	 */
	 if(TLst->Size>0)
	 { Lmp=WADreadEntryI(This,&LmpSz,e);
		if(Lmp!=NULL)
		{ /*copy into new WAD*/
		  Start=WADaddDataI(That, Lmp,LmpSz, -1);
		  Free(Lmp);
		}
	 }
	 if(Start<0) /*could not write data. use fake entry*/
	 { Start=0; LmpSz=0;}
	 /*
	 ** Insert entry at the end of new WAD directory
	 */
	 WADDinsEntryI(That, -1, TLst->Id, TLst->Name, Start, LmpSz);
  }
  /*save directory*/
  WADHdirSaveI(That,-1);
  /*exit, do not restore*/
  WADfreeI(That,WAD_SAVE_NO);
  WADunregister(New);
  return 1;
}


