/*----------------------------------------------------------------------------*
 | This file is part of DEU (Doom Editing Utilities), created by the DEU team:|
 | Raphael Quinet, Brendon Wyber, Ted Vessenes and others.  See README.1ST or |
 | the "about" dialog box for full credits.                                   |
 |                                                                            |
 | DEU is an open project: if you think that you can contribute, please join  |
 | the DEU team.  You will be credited for any code (or ideas) included in    |
 | the next version of the program.                                           |
 |                                                                            |
 | If you want to make any modifications and re-distribute them on your own,  |
 | you must follow the conditions of the DEU license.  Read the file LICENSE  |
 | in this directory or README.1ST in the top directory.  If do not have a    |
 | copy of these files, you can request them from any member of the DEU team, |
 | or by mail: Raphael Quinet, Rue des Martyrs 9, B-4550 Nandrin (Belgium).   |
 |                                                                            |
 | This program comes with absolutely no warranty.  Use it at your own risks! |
 *----------------------------------------------------------------------------*

 I_DIALOG.H

*/

#ifndef _I_DIALOG_H_
#define _I_DIALOG_H_

/* DIALOG BOX SPECIAL COOKIE DEFINES */
/* NONE */
#define DBSC_NONE       -1

/* PROGRAM CONTROL */
#define DBSC_BUTTON      0
/* BUTTON TYPE DEFINES */
#define BUTTON_CANCEL    0
#define BUTTON_OK        1
#define BUTTON_DIALOG    2
#define BUTTON_PP_DIALOG 3
#define BUTTON_UPDATE    4
#define BUTTON_FUNCTION  5
/* BUTTON UPDATE TYPE DEFINES */
#define DB_NONE   0
#define DB_UPDATE 1
#define DB_CUR    2
#define DB_ALL    3

/* INPUT */
#define DBSC_CHECKBOX    1 
#define DBSC_L_CHECKBOX  2
#define DBSC_INPUTBOX    3
#define DBSC_SCROLLBOX   4
/* DBSC SCROLLBOX LIST TYPE DEFINES */
#define LT_NONE      0
#define LT_STR       1
#define LT_THINGS    2
#define LT_L_EFFECTS 3
#define LT_S_EFFECTS 4

/* OUTPUT */
#define DBSC_TEXT        5
#define DBSC_BITMAP      6
/* DBSC BITMAP PICTURE TYPE DEFINES */
#define BMP_THING 0
#define BMP_WALL  1
#define BMP_FLAT  2
#define DBSC_BOX        10 
/* DBSC BOX TYPE DEFINES */
#define BOX_OUTLINE 0
#define BOX_RAISED  1
#define BOX_SUNKEN  2
#define BOX_HOLLOW  3
#define BOX_GRAY    4
#define BOX_GREY    4
/* You would not believe how often I accidently type this */

/* MACRO */
#define DBSC_CANCEL      7
#define DBSC_OK          8
#define DBSC_UPDATE      9


/* KEY DEFINES */
#define KEY_TAB      0x0009
#define KEY_ENTER    0x000D
#define KEY_ESC      0x001B
#define KEY_SHIFTTAB 0x0F00
#define KEY_UP       0x4800
#define KEY_DOWN     0x5000
#define KEY_P_UP     0x4900
#define KEY_P_DOWN   0x5100

/* GRAPHICS OFFSETS */
#define BUTTON_W     ((Int16) 89)
#define BUTTON_H     ((Int16) 18)
#define CHECKBOX_W   ((Int16) 11)
#define CHECKBOX_H   ((Int16) 11)
#define INPUTBOX_W   ((Int16) 61)
#define INPUTBOX_H   ((Int16) 13)
#define SCROLL_H_TAB ((Int16)  8)
#define SCROLL_V_TAB ((Int16)  4)
#define SCROLLBAR_W  ((Int16) 16)
#define ARROW_H      ((Int16) 16)
#define ARROW_TAB    ((Int16)  4)

/* MISCELLANIOUS INFORMATION */
#define UP          1
#define DOWN       -1
#define DOWN_BLACK -2

#define BIT         0
#define SVAL        1
#define TVAL        2

#define FOCUS       1

#define ACTIVE      1

#define WAIT        0
#define REPEAT     -1
#define RESET      -2

#define FULL        0
#define PARTIAL     1

#define INIT        1
#define RUN         0
#define UNDO       -1

/* MACROS */
#define BOUND(x, y, z)    (MAX((x), MIN((y), (z))))
#define top_arg(AP, TYPE) (*((TYPE *) (void *) (char *) (AP)))


/* TYPEDEFS */
typedef struct checkbox_info
{
  Bool    bits;
  Int16  *var, value, backup, last;
  UInt32  chbit;
} checkbox_info;

typedef struct l_checkbox_info
{
  Bool    bits;
  Int16  *var, backup, value, last;
  UInt32  chbit;
  char   *string; /* Note no wraplen; the string should be short anyway */
} l_checkbox_info; /* Labelled input box info */

typedef struct inputbox_info
{
  Int16  *var, backup, min, max; /* Variable and min/max values */
  UInt32  chbit;
} inputbox_info;

typedef struct scrollbox_info
{
  Int16   *var, backup;
  char    *str, bkstr[9];
  Int16    ref; /* Chosen item from list, its array ref */
  Int16    height, wid, top; /* Size of window, ref of top item in window */
  Int16    slider_l, slider_o; /* Slider length, slider offset */
  Int16    barlen; /* Length of scroll bar in pixels */
  UInt32   chbit;
  char   **namest; /* List of names that can be chosen */
  Int16   *numst, listlen; /* Associated numbers and the list sizes */
  Int16    listtype; /* The type of list; see #defines */
  Bool     bar; /* Is there a scroll bar? */
} scrollbox_info;

typedef struct text_info
{
  char  *string;
  Int16  wraplen, lines, color; /* len when string wraps, num of lines */
} text_info;

typedef struct bitmap_info
{
  Int16 *var, last; /* *var is never set; just there to monitor changes */
  char  *str, laststr[9];
  Int16  pictype;
} bitmap_info;

typedef struct box_info
{
  Int16 type, dx, dy;
} box_info;

typedef struct DB
{
  Int16        x, y, dx, dy;
  struct DB   *next;          /* Next Dialog Box in the list */
  struct DBSC *start, *focus; /* Start of SC list, Focused scrollbox */
  char        *bitmap;        /* bitmap backup for redraw */
} DB;

typedef struct DBSC /* Dialog Box Special Cookie */
{
  Int16        x, y, type; /* X and Y offsets, type of Special Cookie */
  char         key;      /* Associated key of the list item; '\0' for no key */
  Bool         active;     /* Is the DBSC active? */
  struct DBSC *next;       /* Link to next item */
  
  union                    /* Special info for different Special Cookies */
    {
      struct button_info
        {
          Int16      type, color, dx, dy;
          char      *name;
          int        repaint;
          
          struct DB *start;
          
          void      *var;
          
          void       (*hookfunc)(va_list *, struct DBSC *, Bool);
          va_list argv[1];
        } button;
      struct checkbox_info   checkbox;
      struct l_checkbox_info l_checkbox;
      struct inputbox_info   inputbox;
      struct scrollbox_info  scrollbox;
      struct text_info       text;
      struct bitmap_info     bitmap;
      struct box_info        box;
    } i; /* info */
} DBSC;

typedef struct button_info button_info;


/* GLOBAL VARIABLES */
extern DB *dbtop; /* Top of the dialog box list */
extern DB *dbcur; /* Current dialog box */
extern UInt32 *changes; /* 32 bits of variable change flags */


/* PROTOTYPES */
UInt32 DrawDialogBox(Int16, Int16, Int16, Int16, UInt32 *, Int16, ...);
UInt32 ProcessDialogBox(DB *, UInt32 *);
DB    *LoadDialogBox(Int16, Int16, Int16, Int16, UInt32 *, Int16, ...);
void   ParseDialogBoxArgs(Int16, Int16, Int16, Int16, UInt32 *, Int16, va_list *,
                          DB **);
Bool   OpenDialogBox(Int16 x, Int16 y, Int16 dx, Int16 dy);
void   AppendDbscList(DB *, UInt32 *, Int16, ...);
Int16  GetFocusRank(DBSC *);
void   SetChangeBits(UInt32);
Bool   IsButton(DBSC *);
void   GetScDim(DBSC *, Int16 *, Int16 *);
void  *GetVar(DBSC *);
Bool   DbscActive(DBSC *);

void DisplayDebugMessage(Int16 x, Int16 y, char *str, ...);
/*! this function should be moved in another file (I_*.C or M_*.C) */
void ViewPalette(int playpalnum);

#endif /* _I_DIALOG_H_ */
/* end of file */
