#global RPT
#set RPT(traceproc) 1
#============================================================================
# TclVSrpt-1.0 -- Tcl Very Simple Report Writer
# 
# Copyright (c) 1995, Steven B. Wahl
# See the file "license.terms" for information on usage and
# redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# History:
#   08/12/95 - sbw - original code 
#
# ---- Public Procedures -----
# rptAddLine        - Add complete line of text
# rptAddText        - Add Text to current line
# rptDefPage        - Set Title, SubTitle, and Footers
# rptDefPaper       - Set Paper size, orientation, margins
# rptDefStyle       - Set font, font style, font size, leading, justification, 
#                     indentation, tabs, and line-wrapping to a symbolic name
# rptDone           - Report complete, send to file / printer
# rptInitialize     - (Re)Initialize Report Writer, set postscript/text mode
# rptNewLine        - Move to new line of text
# rptNewPage        - (Un)Conditionally eject current page,start new one
# rptSetStyle       - Retrieve style definition and put into effect
# 
# ---- Internal Utility Procedures ----
# rptu2point         - convert size/location to points (1/72 inch)
# rptuOptions        - Return arguments as option sublists
# rptuPSflush        - flush postscript instruction buffer for this line
# rptuPSfontEval     - Build proper Postscript font name
# rptuPSpagedecorate - initialize postscript page, put headers and footers 
# rptuPSprolog       - Create Postscript Prolog
# rptuPSsetup        - Create Postscript document structuring comments
# rputPStext         - Postscript, add text to current line
# rptuPStrailer      - Create Postscript end of document structuring comments
# rptuSetDefaults    - Initialize RPT default values
# rptuTabSort        - Sort function for tab list
# rptuTXTfooterOut   - Position to and print page footer for ASCII report
# rptuTXTheaderOut   - Print page header for ASCII report
# rptuTXTsetup       - Set up for ASCII report prior to first text
# rptuTXTtext        - Add text to current line of ASCII report
#
# PUBLIC PROCEDURES
#-----------------------------
# rptAddLine - Add complete line of text
proc rptAddLine {args} {
    global RPT
    # -- provide missing prerequisite calls
    if {![info exists RPT(initializedf)]} {rptInitialize -type ASCII}
    if {!$RPT(initializedf)} {rptInitialize -type ASCII}
    if {!$RPT(paperdeff)} {
        rptDefPaper -pagesize letter -margins 0i -orientation portrait
    }
    if {!$RPT(pagedeff)} {rptDefPage}
    # -- handle list or string arguments
    if {[llength $args] > 1} {
        set args [lindex [rptuOptions $args] 0]
    } else {
        set args [lindex $args 0]
    }
    # -- no newlines allowed, get rid of any encountered
    regsub -all "\n" $args {} args
    # -- if not currently at the start of a new line, flush current line
    if {$RPT(postscriptf)} {
        if {$RPT(textline) != ""} {
            lappend RPT(tlist) [list 2 "$RPT(textline)"]
            lappend RPT(tlist) [list 1 $RPT(strwid)]
            incr RPT(tlistlen) 2
            set RPT(textline) ""
            set RPT(textf) 1
        }
        if {$RPT(textf)} {
            rptNewLine 0
        }
    } else {
        if {!$RPT(atlinestartf)} {
            rptNewLine 0
        }
    }
    # -- handle first line indentation
    if {$RPT(justify) == "L" || $RPT(justify) == "J"} {
        if {$RPT(lineindent) != 0} {
            if {!$RPT(postscriptf)} {
                incr RPT(cpos) [expr int($RPT(lineindent)/$RPT(ppc))]
                set RPT(lasttabposused) $RPT(cpos)
            } else {
                set RPT(cpos) $RPT(lineindent)
                set RPT(strwid) $RPT(lineindent)
                lappend RPT(tlist) [list 1 $RPT(lineindent)]
                set RPT(lasttabposused) $RPT(tlistlen)
                incr RPT(tlistlen)
            }
        }
    }
    # -- handle line prefix characters (don't prefix empty lines)
    if {$RPT(lineprefix) != "" && [string length $args] > 0} {
        rptAddText "$RPT(lineprefix)$args"
    } else {
        # -- add the text of the provided line
        rptAddText "$args"
    }
}

#-----------------------------
# rptAddText - Add Text to current line
proc rptAddText {args} {
    global RPT
    flush $RPT(fd)
    # -- provide missing prerequisite calls
    if {![info exists RPT(initializedf)]} {rptInitialize -type ASCII}
    if {!$RPT(initializedf)} {rptInitialize -type ASCII}
    if {!$RPT(paperdeff)} {
        rptDefPaper -pagesize letter -margins 0i -orientation portrait
    }
    if {!$RPT(pagedeff)} {rptDefPage}
    # -- handle list or string arguments
    if {[llength $args] > 1} {
        set args [lindex [rptuOptions $args] 0]
    } else {
        set args [lindex $args 0]
    }
    # -- empty string will produce an empty line
    if {[string length $args] == 0} {
        if {$RPT(postscriptf)} {
            rptNewLine 1
        } else {
            if {$RPT(atlinestartf)} {
                rptNewLine 1
            }
        }
        return
    }
    # -- if multiple lines provided, send all but last through
    # -- rptAddLine, then process last line locally
    set lines [split $args \n]
    set lastline [expr [llength $lines] - 1]
    set line [lindex $lines $lastline]
    if {$lastline > 0} {
        set startline 0
        if {!$RPT(atlinestartf)} {
            set startline 1
            # -- current line non-empty, recurse to add first line to it
            rptAddText "[lindex $lines 0]"
        }
        # -- process all but last line via rptAddLine
        for {set i $startline} {$i < $lastline} {incr i} {
            rptAddLine "[lindex $lines $i]"
        }
    }
    # -- initialize for tab processing
    if {$RPT(postscriptf)} {
        set tabX -1
    } else {
        set tabX 0
    }
    set tabType $RPT(justify)
    set lineJust [lindex $RPT(style-$RPT(stylecurrent)) 4]
    # -- cycle through tabs
    set text ""
    set tabf 0
    foreach text [split "$line" \t] {
        # -- don't increment curtab for the first text packet through
        if {$tabf} {
            incr RPT(curtab)
            if {$RPT(curtab) >= $RPT(numtabs)} {
                set RPT(curtab) 999
            }
        }
        set tabf 1
        # -- if the packet is that of a tab, count coup and continue
        if {"$text" == ""} {continue}
        # -- fetch the tab position and tab type
        if {$RPT(curtab) != -1 && $RPT(curtab) != 999} {
            if {$RPT(tabstyle) == "A"} {
                # -- get absolute tab
                set tabX [lindex $RPT(tabs) $RPT(curtab)]
                set tabType [lindex $RPT(tabtypes) $RPT(curtab)]
            } else {
                # -- get nearest tab
                if {$RPT(postscriptf)} {
                    set cx $RPT(strwid)
                } else {
                    set cx [expr int(($RPT(cpos)-$RPT(lmargin))/(1.0*$RPT(ppc)))]
                }
                set i -1
                set found 0
                foreach t $RPT(tabs) {
                    incr i
                    if {$t > $cx} {
                        set tabX $t
                        set tabType [lindex $RPT(tabtypes) $i]
                        set found 1
                        break
                    }
                }
                if {!$found} {
                    if {!$RPT(postscriptf)} {
                        set tabX -1
                    } else {
                        set tabX 0
                    }
                    set RPT(curtab) 999
                    set tabType $lineJust
                }
            } 
        }
        if {$RPT(curtab) == -1 || $RPT(curtab) == 999} {
            set RPT(justify) $lineJust
        } else {
            set RPT(justify) $tabType
        }
        if {!$RPT(postscriptf)} {
            if {$RPT(cpos) < \
                    [expr int(($tabX+$RPT(lmargin))/$RPT(ppc))]} {
                set RPT(cpos) [expr int(($tabX+$RPT(lmargin))/$RPT(ppc))]
                set RPT(lasttabposused) $RPT(cpos)
            }
            rptuTXTtext "$text"
            set RPT(atlinestartf) 0
        } else {
            if {$RPT(strwid) > $tabX} {
                rptuPStext -1 $lineJust "$text"
            } else {
                rptuPStext $tabX $tabType "$text"
            }
        }
    }
}

#-----------------------------
# rptDefPage - Set Page Decoration (Title, SubTitle, Footers, etc.)
proc rptDefPage {args} {
    global RPT
    # -- provide missing prerequisite calls
    if {![info exists RPT(initializedf)]} {rptInitialize -type ASCII}
    if {!$RPT(initializedf)} {rptInitialize -type ASCII}
    if {!$RPT(paperdeff)} {
        rptDefPaper -pagesize letter -margins 0i -orientation portrait
    }
    # -- if called before, forget it, otherwise process options
    if {$RPT(pagedeff)} {return}
    set titlefont "Courier"
    set titlesize "18"
    set titleboldf ""
    set titleitalicf ""
    set titleunderlinef ""
    set titlealign "C"
    set titlegap 2

    set subtitlefont "Courier"
    set subtitlesize "12"
    set subtitleboldf ""
    set subtitleitalicf ""
    set subtitleunderlinef ""
    set subtitlealign "L"
    set subtitlegap 2

    set footerfont "Courier"
    set footersize "12"
    set footerboldf ""
    set footeritalicf ""
    set footerunderlinef ""
    set footeralign "L"
    set footergap 2

    foreach option [rptuOptions $args] {
        switch -glob -- [string tolower [lindex $option 0]] {
            -title {set RPT(titletext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -titlefont {set titlefont [lindex $option 1]}
            -titlebold {set titleboldf "-bold"}
            -titleital* {set titleitalicf "-italic"}
            -titleund* {set titleunderlinef "-underline"}
            -titlesize {set titlesize [lindex $option 1]}
            -titlegap {set titlegap [lindex $option 1]}
            -subtitle {set RPT(subtitlectext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -subtitleleft {set RPT(subtitleltext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -subtitlecenter {set RPT(subtitlectext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -subtitleright {set RPT(subtitlertext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -subtitlefont {set subtitlefont [lindex $option 1]}
            -subtitlebold {set subtitleboldf "-bold"}
            -subtitleital* {set subtitleitalicf "-italic"}
            -subtitleund*  {set subtitleunderlinef "-underline"}
            -subtitlesize {set subtitlesize \
                    [lindex $option 1]}
            -subtitlegap {set subtitlegap \
                    [lindex $option 1]}
            -titlebar {set RPT(titlebarf) 1}
            -titlebarindent* {
                set RPT(titlebarf) 1
                set RPT(titlebarindent) [rptu2point [lindex $option 1]]
            }
            -titlebarthick* {
                set RPT(titlebarf) 1
                set RPT(titlebarthick) [rptu2point [lindex $option 1]]
            }
            -titlebargap* {
                set RPT(titlebarf) 1
                set RPT(titlebargap) [rptu2point [lindex $option 1]]
            }
            -footer {set RPT(footerctext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -footerleft {set RPT(footerltext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -footercenter {set RPT(footerctext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -footerright {set RPT(footerrtext) \
                    [lindex [rptuOptions [lrange $option 1 end]] 0]}
            -footerfont {set footerfont [lindex $option 1]}
            -footerbold {set footerboldf "-bold"}
            -footerital* {set footeritalicf "-italic"}
            -footerund* {set footerunderlinef "-underline"}
            -footersize {set footersize \
                    [lindex $option 1]}
            -footergap {set footergap \
                    [lindex $option 1]}
            -footerbar {set RPT(footerbarf) 1}
            -footerbarindent* {
                set RPT(footerbarf) 1
                set RPT(footerbarindent) [rptu2point [lindex $option 1]]
            }
            -footerbarthick* {
                set RPT(footerbarf) 1
                set RPT(footerbarthick) [rptu2point [lindex $option 1]]
            }
            -footerbargap* {
                set RPT(footerbarf) 1
                set RPT(footerbargap) [rptu2point [lindex $option 1]]
            }
            -pagenoca* {set RPT(pagenocallback) [lindex $option 1]}
            -pagenopr* {
                set RPT(pagenopre) "[lindex $option 1]"
                set RPT(pagenof) 1
            }
            -pagenosta* {
                set RPT(pagenostart) [lindex $option 1]
                set RPT(pagenof) 1
            }
            -pagenosty* {
                switch -glob -- [string tolower [lindex $option 1]] {
                    n* {set RPT(pagenostyle) "N"}
                    c* {set RPT(pagenostyle) "C"}
                    r* {set RPT(pagenostyle) "R"}
                    default {set RPT(pagenostyle) "N"}
                }
                set RPT(pagenof) 1
            }
            -pagenosu* {
                set RPT(pagenosuf) "[lindex $option 1]"
                set RPT(pagenof) 1
            }
            -pagenolo* {
                set tbflag "B"
                set lcrflag "C"
                set altflag 0
                foreach word $option {
                    switch -glob -- $word {
                        -pag* {
                            #noop, been there, done that
                        }
                        b* {set tbflag "B"}
                        t* {set tbflag "T"}
                        l* {set lcrflag "L"}
                        c* {set lcrflag "C"}
                        r* {set lcrflag "R"}
                        a* {set altflag 1}
                        default {}
                    }
                }
                set RPT(pagenoloc) "$tbflag $altflag $lcrflag"
                set RPT(pagenof) 1
            }
        }
    }
    set RPT(pagedeff) 1
    # -- set title, subtitle, and footer text styles
    rptDefStyle -name pageheader -font $titlefont -size $titlesize \
        $titleboldf $titleitalicf $titleunderlinef -gap $titlegap \
        -align $titlealign
    rptDefStyle -name pagesubheader -font $subtitlefont -size $subtitlesize \
        $subtitleboldf $subtitleitalicf $subtitleunderlinef \
        -gap $subtitlegap -align $subtitlealign
    rptDefStyle -name pagefooter -font $footerfont -size $footersize \
        $footerboldf $footeritalicf $footerunderlinef \
        -gap $footergap -align $footeralign
    if {$RPT(postscriptf)} {
        # -- generate appropriate Postscript startup stuff
        rptuPSsetup
        rptuPSprolog
    } else {
        # -- generate ASCII report startup stuff
        rptuTXTsetup
        rptuTXTheaderOut
    }
}

#-----------------------------
# rptDefPaper   - Set paper size, orientation, margins
proc rptDefPaper {args} {
    global RPT
    # -- provide missing prerequisite calls
    if {![info exists RPT(initializedf)]} {
        rptInitialize -type ASCII
    }
    if {!$RPT(initializedf)} {
        rptInitialize -type ASCII
    }
    set RPT(pw) 0
    set RPT(ph) 0
    # -- if called before, forget it, otherwise process options
    if {$RPT(paperdeff)} {return}
    foreach option [rptuOptions $args] {
        switch -glob -- [string tolower [lindex $option 0]] {
            -pagesize {
                switch -glob -- [string tolower [lindex $option 1]] {
                    let* {set RPT(pw) 612; set RPT(ph) 792}
                    leg* {set RPT(pw) 612; set RPT(ph) 1008}
                    tab* {set RPT(pw) 792; set RPT(ph) 1224}
                    led* {set RPT(pw) 1224; set RPT(ph) 792}
                    sta* {set RPT(pw) 396; set RPT(ph) 612}
                    exe* {set RPT(pw) 540; set RPT(ph) 720}
                    a3 {set RPT(pw) 842; set RPT(ph) 1190}
                    a4 {set RPT(pw) 595; set RPT(ph) 842}
                    a5 {set RPT(pw) 420; set RPT(ph) 595}
                    b4 {set RPT(pw) 729; set RPT(ph) 1032}
                    b5 {set RPT(pw) 516; set RPT(ph) 729}
                    fol* {set RPT(pw) 612; set RPT(ph) 936}
                    qua* {set RPT(pw) 610; set RPT(ph) 780}
                    10x14 {set RPT(pw) 720; set RPT(ph) 1008}
                    default {set RPT(pw) 612; set RPT(ph) 792}
                }
            }
            -customw* {set RPT(pw) [rptu2point [lindex $option 1]]}
            -customh* {set RPT(ph) [rptu2point [lindex $option 1]]}
            -orient* {
                switch -glob -- [string tolower [lindex $option 1]] {
                    por* {set RPT(landscapef) 0}
                    lan* {set RPT(landscapef) 1}
                    default {set RPT(landscapef) 0}
                }
            }
            -margins {
                set m [rptu2point [lindex $option 1]]
                set RPT(lmargin) $m
                set RPT(rmargin) $m
                set RPT(tmargin) $m
                set RPT(bmargin) $m
            }
            -lef* {set RPT(lmargin) [rptu2point [lindex $option 1]]}
            -rig* {set RPT(rmargin) [rptu2point [lindex $option 1]]}
            -top* {set RPT(tmargin) [rptu2point [lindex $option 1]]}
            -bot* {set RPT(bmargin) [rptu2point [lindex $option 1]]}
            -leftd* {set RPT(lmdead) [rptu2point [lindex $option 1]]}
            -rightd* {set RPT(rmdead) [rptu2point [lindex $option 1]]}
            -topd* {set RPT(tmdead) [rptu2point [lindex $option 1]]}
            -botd* {set RPT(bmdead) [rptu2point [lindex $option 1]]}
            -pointsperc* {set RPT(ppc) [rptu2point [lindex $option 1]]}
            -pointsperl* {set PRT(ppl) [rptu2point [lindex $option 1]]}
        }
    }
    # -- check for custom page size screwup and default to US Letter
    if {$RPT(pw) == 0 || $RPT(ph) == 0} {
        set RPT(pw) 612
        set RPT(ph) 792
    }
    if {!$RPT(postscriptf)} {
        # -- define some constants to describe the text printer in points
        # -- these are set for 60 lines of 80 characters on 8.5"x11" paper
        # -- YOU MAY HAVE TO ADJUST THESE FOR YOUR PRINTER
        set RPT(lmdead) [rptu2point .25i];  # .25" hard left margin
        set RPT(rmdead) [rptu2point .25i];  # .25" hard right margin
        set RPT(tmdead) [rptu2point .5i];   # .5" hard top margin
        set RPT(bmdead) [rptu2point .5i];   # .5" hard bottom margin
        set RPT(ppc) [expr [rptu2point 1i] / 10.0]; # 7.2 points per char
        set RPT(ppl) [expr [rptu2point 1i] / 6.0];  # 12 points per line
    } else {
        # -- if in landscape mode, flip page dimensions
        if {$RPT(landscapef)} {
            set x $RPT(ph)
            set RPT(ph) $RPT(pw)
            set RPT(pw) $x
        }
    }
    # -- take non-printing area into account for page size and margins
    set RPT(pw) [expr $RPT(pw) - ($RPT(lmdead)+$RPT(rmdead))]
    set RPT(ph) [expr $RPT(ph) - ($RPT(tmdead)+$RPT(bmdead))]
    if {$RPT(lmargin) >= $RPT(lmdead)} {
        set RPT(lmargin) [expr $RPT(lmargin) - $RPT(lmdead)]
    } else {
        set RPT(lmargin) $RPT(lmdead)
    }
    if {$RPT(rmargin) >= $RPT(rmdead)} {
        set RPT(rmargin) [expr $RPT(rmargin) - $RPT(rmdead)]
    } else {
        set RPT(rmargin) $RPT(rmdead)
    }
    if {$RPT(tmargin) >= $RPT(tmdead)} {
        set RPT(tmargin) [expr $RPT(tmargin) - $RPT(tmdead)]
    } else {
        set RPT(tmargin) $RPT(tmdead)
    }
    if {$RPT(bmargin) >= $RPT(bmdead)} {
        set RPT(bmargin) [expr $RPT(bmargin) - $RPT(bmdead)]
    } else {
        set RPT(bmargin) $RPT(bmdead)
    }
    set RPT(paperdeff) 1
    if {!$RPT(postscriptf)} {
        set RPT(lpp) [expr int(($RPT(ph)-$RPT(bmargin))/$RPT(ppl))]
    }
}                    

#-----------------------------
# rptDefStyle - Set font, font style and size, leading, 
#                  justification, line-wrapping, assign to symbolic name
proc rptDefStyle {args} {
    global RPT
    # -- provide missing prerequisite calls (only requires init)
    if {![info exists RPT(initializedf)]} {rptInitialize -type ASCII}
    if {!$RPT(initializedf)} {rptInitialize -type ASCII}
    # -- set default style options
    set wrapf 1
    set font Courier
    set size 10
    set bold 0
    set italic 0
    set gap 2
    set underlinef 0
    set just L
    set name {}
    set fontflag 0
    set lindent 0
    set rindent 0
    set lineprefix [list ""]
    set wrapindent 0
    set lineindent 0
    set wrapchar [list ""]
    set tabstyle "A"
    set tabs [list .5iL 1iL 1.5iL 2iL 2.5iL 3iL 3.5iL 4iL 4.5iL 5iL 5.5iL 6iL 6.5iL]
    # -- process options
    foreach option [rptuOptions $args] {
        switch -glob -- [string tolower [lindex $option 0]] {
            -nam* {
                set name [lindex $option 1]
                if {[info exists RPT(style-$name)]} {
                    unset RPT(style-$name)
                }
            }
            -fo* {set font [lindex $option 1]; set fontflag 1}
            -b* {set bold 1; set fontflag 1}
            -i* {set italic 1; set fontflag 1}
            -s* {set size [rptu2point [lindex $option 1]]; set fontflag 1}
            -g* {set gap [rptu2point [lindex $option 1]]}
            -u* {set underlinef 1}
            -a* {
                switch -glob -- [string tolower [lindex $option 1]] {
                    l* {set just L}
                    r* {set just R}
                    c* {set just C}
                    j* {set just J}
                    default {set just L}
                }
            }
            -wrap {set wrapf 1}
            -now* {set wrapf 0}
            -wrapi* {set wrapindent [rptu2point [lindex $option 1]]}
            -wrapc* {
                set wrapchar {}
                if {[llength $option] > 1} {
                    set wrapchar [lindex $option 1]
                }
            }
            -linep* {
                set lineprefix {}
                if {[llength $option] > 1} {
                    set lineprefix [lindex $option 1]
                }
            }
            -linei* {set lineindent [rptu2point [lindex $option 1]]}
            -tabst* {
                switch -glob -- [string tolower [lindex $option 1]] {
                    a* {set tabstyle "A"}
                    n* {set tabstyle "N"}
                    default {set tabstyle "A"}
                }
            }
            -tabs {
                set tabs {}
                if {[llength $option] > 1} {set tabs [lrange $option 1 end]}
            }
            -le* {set lindent [rptu2point [lindex $option 1]]}
            -r* {set rindent [rptu2point [lindex $option 1]]}
            default {
                puts stderr "WARNING: rptDefStyle received unrecognized option:  \[$option\]"
            }
        }   
    }
    # -- build appropriate Postscript font name from options
    if {$fontflag} {
        set font [rptuPSfontEval $font $bold $italic]
    }
    # -- process tabs
    set sortlist {}
    set tabtyp {}
    set tabpos {}
    foreach tab $tabs {
        set lchar [string tolower [string index $tab [expr [string length $tab] - 1]]]
        switch -exact -- $lchar {
            l {lappend sortlist "[rptu2point [string range $tab 0 [expr [string length $tab] - 2]]] L"}
            c {lappend sortlist "[rptu2point [string range $tab 0 [expr [string length $tab] - 2]]] C"}
            r {lappend sortlist "[rptu2point [string range $tab 0 [expr [string length $tab] - 2]]] R"}
            default {lappend sortlist "[rptu2point $tab] L"}
        }
    }
    if {[llength $sortlist] > 0} {
        set sortlist [lsort -command rptuTabSort $sortlist]
        foreach tab $sortlist {
            lappend tabpos [lindex $tab 0]
            lappend tabtyp [lindex $tab 1]
        }
    }
    # -- store style options for later recall
    if {$name != ""} {
        set RPT(style-$name) \
            [list $font $size $gap $underlinef $just $wrapf $lindent \
            $rindent $lineindent "$lineprefix" $wrapindent "$wrapchar" \
            $tabstyle "$tabpos" "$tabtyp"]
    }
}

#-----------------------------
# rptDone - Report complete, send to file / printer
proc rptDone {args} {
    global RPT
    # -- provide missing prerequisite calls
    if {![info exists RPT(initializedf)]} {rptInitialize -type ASCII}
    if {!$RPT(initializedf)} {rptInitialize -type ASCII}
    if {!$RPT(paperdeff)} {
        rptDefPaper -pagesize letter -margins 0i -orientation portrait
    }
    if {!$RPT(pagedeff)} {rptDefPage}
    # -- process options
    set printer {}
    set destfile {}
    set copies 1
    foreach option [rptuOptions $args] {
        switch -glob -- [string tolower [lindex $option 0]] {
            -P* {
                if {[llength $option] == 1} {
                    set plen [string length [lindex $option 0]]
                    incr plen -1
                    set printer [string range [lindex $option 0] 2 $plen]
                } else {
                    set printer [lindex $option 1]
                }
            }
            -f {set destfile [lindex $option 1]}
            -c* {
                if {[catch {expr int(1 * [lindex $args 1])} copies]} {
                    puts stderr "rptDone -- non-numeric copy count \[$args\]; default to 1"
                set copies 1
                }
            }
        }
    }
    # -- eject last non-empty page and do whatever termination required
    if {$RPT(postscriptf)} {
        if {$RPT(atlinestartf) != 1 || $RPT(lineno) != 1} {
            puts $RPT(fd) "showpage"
            puts $RPT(fd) "restore"
            puts $RPT(fd) "end"
        }
        if {$RPT(pagenocallback) != ""} {
            catch {eval "$RPT(pagenocallback) [expr $RPT(pageno)-1]"}
        }
        rptuPStrailer
    } else {
        if {!$RPT(atlinestartf)} {
            if {0 == $RPT(footerlines)} {
                rptAddText "\f"
                rptNewLine 0
            } else {
                rptNewLine 0
                rptuTXTfooterOut
            }
        } else {
            if {0 == $RPT(footerlines)} {
                puts -nonewline $RPT(fd) "\f"
            } else {
                rptuTXTfooterOut
            }
        }
        if {$RPT(pagenocallback) != ""} {
            catch {eval "$RPT(pagenocallback) $RPT(pageno)"}
        }
    }
    # -- close temporary work file, print and or copy report output
    catch {close $RPT(fd)}
    if {$printer != ""} {
        set spooler "/usr/ucb/lpr"
        if {![file exists $spooler]} {set spooler "/bin/lp"}
        for {set i 0} {$i < $copies} {incr i} {
            catch {exec /bin/cat $RPT(tempfile) | $spooler}
        }
    }
    if {$destfile != ""} {
        catch {exec /bin/cp $RPT(tempfile) $destfile}
    }
    catch {exec /bin/rm $RPT(tempfile)}
    unset RPT
}

#-----------------------------
# rptInitialize - (Re)Initialize Report Writer, set postscript/text mode
proc rptInitialize {args} {
    global RPT
    # -- if reinitializing, delete any work files and delete RPT
    if {[info exists RPT(initializedf)]} {
        if {$RPT(initializedf)} {
            if {$RPT(fd) != ""} {
                close $RPT(fd)
                catch [exec /bin/rm $RPT(tempfile)]
            }
        }
        unset RPT
    }
    # -- set RPT default values, process report type argument
    rptuSetDefaults
    foreach option [rptuOptions $args] {
        switch -glob -- [string tolower [lindex $option 0]] {
            -type {
                switch -glob -- [string tolower [lindex $option 1]] {
                    po* {set RPT(postscriptf) 1}
                    default {set RPT(postscriptf) 0}
                }
            }
        }
    }
    # -- create a temporary work file for this report
    set RPT(tempfile) "/tmp/rpt[pid][exec /bin/date +%M%S]"
    if {[catch "open $RPT(tempfile) w" RPT(fd)]} {
        error "error -- rptInitialize could not create temp file \[$RPT(tempfile)\]"
    }
    set RPT(initializedf) 1
}

#-----------------------------
# rptNewLine - Move to new line of text
proc rptNewLine {args} {
    global RPT
    # -- provide missing prerequisite calls
    if {![info exists RPT(initializedf)]} {rptInitialize -type ASCII}
    if {!$RPT(initializedf)} {rptInitialize -type ASCII}
    if {!$RPT(paperdeff)} {
        rptDefPaper -pagesize letter -margins 0i -orientation portrait
    }
    if {!$RPT(pagedeff)} {rptDefPage}
    # -- determine number of newlines
    set numnewlines 1
    if {[llength $args] > 0} {
        if {[catch {expr int(1 * [lindex $args 0])} numnewlines]} {
            puts stderr "rptNewLine -- non-numeric newline count \[$args\], 1 assumed"
            set numnewlines 1
        }
    }
    set RPT(justify) [lindex $RPT(style-$RPT(stylecurrent)) 4]
    # -- flush the current line
    if {!$RPT(atlinestartf) && !$RPT(postscriptf)} {
        if {$RPT(lineno) >= $RPT(lastbodylineno)} {
            # -- capture line context, newpage, restore context, print
            set atlinestartf $RPT(atlinestartf)
            set text "$RPT(textline)"
            set ultext "$RPT(utextline)"
            set ulflag $RPT(tunderf)
            set spos $RPT(spos)
            set cpos $RPT(cpos)
            set epos $RPT(epos)
            set RPT(atlinestartf) 1
            rptNewPage
            set RPT(atlinestartf) $atlinestartf
            set RPT(textline) "$text"
            set RPT(utextline) "$ultext"
            set RPT(tunderf) $ulflag
            set RPT(spos) $spos
            set RPT(cpos) $cpos
            set RPT(epos) $epos
        }
        # -- output current ASCII text buffer and any underlining
        set line "$RPT(textline)"
        if {$RPT(tunderf)} {
            catch {puts -nonewline $RPT(fd) "$line\r"}
            set line "$RPT(utextline)"
            set RPT(tunderf) 0
        }
        catch {puts $RPT(fd) "$line"}
        incr RPT(lineno)
        # -- put out any additional blank lines for body leading
        for {set i 0} {$i<$RPT(txtnocrs)} {incr i} {
            if {$RPT(lineno) >= $RPT(lastbodylineno)} {
                # -- capture line context, newpage, restore context, print
                set atlinestartf $RPT(atlinestartf)
                set text "$RPT(textline)"
                set ultext "$RPT(utextline)"
                set ulflag $RPT(tunderf)
                set spos $RPT(spos)
                set cpos $RPT(cpos)
                set epos $RPT(epos)
                set RPT(atlinestartf) $atlinestartf
                rptNewPage
                set RPT(atlinestartf) $atlinestartf
                set RPT(textline) "$text"
                set RPT(utextline) "$ultext"
                set RPT(tunderf) $ulflag
                set RPT(spos) $spos
                set RPT(cpos) $cpos
                set RPT(epos) $epos
            }
            puts $RPT(fd) ""
            incr RPT(lineno)
        }
        # -- reset text buffers to empty
        set RPT(textline) $RPT(blankline)
        set RPT(utextline) $RPT(blankline)
        set RPT(tunderf) 0
        set RPT(curtab) -1
        if {$RPT(wrappingf)} {
            set RPT(spos) [expr int(($RPT(lmargin)+$RPT(wrapindent)+$RPT(indent))/$RPT(ppc))]
            set RPT(wrappingf) 0
        } else {
            set RPT(spos) [expr int(($RPT(lmargin)+$RPT(indent))/$RPT(ppc))]
        }
        set RPT(epos) [expr int(($RPT(pw)-($RPT(rmargin)+$RPT(rindent)))/$RPT(ppc))]
        switch -exact -- $RPT(justify) {
            L {set RPT(cpos) $RPT(spos)}
            J {set RPT(cpos) $RPT(spos)}
            C {set RPT(cpos) [expr $RPT(spos)+int(($RPT(epos)-$RPT(spos))/2.0)]}
            R {set RPT(cpos) $RPT(epos)}
        }
        set RPT(lasttabposused) $RPT(cpos)
        set RPT(atlinestartf) 1
        set RPT(curtab) -1
    } 
    if {$RPT(postscriptf)} {
        # -- flush the current line buffer
        if {$RPT(textline) != ""} {
            lappend RPT(tlist) [list 2 "$RPT(textline)"]
            lappend RPT(tlist) [list 1 $RPT(strwid)]
            incr RPT(tlistlen) 2
            set RPT(textline) ""
            set RPT(textf) 1
        }
        if {$RPT(textf)} {
            rptuPSflush
            # -- new line
            set ydelta [expr $RPT(gap)+$RPT(size)]
            if {[expr $RPT(cy) - $ydelta] < $RPT(footerlines)} {
                rptNewPage
                set ydelta 0
            } 
            set RPT(cy) [expr $RPT(cy) - $ydelta]
            incr RPT(lineno)
            set RPT(psyloc) $RPT(cy)
            set just [lindex $RPT(style-$RPT(stylecurrent)) 4]
            switch -exact -- $just {
                L {set RPT(cpos) 0}
                J {set RPT(cpos) 0}
                C {set RPT(cpos) [expr $RPT(linwid) / 2.0]}
                R {set RPT(cpos) $RPT(linwid)}
            }
            # -- reset packets list and line position characteristics
            set RPT(tlist) ""
            set RPT(tlistlen) 0
            set RPT(lasttabposused) -1
            set RPT(blankcount) 0
            set RPT(curtab) -1
            set RPT(textline) ""
            set RPT(strwid) 0
            set RPT(textf) 0
        }
    }
    # -- put out the requested number of blank lines (0=do nothing)
    if {!$RPT(postscriptf)} {
        if {$RPT(txtnocrs) > 0} {
            set numnewlines [expr $numnewlines*$RPT(txtnocrs)]
        }
        for {set i 0} {$i < $numnewlines} {incr i} {
            if {$RPT(lineno) >= $RPT(lastbodylineno)} {
                # -- capture line context, newpage, restore context, print
                set atlinestartf $RPT(atlinestartf)
                set text "$RPT(textline)"
                set ultext "$RPT(utextline)"
                set ulflag $RPT(tunderf)
                set spos $RPT(spos)
                set cpos $RPT(cpos)
                set epos $RPT(epos)
                set RPT(atlinestartf) $atlinestartf
                rptNewPage
                set RPT(atlinestartf) $atlinestartf
                set RPT(textline) "$text"
                set RPT(utextline) "$ultext"
                set RPT(tunderf) $ulflag
                set RPT(spos) $spos
                set RPT(cpos) $cpos
                set RPT(epos) $epos
            }
            puts $RPT(fd) ""
            incr RPT(lineno)
        }
    } else {
        # -- Postscript, move down the page for the number of blank lines
        set i 0
        set ydelta [expr $RPT(gap)+$RPT(size)]
        while {$i < $numnewlines} {
            incr i
            if {[expr $RPT(cy) - $ydelta] < $RPT(footerlines)} {
                rptNewPage
                set ydelta 0
            } 
            set RPT(cy) [expr $RPT(cy) - $ydelta]
            incr RPT(lineno)
            set RPT(psyloc) $RPT(cy)
            set RPT(psxloc) [expr $RPT(leftedge) + $RPT(cpos)]
        }
        set RPT(atlinestartf) 1
    } 
    set RPT(curtab) -1
}

#-----------------------------
# rptNewPage - (Un)Conditionally eject current page and start a new page
proc rptNewPage {args} {
    global RPT
    # -- provide missing prerequisite calls
    if {![info exists RPT(initializedf)]} {rptInitialize -type ASCII}
    if {!$RPT(initializedf)} {rptInitialize -type ASCII}
    if {!$RPT(paperdeff)} {
        rptDefPaper -pagesize letter -margins 0i -orientation portrait
    }
    if {!$RPT(pagedeff)} {rptDefPage}
    # -- check for conditional new page
    set guarddist -1
    set guardlines -1
    foreach option [rptuOptions $args] {
        switch -glob -- [string tolower [lindex $option 0]] {
            -guard {set guarddist [lindex $option 1]}
            -guardl* {set guardlines [lindex $option 1]}
        }
    }
    if {$guardlines > 0} {
        if {!$RPT(postscriptf)} {
            if {[expr $RPT(lpp)-$RPT(footerlines)] > \
                    [expr $RPT(lineno) + $guardlines - 1]} {
                return 0
            }
        } else {
            # -- Postscript
            set ydelta [expr $guardlines * ($RPT(size)+$RPT(gap))]
            if {[expr $RPT(cy) - $ydelta] >= $RPT(footerlines)} {return 0}
        }
    }
    if {$guarddist > 0} {
        if {!$RPT(postscriptf)} {
            set cy [expr ($RPT(footerlines)-$RPT(lineno)) * RPT(ppl)]
            if {$cy > $guarddist} {return 0}
        } else {
            # -- Postscript
            if {[expr $RPT(cy) - $guarddist] >= $RPT(footerlines)} {return 0}
        }
    }
    # -- eject current page, set up new page
    if {$RPT(postscriptf)} {
        # -- flush the current line buffer
        if {$RPT(textline) != ""} {
            lappend RPT(tlist) [list 2 "$RPT(textline)"]
            lappend RPT(tlist) [list 1 $RPT(strwid)]
            incr RPT(tlistlen) 2
            set RPT(textline) ""
            set RPT(textf) 1
        }
        if {$RPT(textf)} {
            rptuPSflush
        }
        puts $RPT(fd) "showpage"
        puts $RPT(fd) "restore"
        puts $RPT(fd) "end"
        if {$RPT(pagenocallback) != ""} {
            catch {eval "$RPT(pagenocallback) [expr $RPT(pageno) - 1]"}
        }
        rptuPSpagedecorate
        rptuPSflush
        set RPT(atlinestartf) 1
        # -- clean up justification with empty line
        set just [lindex $RPT(style-$RPT(stylecurrent)) 4]
        switch -exact -- $just {
            L {set RPT(cpos) 0}
            J {set RPT(cpos) 0}
            C {set RPT(cpos) [expr $RPT(linwid) / 2.0]}
            R {set RPT(cpos) $RPT(linwid)}
        }
        # -- reset packets list and line position characteristics
        set RPT(tlist) ""
        set RPT(tlistlen) 0
        set RPT(lasttabposused) -1
        set RPT(blankcount) 0
        set RPT(curtab) -1
        set RPT(textline) ""
        set RPT(strwid) 0
        set RPT(lineno) 1
        set RPT(textf) 0
    } else {
        set stylecurrent $RPT(stylecurrent)
        if {$RPT(footerlines) == 0} {
            if {$RPT(lineno) <= [expr $RPT(ph)/$RPT(ppl)]} {
                catch {puts -nonewline $RPT(fd) "\f"}
            }
        } else {
            rptuTXTfooterOut
        }
        if {$RPT(pagenocallback) != ""} {
            catch {eval "$RPT(pagenocallback) $RPT(pageno)"}
        }
        incr RPT(pageno)
        set RPT(lineno) 1
        rptuTXTheaderOut
        rptSetStyle $stylecurrent
        set RPT(textline) $RPT(blankline)
        set RPT(utextline) $RPT(blankline)
        if {$RPT(wrappingf)} {
            set RPT(spos) [expr int(($RPT(lmargin)+$RPT(wrapindent)+$RPT(indent))/$RPT(ppc))]
            set RPT(wrappingf) 0
        } else {
            set RPT(spos) [expr int(($RPT(lmargin)+$RPT(indent))/$RPT(ppc))]
        }
        set RPT(epos) [expr int(($RPT(pw)-($RPT(rmargin)+$RPT(rindent)))/$RPT(ppc))]
        switch -exact -- $RPT(justify) {
            L {set RPT(cpos) $RPT(spos)}
            J {set RPT(cpos) $RPT(spos)}
            C {set RPT(cpos) [expr $RPT(spos)+int(($RPT(epos)-$RPT(spos))/2.0)]}
            R {set RPT(cpos) $RPT(epos)}
        }
        set RPT(lasttabposused) $RPT(cpos)
    }
    return 1
}
        
#-----------------------------
# rptSetStyle - retrieve a style definition and put it into effect
proc rptSetStyle {name} {
    global RPT
    # -- provide missing prerequisite calls
    if {![info exists RPT(initializedf)]} {rptInitialize -type ASCII}
    if {!$RPT(initializedf)} {rptInitialize -type ASCII}
    # -- fetch style settings and put into effect if different than current
    if {![info exists RPT(style-$name)]} {
        puts stderr "WARNING! TclVSrpt:rptSetStyle - style $name not found"
        return
    }
    set RPT(stylecurrent) $name
    set font [lindex $RPT(style-$name) 0]
    set size [lindex $RPT(style-$name) 1]
    set gap [lindex $RPT(style-$name) 2]
    set underlinef [lindex $RPT(style-$name) 3]
    set justify [lindex $RPT(style-$name) 4]
    set wrap [lindex $RPT(style-$name) 5]
    set lindent [lindex $RPT(style-$name) 6]
    set rindent [lindex $RPT(style-$name) 7]
    set lineindent [lindex $RPT(style-$name) 8]
    set lineprefix "[lindex $RPT(style-$name) 9]"
    if {$lineprefix == "{}"} {set lineprefix ""}
    set wrapindent [lindex $RPT(style-$name) 10]
    set wrapchar "[lindex $RPT(style-$name) 11]"
    if {$wrapchar == "{}"} {set wrapchar ""}
    set RPT(tabstyle) [lindex $RPT(style-$name) 12]
    set RPT(tabs) ""
    set RPT(tabtypes) "" 
    set RPT(tabs) [lindex $RPT(style-$name) 13]
    set RPT(tabtypes) [lindex $RPT(style-$name) 14]
    set RPT(numtabs) [llength $RPT(tabs)]
    # -- process font and font options
    if {$RPT(font) != $font || $RPT(size) != $size || 
        $RPT(uflag) != $underlinef} {
        set RPT(font) $font
        set RPT(size) $size
        set RPT(uflag) $underlinef
        if {$RPT(postscriptf)} {
            # -- if this font not seen before, load the character widths
            if {![info exists RPT($RPT(font),a)]} {
                source "[info library]/TclVS/fontwidths/$RPT(font).tcl"
            }
            if {-1 == [lsearch $RPT(fontsloaded) $RPT(font)]} {
                lappend RPT(fontsloaded) $RPT(font)
            }
            set RPT(blankwidth) [rptuPSstrwid " "]
            # -- if there is accumulated text, format the packet
            if {$RPT(textline) != ""} {
                lappend RPT(tlist) [list 2 "$RPT(textline)"]
                lappend RPT(tlist) [list 1 $RPT(strwid)]
                incr RPT(tlistlen) 2
                set RPT(textline) ""
                set RPT(textf) 1
            }
            # -- create a font change packet
            set packet [list 3 $RPT(font) $RPT(size) $RPT(uflag)]
            lappend RPT(tlist) $packet
            incr RPT(tlistlen)
        }
    }
    # -- process line leading
    if {$RPT(gap) != $gap} {
        set RPT(gap) $gap
        if {!$RPT(postscriptf)} {
            set RPT(txtnocrs) [expr int(($RPT(gap)/$RPT(ppl))+.5)]
            if {$RPT(txtnocrs) < 0} {set RPT(txtnocrs) 0}
        }
    }
    # -- process underlining
    if {$RPT(underlinef) != $underlinef} {
        set RPT(underlinef) $underlinef
    }
    # -- process text alignment
    if {$RPT(justify) != $justify} {
        set RPT(justify) $justify
    }
    if {$RPT(wrapf) != $wrap} {
        set RPT(wrapf) $wrap
    }
    # -- process incremental left indentation
    if {$RPT(indent) != $lindent} {
        set RPT(indent) $lindent
    }
    # -- process incremental right indentation
    if {$RPT(rindent) != $rindent} {
        set RPT(rindent) $rindent
    }
    # -- process first line indentation
    if {$RPT(lineindent) != $lineindent} {
        set RPT(lineindent) $lineindent
    }
    # -- process line prefix characters
    if {$RPT(lineprefix) != $lineprefix} {
        set RPT(lineprefix) $lineprefix
    }
    # -- process wrapped line indentation
    if {$RPT(wrapindent) != $wrapindent} {
        set RPT(wrapindent) $wrapindent
    }
    # -- process wrapped line suffix characters
    if {$RPT(wrapchar) != "$wrapchar"} {
        set RPT(wrapchar) "$wrapchar"
        if {!$RPT(postscriptf)} {
            set RPT(wrapcharwid) [string length "$RPT(wrapchar)"]
        }
    }
    # -- process tabs
    if {$RPT(curtab) != -1 && $RPT(curtab) != 999} {
        if {$RPT(curtab) > [llength $RPT(tabs)]} {set RPT(curtab) 999}
    }
    # -- reset text buffer variables if at start of line
    if {!$RPT(postscriptf)} {
        if {$RPT(atlinestartf)} {
            set RPT(textline) $RPT(blankline)
            set RPT(utextline) $RPT(blankline)
            set RPT(tunderf) 0
            set RPT(curtab) -1
            if {$RPT(wrappingf)} {
                set RPT(spos) [expr int(($RPT(lmargin)+$RPT(wrapindent)+$RPT(indent))/$RPT(ppc))]
                set RPT(wrappingf) 0
            } else {
                set RPT(spos) [expr int(($RPT(lmargin)+$RPT(indent))/$RPT(ppc))]
            }
            set RPT(epos) [expr int(($RPT(pw)-($RPT(rmargin)+$RPT(rindent)))/$RPT(ppc))]
            switch -exact -- $RPT(justify) {
                L {set RPT(cpos) $RPT(spos)}
                J {set RPT(cpos) $RPT(spos)}
                C {set RPT(cpos) [expr $RPT(spos)+int(($RPT(epos)-$RPT(spos))/2.0)]}
                R {set RPT(cpos) $RPT(epos)}
            }
            set RPT(lasttabposused) $RPT(cpos)
        }
    } else {
        # -- set up postscript constants for this style
        if {$RPT(tlistlen) == 0 || $RPT(atlinestartf)} {set RPT(curtab) -1}
        set RPT(blankwidth) [rptuPSstrwid " "]
        if {$RPT(wrapchar) != ""} {
            set RPT(wrapcharwid) [rptuPSstrwid "$RPT(wrapchar)"]
        } else {
            set RPT(wrapcharwid) 0.0
        }
        set RPT(leftedge) [expr $RPT(lmargin)+$RPT(indent)]
        set RPT(rightedge) [expr $RPT(pw) - ($RPT(rindent)+$RPT(rmargin))]
        set RPT(linwid) [expr $RPT(rightedge) - $RPT(leftedge)]
        set indent 0
        # -- see if we should apply line start indentation
        if {$RPT(justify) == "L" || $RPT(justify) == "J"} {
            if {$RPT(tlistlen) == 0 && $RPT(textline) == "" && \
                    $RPT(lineindent) != 0} {
                lappend RPT(tlist) [list 1 $RPT(lineindent)]
                set RPT(strwid) $RPT(lineindent)
                set RPT(cpos) $RPT(lineindent)
                set RPT(lasttabposused) $RPT(tlistlen)
                incr RPT(tlistlen)
                set indent $RPT(lineindent)
            } else {
                if {$RPT(wrappingf)} {
                    set indent $RPT(wrapindent)
                }
            }
        }
        if {!$RPT(textf)} {
            switch -exact -- $RPT(justify) {
                L {set RPT(cpos) $indent}
                J {set RPT(cpos) $indent}
                C {set RPT(cpos) [expr $RPT(linwid) / 2.0]}
                R {set RPT(cpos) $RPT(linwid)}
            }
        }
    }            
    set RPT(numtabs) [llength $RPT(tabs)]
}

#-----------------------------
# INTERNAL UTILITY PROCEDURES:
#=============================
#
#-----------------------------
# rptu2point -- parse a size / coordinate value into printer points 
#               (1/72 inch).  "1i" = 1 inch = 72 points = 72; 
#               "2.54c" = 2.54 centimeters = 72 points = 72;
#               "72" = 72 points = 72
#
proc rptu2point {value} {
    set lc [string index $value [expr [string length $value] - 1]]
    set factor "1 72 [expr 72/2.54]"
    set val [string range $value 0 [expr [string length $value] - 2]]
    switch -exact -- $lc {
        p {set fac [lindex $factor 0]}
        i {set fac [lindex $factor 1]}
        c {set fac [lindex $factor 2]}
        default {set val $value; set fac 1}
    }
    if {[catch {expr $val * $fac} val]} {
        set procname [lindex [info level [expr [info level] - 1]] 0]
        error \
           "error -- $procname called rptu2point with non-numeric value \[$value\]"
    }
    return $val
}

#-----------------------------
# rptuNum2Rom -- convert a number (page number) into equivalent
#                roman numeral string.  Max value is "9999".
proc rptuNum2Rom {args} {
    set capflag 0
    if {[llength $args] > 1} {set capflag 1}
    set val [lindex $args 0]
    set ones "i ii iii iv v vi vii viii ix" 
    set tens "x xx xxx xl l lx lxx lxxx xc"
    set hundreds "c cc ccc cd d dc dcc dcc cm"
    set thousands "m mm mmm mmmm mmmmm mmmmmm mmmmmmm mmmmmmmm mmmmmmmmm"
    set val [expr $val % 10000]
    set string ""   
    set v [expr $val / 1000]
    set val [expr $val % 1000]
    if {$v > 0} {set string "$string[lindex $thousands [expr $v - 1]]"}
    set v [expr $val / 100]
    set val [expr $val  % 100]
    if {$v > 0} {set string "$string[lindex $hundreds [expr $v - 1]]"}
    set v [expr $val / 10]
    set val [expr $val % 10]
    if {$v > 0} {set string "$string[lindex $tens [expr $v - 1]]"}
    if {$val > 0} {set string "$string[lindex $ones [expr $val - 1]]"}
    if {$capflag} {set string [string toupper $string]}
    return $string
}

#-----------------------------
# rptuOptions -- take a variable argument list and parse into a list
#                of sublists, each starting with the "-optionname"
#                element.
#
proc rptuOptions {list} {
    # -- take a command option list and parse into sublists
    set result {}
    set sublist {}
    # -- see if the argument list was given as a string rather than list
    if {[llength $list] == 1} {
        if {[regexp {\ |\t|\n} $list]} {
            set list [lindex $list 0]
        }
    }
    foreach item $list {
        # -- find option name
        if {[string index $item 0] == "-"} {
            if {$sublist != ""} {
                lappend result $sublist
                set sublist {}
            }
        }
        lappend sublist $item
    }
    if {$sublist != ""} {
        lappend result $sublist
    }
    return $result
}

#-----------------------------
# rptuPSfontEval -- translate an identifiable font name fragment, 
#                   bolding flag, and italics flag into a valid 
#                   Postscript font name.
#
proc rptuPSfontEval {font boldf italicf} {
    global RPT
    set idx 0
    switch -glob -- [string tolower $font] {
        cour* {set idx 0}        # Courier
        time* {set idx 1}        # Times
        helvetica {set idx 2}    # Helvetica
        helvetica-n* {set idx 3} # Helvetica-Narrow
        symb* {set idx 4}        # Symbol
        avan* {set idx 5}        # AvanteGarde
        newc* {set idx 6}        # NewCenturySchlbk
        book* {set idx 7}        # Bookman
        pala* {set idx 8}        # Palatino
        *chan* {set idx 9}       # ZapfChancery-MediumItalic
        *ding* {set idx 10}      # ZapfDingbats
        default {set idx 0}      # set unrecognized font to Courier
    }
    set f [lindex $RPT(psfonts) $idx]
    set m1 ""
    set m2 ""
    if {$boldf == 0 && $italicf == 0} {
        set m1 [lindex $RPT(psfontsnormal) $idx]
    } else {
        if {$boldf} {
            set m1 [lindex $RPT(psfontsbold) $idx]
        } else {
            set m1 [lindex $RPT(psfontslight) $idx]
        }
        if {$italicf} {
            set m2 [lindex $RPT(psfontsangle) $idx]
        }
    }
    return "$f$m1$m2"
}

#-----------------------------
# rptuPSflush -- flush the current accumulation if Postscript directives
proc rptuPSflush {} {
    global RPT
    set pstext ""
    set i -1
    set RPT(psxloc) $RPT(leftedge)
    set RPT(psyloc) $RPT(cy)
    set bcnt 0
    set last [expr $RPT(tlistlen) - 1]
    foreach packet $RPT(tlist) {
        incr i
        if {$i > $RPT(tlistlen)} {break}
        switch -exact -- [lindex $packet 0] {
            1 { 
                set RPT(psxloc) [expr $RPT(leftedge) + [lindex $packet 1]]
                if {$i > $RPT(lasttabposused) && $RPT(blankincr) > 0} {
                    set RPT(psxloc) [expr $RPT(psxloc)+($RPT(blankincr)*$bcnt)]
                }
            }
            2 {
                # -- text; if not underlining, handle immediately
                set text "[lindex $packet 1]"
                set slen [string length "$text"]
                if {!$RPT(psuflag)} {
                    regsub -all {\(|\)|\/|\\} $text {\\&} pstext
                    if {$i > $RPT(lasttabposused) && $RPT(blankincr) > 0} {
                        puts $RPT(fd) "$RPT(blankincr) $RPT(psxloc) $RPT(psyloc) ($pstext) TJ"
                        foreach c [split "$text" ""] {if {$c == " "} {incr bcnt}}
                    } else {
                        puts $RPT(fd) "$RPT(psxloc) $RPT(psyloc) ($pstext) T"
                    }
                } else {
                    # -- underlined text (ugh!)
                    # -- split text into a list of characters, put out a 
                    # -- text and a line packet for each set of non-blank chars
                    set pc [string index $text 0]
                    set word ""
                    set strwid 0
                    if {$i > $RPT(lasttabposused) && $RPT(blankincr) > 0} {
                        set bwid [expr $RPT(blankincr) + ($RPT($RPT(psfont), )*$RPT(pssize))]
                        foreach c [split "$text" ""] {if {$c == " "} {incr bcnt}}
                    } else {
                        set bwid [expr $RPT($RPT(psfont), )*$RPT(pssize)]
                    }
                    set thick [expr $RPT(pssize) / 15.0]
                    set dy [expr $RPT(psyloc) - (1.0 + ($RPT(pssize) / 10.0))]
                    foreach c [split "$text" ""] {
                        if {$c == " "} {
                            if {$pc == " "} {
                                set RPT(psxloc) [expr $RPT(psxloc) + $bwid]
                            } else {
                                # -- word to blank transition
                                set ex [expr $RPT(psxloc) + $strwid]
                                regsub -all {\(|\)|\/|\\} $word {\\&} pstext
                                puts $RPT(fd) "$RPT(psxloc) $RPT(psyloc) ($pstext) T"
                                puts $RPT(fd) "$thick $RPT(psxloc) $dy $ex $dy DL"
                                set RPT(psxloc) [expr $ex + $bwid]
                                set word ""
                                set strwid 0
                            }
                        } else {
                            if {$pc == " "} {
                                # -- blank to word transition
                                set word "$c"
                                if {$c < " " || $c > "~"} {
                                    scan $c %c fc
                                    set fc [format %02 $fc]
                                    set strwid [expr $RPT($RPT(psfont),$fc) * $RPT(pssize)]
                                } else {
                                    set strwid [expr $RPT($RPT(psfont),$c) * $RPT(pssize)]
                                }
                            } else {
                                set word "${word}${c}"
                                if {$c < " " || $c > "~"} {
                                    scan $c %c fc
                                    set fc [format %02 $fc]
                                    set strwid [expr $strwid + ($RPT($RPT(psfont),$fc) * $RPT(pssize))]
                                } else {
                                    set strwid [expr $strwid + ($RPT($RPT(psfont),$c) * $RPT(pssize))]
                                }
                            }
                        }
                        set pc $c
                    }
                    if {$word != ""} {
                        set ex [expr $RPT(psxloc) + $strwid]
                        regsub -all {\(|\)|\/|\\} $word {\\&} pstext
                        puts $RPT(fd) "$RPT(psxloc) $RPT(psyloc) ($pstext) T"
                        puts $RPT(fd) "$thick $RPT(psxloc) $dy $ex $dy DL"
                        set RPT(psxloc) $ex
                        set strwid 0
                    }
                }
            }
            3 {
                # -- font / size change, dispatch accumulated text
                set RPT(psfont) [lindex $packet 1]
                set RPT(pssize) [lindex $packet 2]
                set RPT(psuflag) [lindex $packet 3]
                puts $RPT(fd) "/$RPT(psfont) $RPT(pssize) F"
            }
            default {
                puts stderr "rptuPSflush - received unrecognized packet \[$packet\]"
            }
        }
    }
    # -- reset packets list and line position characteristics
    set RPT(tlist) ""
    set RPT(tlistlen) 0
    set RPT(lasttabposused) -1
    set RPT(blankcount) 0
    set RPT(curtab) -1
    set RPT(textline) ""
    set RPT(blankincr) 0
    set RPT(textf) 0
}

#----------------------------
# rptuPSpagedecorate -- put header and footers on postscript page
proc rptuPSpagedecorate {} {
    global RPT
    # -- capture current style information
    set stylecurrent $RPT(stylecurrent)
    set stylecurrentsize $RPT(size)
    # -- capture and reset postscript list
    set tlist $RPT(tlist)
    set tlistlen $RPT(tlistlen)
    set blankincr $RPT(blankincr)
    set RPT(tlist) ""
    set RPT(tlistlen) 0
    set RPT(cy) [expr $RPT(ph) - $RPT(tmargin)]
    set RPT(psyloc) $RPT(cy)
    set RPT(blankincr) 0
    # -- format page number
    set pval [expr $RPT(pageno) + ($RPT(pagenostart) - 1)]
    switch -exact -- $RPT(pagenostyle) {
        N {}
        R {set pval [rptuNum2Rom $pval]}
        C {set pval [rptuNum2Rom $pval C]}
    }
    # -- put out the Postscript structural comment for page start
    puts $RPT(fd) "%%Page: $pval $RPT(pageno)"
    puts $RPT(fd) "TclVSrpt begin"
    # -- set the page orientation and clip area
    puts $RPT(fd) "save"
    if {$RPT(landscapef)} {
        puts $RPT(fd) "$RPT(ph) 0 translate 90 rotate"
    }
    # -- set the clip area
    set top [expr $RPT(ph) - $RPT(tmargin)]
    set RPT(cy) $top
    set right [expr $RPT(pw) - $RPT(rmargin)]
    puts $RPT(fd) "$RPT(lmargin) $RPT(bmargin) $right $top CW"
    # -- finish formating page number with its prefix and suffix
    set pval "$RPT(pagenopre)$pval$RPT(pagenosuf)"
    # -- put out page headers
    # -- put out title line
    if {$RPT(titletext) != ""} {
        rptSetStyle pageheader
        set strwid [rptuPSstrwid "$RPT(titletext)"]
        set RPT(cy) [expr ($RPT(ph) - $RPT(tmargin)) - $RPT(size)]
        lappend RPT(tlist) [list 1 [expr ($RPT(linwid)/2.0)-($strwid/2.0)]]
        lappend RPT(tlist) [list 2 "$RPT(titletext)"]
        incr RPT(tlistlen) 2
        rptuPSflush
        # -- move down page the amount of the title gap
        set RPT(cy) [expr $RPT(cy) - $RPT(gap)]
    }
    # -- if page number is to go into subtitle line, put it there
    if {[lindex $RPT(pagenoloc) 0] == "T" && $RPT(pagenof)} {
        if {[lindex $RPT(pagenoloc) 1]} {
            set l "$RPT(subtitleltext)"
            set r "$RPT(subtitlertext)"
        }
        switch -exact -- [lindex $RPT(pagenoloc) 2] {
            L {set RPT(subtitleltext) "$pval"}
            C {set RPT(subtitlectext) "$pval"}
            R {set RPT(subtitlertext) "$pval"}
        }
        if {[lindex $RPT(pagenoloc) 1]} {
            switch -exact -- [lindex $RPT(pagenoloc) 2] {
                L {set RPT(subtitlertext) "$l"; set RPT(pagenoloc) "T 1 R"}
                C {}
                R {set RPT(subtitleltext) "$r"; set RPT(pagenoloc) "T 1 L"}
            }
        }
    }
    # -- is there a subtitle to be printed?
    if {$RPT(subtitleltext) != "" || $RPT(subtitlectext) != "" || \
            $RPT(subtitlertext) != ""} {
        # -- set the subtitle style
        rptSetStyle pagesubheader
        set RPT(cy) [expr $RPT(cy) - $RPT(size)]
        # -- is there left text?
        if {$RPT(subtitleltext) != ""} {
            lappend RPT(tlist) [list 1 0]
            lappend RPT(tlist) [list 2 "$RPT(subtitleltext)"]
            incr RPT(tlistlen) 2
        }
        # -- is there center text?
        if {$RPT(subtitlectext) != ""} {
            set strwid [rptuPSstrwid $RPT(subtitlectext)]
            lappend RPT(tlist) [list 1 [expr ($RPT(linwid)/2.0)-($strwid/2.0)]]
            lappend RPT(tlist) [list 2 "$RPT(subtitlectext)"]
            incr RPT(tlistlen) 2
        }
        # -- is there right text?
        if {$RPT(subtitlertext) != ""} {
            set strwid [rptuPSstrwid $RPT(subtitlertext)]
            lappend RPT(tlist) [list 1 [expr $RPT(linwid)-$strwid]]
            lappend RPT(tlist) [list 2 "$RPT(subtitlertext"]
            incr RPT(tlistlen) 2
        }
        rptuPSflush
        # -- move down page by subtitle gap
        set RPT(cy) [expr $RPT(cy) - $RPT(gap)]
    }
    # -- is there a title bar to be drawn?
    if {$RPT(titlebarf)} {
        set RPT(psyloc) $RPT(cy)
        set cx [expr $RPT(leftedge) + $RPT(titlebarindent)]
        set ex [expr $RPT(leftedge) + ($RPT(linwid) - $RPT(titlebarindent))]
        puts $RPT(fd) "$cx $RPT(psyloc) M"
        puts $RPT(fd) "$RPT(titlebarthick) $cx $RPT(cy) $ex $RPT(cy) DL"
        set RPT(cy) [expr $RPT(cy) - $RPT(titlebargap)]
    }
    # -- set the y position where we will end up
    set startingcy [expr $RPT(cy) - $stylecurrentsize]
    set RPT(cy) $RPT(footerlines)
    # -- is there a footer bar to draw?
    if {$RPT(footerbarf)} {
        set RPT(cy) [expr $RPT(footerlines) - $RPT(footerbargap)]
        set RPT(psyloc) $RPT(cy)
        set cx [expr $RPT(leftedge) + $RPT(footerbarindent)]
        set ex [expr $RPT(leftedge) + ($RPT(linwid) - $RPT(footerbarindent))]
        puts $RPT(fd) "$cx $RPT(psyloc) M"
        puts $RPT(fd) "$RPT(footerbarthick) $cx $RPT(cy) $ex $RPT(cy) DL"
    }
    # -- if page number is to go into footer line, put it there
    if {[lindex $RPT(pagenoloc) 0] == "B" && $RPT(pagenof)} {
        if {[lindex $RPT(pagenoloc) 1]} {
            set l "$RPT(footerltext)"
            set r "$RPT(footerrtext)"
        }
        switch -exact -- [lindex $RPT(pagenoloc) 2] {
            L {set RPT(footerltext) "$pval"}
            C {set RPT(footerctext) "$pval"}
            R {set RPT(footerrtext) "$pval"}
        }
        if {[lindex $RPT(pagenoloc) 1]} {
            switch -exact -- [lindex $RPT(pagenoloc) 2] {
                L {set RPT(footerrtext) "$l"; set RPT(pagenoloc) "B 1 R"}
                C {}
                R {set RPT(footerltext) "$r"; set RPT(pagenoloc) "B 1 L"}
            }
        }
    }
    # -- if there is a page footer to write, set its style
    if {$RPT(footerltext)!=""||$RPT(footerctext)!=""||$RPT(footerrtext)!=""} {
        rptSetStyle pagefooter
    }
    set RPT(cy) [expr $RPT(cy) - ($RPT(gap)+$RPT(size))]
    # -- is there left text?
    if {$RPT(footerltext) != ""} {
        lappend RPT(tlist) [list 1 0]
        lappend RPT(tlist) [list 2 "$RPT(footerltext)"]
        incr RPT(tlistlen) 2
    }
    # -- is there center text?
    if {$RPT(footerctext) != ""} {
        set strwid [rptuPSstrwid $RPT(footerctext)]
        lappend RPT(tlist) [list 1 [expr ($RPT(linwid)/2.0)-($strwid/2.0)]]
        lappend RPT(tlist) [list 2 "$RPT(footerctext)"]
        incr RPT(tlistlen) 2
    }
    # -- is there right text?
    if {$RPT(footerrtext) != ""} {
        set strwid [rptuPSstrwid $RPT(footerrtext)]
        lappend RPT(tlist) [list 1 [expr $RPT(linwid)-$strwid]]
        lappend RPT(tlist) [list 2 "$RPT(footerrtext)"]
        incr RPT(tlistlen) 2
    }
    if {$RPT(tlistlen) > 0} {rptuPSflush}
    # -- restore postscript buffer state and move to the top line of page
    set RPT(tlist) ""
    set RPT(tlistlen) 0
    set RPT(textf) 0
    set RPT(cy) $startingcy
    set RPT(psyloc) $RPT(cy)
    rptSetStyle $stylecurrent
    lappend RPT(tlist) [list 3 $RPT(font) $RPT(size) $RPT(underlinef)]
    incr RPT(tlistlen)
    rptuPSflush
    foreach packet $tlist {lappend RPT(tlist) $packet}
    incr RPT(tlistlen) $tlistlen
    if {$RPT(lasttabposused) != -1} {incr RPT(lasttabposused)}
    set RPT(blankincr) $blankincr
    incr RPT(pageno)
    set RPT(curtab) -1
}

#----------------------------
# rptuPSprolog - Print Postscript Prolog
proc rptuPSprolog {} {
    global RPT
    puts $RPT(fd) "%%BeginProlog "
    puts $RPT(fd) "save"
    puts $RPT(fd) "/TclVSrpt 50 dict def"
    puts $RPT(fd) "TclVSrpt begin"
    puts $RPT(fd) "/M {"
    puts $RPT(fd) "    moveto"
    puts $RPT(fd) "} bind def"
    puts $RPT(fd) "/T {"
    puts $RPT(fd) "    /text exch def"
    puts $RPT(fd) "    /yloc exch def"
    puts $RPT(fd) "    /xloc exch def"
    puts $RPT(fd) "    xloc yloc moveto text show"
    puts $RPT(fd) "} bind def"
    puts $RPT(fd) "/TJ {"
    puts $RPT(fd) "    /text exch def"
    puts $RPT(fd) "    /yloc exch def"
    puts $RPT(fd) "    /xloc exch def"
    puts $RPT(fd) "    /bincr exch def"
    puts $RPT(fd) "    xloc yloc moveto bincr 0 8#040 text widthshow"
    puts $RPT(fd) "} bind def"
    puts $RPT(fd) "/DL {"
    puts $RPT(fd) "    /ey exch def"
    puts $RPT(fd) "    /ex exch def"
    puts $RPT(fd) "    /sy exch def"
    puts $RPT(fd) "    /sx exch def"
    puts $RPT(fd) "    /lthick exch def"
    puts $RPT(fd) "    currentpoint"
    puts $RPT(fd) "    /fy exch def"
    puts $RPT(fd) "    /fx exch def"
    puts $RPT(fd) "    gsave"
    puts $RPT(fd) "    newpath"
    puts $RPT(fd) "    lthick setlinewidth"
    puts $RPT(fd) "    sx sy moveto ex ey lineto stroke"
    puts $RPT(fd) "    grestore"
    puts $RPT(fd) "    fx fy moveto"
    puts $RPT(fd) "} bind def"
    puts $RPT(fd) "/F {"
    puts $RPT(fd) "    /fsize exch def"
    puts $RPT(fd) "    /fname exch def"
    puts $RPT(fd) "    fname findfont fsize scalefont setfont"
    puts $RPT(fd) "} bind def"
    puts $RPT(fd) "/CW {"
    puts $RPT(fd) "    /ury exch 4 add def"
    puts $RPT(fd) "    /urx exch def"
    puts $RPT(fd) "    /lly exch 4 sub def"
    puts $RPT(fd) "    /llx exch def"
    puts $RPT(fd) "    initclip"
    puts $RPT(fd) "    1 setgray"
    puts $RPT(fd) "    newpath"
    puts $RPT(fd) "    llx lly moveto"
    puts $RPT(fd) "    llx ury lineto"
    puts $RPT(fd) "    urx ury lineto"
    puts $RPT(fd) "    urx lly lineto"
    puts $RPT(fd) "    llx lly lineto"
    puts $RPT(fd) "    closepath"
    puts $RPT(fd) "    clip"
    puts $RPT(fd) "    0 setgray"
    puts $RPT(fd) "} bind def"
    puts $RPT(fd) "end"
    puts $RPT(fd) "%%EndProlog"
    # -- calculate header line and footer lines
    set RPT(headerlines) [expr $RPT(ph) - $RPT(tmargin)]
    if {$RPT(titletext) != ""} {
        rptSetStyle pageheader
        set RPT(headerlines) [expr $RPT(headerlines) - ($RPT(size)+$RPT(gap))]
    }
    if {$RPT(subtitleltext) != "" || $RPT(subtitlectext) != "" || \
            $RPT(subtitlertext) != "" || [lindex $RPT(pagenoloc) 0] == "T"} {
        rptSetStyle pagesubheader
        set RPT(headerlines) [expr $RPT(headerlines) - ($RPT(size)+$RPT(gap))]
    }
    if {$RPT(titlebarf)} {
        set RPT(headerlines) [expr $RPT(headerlines) - $RPT(titlebargap)]
    }
    set RPT(footerlines) $RPT(bmargin)
    if {$RPT(footerltext) != "" || $RPT(footerctext) != "" || \
            $RPT(footerrtext) != "" || [lindex $RPT(pagenoloc) 0] == "B"} {
        rptSetStyle pagefooter
        set RPT(footerlines) [expr $RPT(footerlines) + $RPT(size) + $RPT(gap)]
    }
    if {$RPT(footerbarf)} {
        set RPT(footerlines) [expr $RPT(footerlines) + $RPT(footerbargap)]
    }
    # -- create the first page
    rptuPSpagedecorate
}        

#----------------------------
# rptuPSsetup - Print Postscript document structuring comments
proc rptuPSsetup {} {
    global RPT
    puts $RPT(fd) "%!PS-Adobe-3.0 EPSF-3.0"
    puts $RPT(fd) "%%Creator:  TclVSrpt 1.0:[lindex [exec /bin/who am i] 0]@[lindex [exec /bin/uname -a] 1]"
    puts $RPT(fd) "%%CreationDate: [exec /bin/date]"
    puts $RPT(fd) "%%DocumentData: Clean7Bit"
    if {$RPT(landscapef)} {
        puts $RPT(fd) "%%Orientation: Landscape"
    } else {
        puts $RPT(fd) "%%Orientation: Portrait"
    }
    puts $RPT(fd) "%%PageOrder: Ascend"
    puts $RPT(fd) "%%Title: TclVSrpt Document"
    puts $RPT(fd) "%%BoundingBox: 0 0 $RPT(pw) $RPT(ph)"
    puts $RPT(fd) "%%Pages: (atend)"
    puts $RPT(fd) "%%DocumentFonts: (atend)"
    puts $RPT(fd) "%%EndComments"
}
#----------------------------
# rptuPSstrwid - Return the width of a string in current font and size
proc rptuPSstrwid {text} {
    global RPT
    set w 0.0
    set f $RPT(font)
    set s $RPT(size)
    foreach c [split $text ""] {
        if {$c < " " || $c > "~"} {
            scan $c %c c
            set c [format %02x $c]
        }
        set w [expr $w + ($RPT($f,$c) * $s)]
    }
    return $w
}

#----------------------------
# rptuPStext - add Postscript text to line at current position
proc rptuPStext {xpos just text} {
    global RPT
    # -- capture desired X position or default to current X position
    set cpos $RPT(cpos)
    if {$xpos == -1} {
        set just $RPT(justify)
    } else {
        set cpos $xpos
    }
    # -- return if given an empty string
    set slen [string length "$text"]
    if {$slen == 0} {return}
    set strwid [rptuPSstrwid "$text"]
    if {$strwid <= 0.0} {return}
    # -- figure out some width constants for this font
    if {$RPT(wrapchar) != ""} {
        set wlinwid [expr $RPT(linwid) - $RPT(wrapcharwid)]
    } else {
        set wlinwid $RPT(linwid)
    }
    set wrappedtext ""
    set RPT(wrappingf) 0
    # -- adjust X position based on justification of text string
    switch -exact -- $just {
        C {set cpos [expr $cpos - ($strwid/2.0)]}
        R {set cpos [expr $cpos - $strwid]}
    }
    # -- adjust starting position if outside of margins
    if {$cpos < 0.0} {set cpos 0}
    if {$cpos > $wlinwid} {
        if {$RPT(wrapf)} {
            # -- drain off leading blanks, rptNewLine to flush, try again
            set start 0
            set last [expr [string length "$text"] - 1]
            while {[string index "$text" $start] == " "} {incr start}
            if {$start > 0} {set text "[string range "$text" $start $last]"}
            if {"$text" != ""} {
                set RPT(wrappingf) 1
                rptNewLine 0
                rptAddText "$text"
            }
        }
        return
    }
    # -- if starting position has changed, create a "move absolute" packet
    if {$cpos != $RPT(cpos)} {
        # -- flush text so far
        if {$RPT(textline) != ""} {
            lappend RPT(tlist) [list 2 "$RPT(textline)"]
            lappend RPT(tlist) [list 1 $RPT(strwid)]
            incr RPT(tlistlen) 2
            set RPT(textline) ""
            set RPT(blankcount) 0
            set RPT(blankincr) 0
            set RPT(textf) 1
        }
        set RPT(cpos) $cpos
        set RPT(strwid) $RPT(cpos)
        lappend RPT(tlist) [list 1 $RPT(cpos)]
        set RPT(lasttabposused) $RPT(tlistlen)
        incr RPT(tlistlen)
    }
    # -- setup for walking through the text character by character
    set charlist [split "$text" ""]
    set si 0
    set i -1
    set strwid $RPT(strwid)
    set overf 0
    set pc [lindex $charlist 0]
    set RPT(blankincr) 0
    # -- split the line into a list characters
    foreach c $charlist {
        # -- get width of character
        if {$c < " " || $c > "~"} {
            scan $c %c fc
            set fc [format %02x $fc]
            set cw [expr $RPT($RPT(font),$fc) * $RPT(size)]
        } else {
            set cw [expr $RPT($RPT(font),$c) * $RPT(size)]
        }
        # -- break if adding character to line exceeds line width
        if {[expr $strwid + $cw] > $wlinwid} {
            set overf 1
            break
        }
        # -- process character
        if {$c == " " && $pc != " "} {
            # -- at transition from word to blank(s)
            set RPT(atlinestartf) 0
            set RPT(strwid) $strwid
            set si [expr $i + 1]
        } elseif {$c != " " && $pc == " "} {
            # -- at transition blank(s) to word
            incr RPT(blankcount) [expr 1 + ($i-$si)]
            set RPT(strwid) $strwid
            set si [expr $i + 1]
        }
        # -- accumulate current character
        set strwid [expr $strwid + $cw]
        set pc $c
        incr i
    }
    # -- add last packet from string
    if {!$overf} {
        if {$c == " "} {
            # -- create a blank finalization
            incr RPT(blankcount) [expr 1 + ($i-$si)]
        } else {
            # -- create a word finalization
            set RPT(atlinestartf) 0
        }
        set RPT(strwid) $strwid
        incr i
        set si $i
    }
    # -- split and add the word that fills the entire line and then some
    if {$overf && $RPT(blankcount) == 0} {
        set line "$RPT(textline)"
        set RPT(textline) ""
        set strlen 0
        set j -1
        set lc [expr [string length "$line"] - 1]
        set pc [string index "$line" 0]
        foreach c [split "$line" ""] {
            if {$c < " " || $c > "~"} {
                scan $c %c fc
                set fc [format %02x $fc]
                set strwid [expr ($RPT($RPT(font),$fc) * $RPT(size))]
            } else {
                set strwid [expr ($RPT($RPT(font),$c) * $RPT(size))]
            }
            if {[expr $RPT(strwid)+$strwid] < $wlinwid} {
                incr j
                set RPT(strwid) [expr $RPT(strwid) + $strwid]
            } else {
                incr j
                set i $j
                set si $i
                break
            }
        }
    }
    set RPT(cpos) $RPT(strwid)
    if {$si > 0} {
        set RPT(textline) "$RPT(textline)[string range "$text" 0 [expr $si - 1]]"
    }
    if {!$RPT(wrapf)} {return}
    if {!$overf} {
        if {"$RPT(textline)" == ""} {return}
        # -- if the last character in textline was not a blank, add one
        if {[string index "$RPT(textline)" [expr [string length $RPT(textline)] - 1]] != " "} {
            if {[expr $RPT(strwid) + $RPT(blankwidth)] > $wlinwid} {
                set overf 1
            } else {
                set RPT(textline) "$RPT(textline) "
                set RPT(strwid) [expr $RPT(strwid) + $RPT(blankwidth)]
                incr RPT(blankcount)
                set RPT(cpos) $RPT(strwid)
                return
            }
        } else {
            return
        }
    }
    set RPT(wrappingf) 1
    # -- add and process any wrap indicator characters
    if {$RPT(wrapchar) != ""} {
        foreach c [split $RPT(wrapchar) ""] {
            if {$c == " "} {
                incr RPT(blankcount)
                set RPT(strwid) [expr $RPT(strwid) + $RPT(blankwidth)]
            } else {
                if {$c < " " || $c > "~"} {
                    scan $c %c fc
                    set fc [format %02x $fc]
                    set RPT(strwid) [expr $RPT(strwid) + ($RPT($RPT(font),$fc) * $RPT(size))]
                } else {
                    set RPT(strwid) [expr $RPT(strwid) + ($RPT($RPT(font),$c) * $RPT(size))]
                }
            }
        }
        set RPT(cpos) $RPT(strwid)
        set RPT(textline) "$RPT(textline)$RPT(wrapchar)"
    }
    # -- get rid of leading blanks in wrapped text
    while {$si < $slen && [string index $text $si] == " "} {incr si}
    incr slen -1
    set wrappedtext "[string range "$text" $si $slen]"
    if {$just == "J"} {
        # -- if there are trailing blanks to this line, get rid of them
        set jj [expr [string length "$RPT(textline)"] - 1]
        set j $jj
        while {[string index $RPT(textline) $j] == " "} {
            incr j -1
            incr RPT(blankcount) -1
            set RPT(strwid) [expr $RPT(strwid) - $RPT(blankwidth)]
            set RPT(cpos) $RPT(strwid)
        }
        if {$j != $jj} {
            set RPT(textline) "[string range "$RPT(textline)" 0 $j]"
        }
        if {$RPT(textline) != ""} {
            set RPT(blankincr) 0
            if {$RPT(blankcount) > 0} {
                set RPT(blankincr) [expr \
                        ($RPT(linwid)-$RPT(strwid))/(1.0*$RPT(blankcount))]
            }
            lappend RPT(tlist) [list 2 "$RPT(textline)"]
            lappend RPT(tlist) [list 1 $RPT(strwid)]
            incr RPT(tlistlen) 2
            set RPT(textline) ""
            set RPT(cpos) $RPT(linwid)
            set RPT(strwid) $RPT(linwid)
            set RPT(textf) 1
        }
    }
    # -- flush the line and add wrapped text
    rptNewLine 0
    if {$RPT(wrapindent) != 0} {
        lappend RPT(tlist) [list 1 $RPT(wrapindent)]
        set RPT(lasttabposused) $RPT(tlistlen)
        incr RPT(tlistlen)
        set RPT(strwid) $RPT(wrapindent)
        set RPT(cpos) $RPT(wrapindent)
    }
    rptAddText "$wrappedtext"
}

#----------------------------
# rptuPStrailer - Print Postscript end of document structuring comments
proc rptuPStrailer {} {
    global RPT
    puts $RPT(fd) "restore"
    puts $RPT(fd) "%%Trailer"
    puts $RPT(fd) "%%EndPages: [expr $RPT(pageno) - 1]"
    puts $RPT(fd) "%%DocumentFonts: [lindex $RPT(fontsloaded) 0]"
    for {set i 1} {$i < [llength $RPT(fontsloaded)]} {incr i} {
        puts $RPT(fd) "%%+ [lindex $RPT(fontsloaded) $i]"
    }
}

#-----------------------------
# rptuSetDefaults - set report defaults
proc rptuSetDefaults {args} {
    global RPT
    set RPT(atlinestartf) 1;            # flag if at line start
    set RPT(blankcount) 0;              # number of spaces on current line
    set RPT(blankincr) 0;               # increment to add to each blank
    set RPT(blankline) " ";             # blank line for ASCII newlines
    set RPT(blankwidth) 0;              # width of space character
    set RPT(bmargin) 0;                 # bottom page margin in points
    set RPT(bmdead) 0;                  # unprintable bottom margin
    set RPT(cpos) 0;                    # current output index (ASCII)
    set RPT(curtab) -1;                 # applicatable tab index
    set RPT(cy) 0;                      # postscript current y position
    set RPT(epos) 0;                    # last output index (ASCII)
    set RPT(fd) "";                     # working file file descriptor
    set RPT(font) "Courier";            # current body text font
    set RPT(fontsloaded) "";            # list of Postscript fonts used
    set RPT(footerbarf) 0;              # draw horiz line above footer
    set RPT(footerbargap) 6;            # leading above footer bar
    set RPT(footerbarindent) 0;         # margin inset for footer bar
    set RPT(footerbarthick) .1;         # thickness of footer bar
    set RPT(footerboldf) 0;             # footer text bold flag
    set RPT(footerctext) "";            # footer centered text
    set RPT(footerfont) "Times";        # footer text font
    set RPT(footergap) 6;               # leading above footer text
    set RPT(footeritalicf) 1;           # footer text italic flag
    set RPT(footerlines) 0;             # num of ASCII lines for footer
    set RPT(footerltext) "";            # footer left-aligned text
    set RPT(footerrtext) "";            # footer right-aligned text
    set RPT(footersize) 8;              # footer font point size
    set RPT(gap) 2;                     # leading between body lines
    set RPT(headerlines) 0;             # num of ASCII lines for header
    set RPT(indent) 0;                  # points to indent from left margin
    set RPT(initializedf) 0;            # flag if TclVSrpt initialized
    set RPT(justify) "L";               # current text justification type
    set RPT(landscapef) 0;              # flag if page in landscape mode
    set RPT(lastbodylineno) 0;          # last body text line number
    set RPT(lastcpos) -1;               # last postscript output x position
    set RPT(lasttabposused) 0;          # last tab position used in this line
    set RPT(leftedge) 0;                # x position of postscript left extreme
    set RPT(lineindent) 0;              # left for initial line of style
    set RPT(lineno) 1;                  # current line number on page
    set RPT(lineprefix) "";             # chars to add to beginning of line
    set RPT(linwid) 0;                  # width of ASCII/Postscript text line
    set RPT(lmargin) 0;                 # left page margin in points
    set RPT(lmdead) 0;                  # unprintable left margin
    set RPT(lpp) 0;                     # estimate of number of lines on a page
    set RPT(pagenof) 0;                 # flag if to print page numbers
    set RPT(pagedeff) 0;                # flag if page decorations defined
    set RPT(pageno) 1;                  # current page number (actual)
    set RPT(pagenocallback) "";         # callback proc with each page out
    set RPT(pagenoloc) "B 0 C";         # page number location (bottom center)
    set RPT(pagenopre) "";              # page number prefix string
    set RPT(pagenostart) 1;             # starting page number
    set RPT(pagenostyle) "N";           # page number presentation style
    set RPT(pagenosuf) "";              # page number suffix string
    set RPT(paperdeff) 0;               # flag if paper geometry defined
    set RPT(ph) 0;                      # page height in points
    set RPT(postscriptf) 0;             # flag if Postscript report type
    set RPT(ppc) 7.2;                   # points per character (width)
    set RPT(ppl) 12;                    # points per line (height)
    set RPT(psfont) "";                 # current output font name
    set RPT(pssize) 0;                  # current output font size
                                        # Postscript font name elements
    set RPT(psfonts) "Courier Times Helvetica Helvetica-Narrow Symbol AvantGarde NewCenturySchlbk Bookman Palatino ZapfChancery-MediumItalic ZapfDingbats"
    set RPT(psfontsnormal) {"" -Roman "" "" "" -Book -Roman -Light -Roman "" ""}
    set RPT(psfontslight) {- - - - "" -Book - -Light - "" ""}
    set RPT(psfontsbold)  {-Bold -Bold -Bold -Bold "" -Demi -Bold -Demi -Bold "" ""}
    set RPT(psfontsangle) {Oblique Italic Oblique Oblique "" Oblique Italic Italic Italic "" ""}
    set RPT(psuflag) 0;                 # current postscript underline flag
    set RPT(psxloc) 0;                  # x location for postscript text
    set RPT(psyloc) 0;                  # y location for postscript text
    set RPT(pw) 0;                      # page height in points
    set RPT(rightedge) 0;                 # x position of right extreme
    set RPT(rindent) 0;                 # right margin inset
    set RPT(rmargin) 0;                 # right margin offset from page edge
    set RPT(rmdead) 0;                  # unprintable right margin width
    set RPT(size) 10;                   # current body font point size
    set RPT(spos) 0;                    # 1st char index in ASCII line
    set RPT(strwid) 0;                  # current postscript X location
    set RPT(stylecurrent) "default";    # current body style name
    set RPT(style-default) "Courier 10 2 0 L 0 0 0 0 {} 0 {} A {} {}"
                                        # default initial style definition
    set RPT(subtitleboldf) 0;           # flag if subtitle is bold
    set RPT(subtitlectext) "";          # centered subtitle text string
    set RPT(subtitlefont) "Times";      # subtitle font
    set RPT(subtitlegap) 2;             # leading below subtitle
    set RPT(subtitleitalicf) 0;         # flag if subtitle is italic
    set RPT(subtitleltext) "";          # left-aligned subtitle text
    set RPT(subtitlertext) "";          # right-aligned subtitle text
    set RPT(subtitlesize) 10;           # subtitle font point size
    set RPT(tabs) "";                   # list of tab stops
    set RPT(tabstyle) "A";              # set tab style to absolute
    set RPT(tabtypes) "";               # list of tab stop justification types
    set RPT(tempfile) "/tmp/xxx";       # working file name (this isn't it)
    set RPT(textf) 0;                   # flag if text in postscript list
    set RPT(textline) "";               # current ASCII text line
    set RPT(titlebarf) 0;               # flag to draw horiz. line below title
    set RPT(titlebargap) 2;             # leading below title bar
    set RPT(titlebarindent) 0;          # margin inset for title bar
    set RPT(titlebarthick) .1;          # thickness of title bar
    set RPT(titleboldf) 1;              # flag if title is bold
    set RPT(titlefont) "Times";         # title font
    set RPT(titlegap) 2;                # leading below title
    set RPT(titleitalicf) 0;            # flag if title is italic
    set RPT(titlesize) 14;              # title font point size
    set RPT(titletext) "";              # centered text for title
    set RPT(tlist) "";                  # postscript instruction buffer
    set RPT(tlistlen) 0;                # length of instruction buffer
    set RPT(tmargin) 0;                 # top margin offset from page edge
    set RPT(tmdead) 0;                  # unprintable top margin width
    set RPT(tunderf) 0;                 # flag that ASCII line has underlines
    set RPT(txtnocrs) 0;                # number of <CR>'s in ASCII text
    set RPT(uflag) 0;                   # postscript underline flag
    set RPT(underlinef) 0;              # flag to underline subsequent chars
    set RPT(utextline) "";              # buffer for ASCII underline chars
    set RPT(wrapchar) "";               # char printed at end of wrapped line
    set RPT(wrapcharwid) 0;             # width of wrapped line suffix
    set RPT(wrapf) 0;                   # flag if auto line-wrap enabled
    set RPT(wrapindent) 0;              # left indentation for wrapped lines
    set RPT(wrappingf) 0;               # current line result of wrapped text
}

#----------------------------
# rptuTabSort -- sort tab list into ascending order
proc rptuTabSort {a b} {
    set retval [expr [lindex $a 0] - [lindex $b 0]]
    if {$retval != 0} {
        if {$retval < 0} {
            return -1
        } else { return 1}
    }
    return [string compare [lindex $a 1] [lindex $b 1]]
}

#----------------------------
# rptuTXTfooterOut -- write page footer to ASCII report
proc rptuTXTfooterOut {} {
    global RPT
    # -- should not happen
    if {$RPT(footerlines) == 0} {return}
    # -- set page footer style
    rptSetStyle pagefooter
    # -- format page number
    if {[lindex $RPT(pagenoloc) 0] == "B" && $RPT(pagenof)} {
        set pval [expr $RPT(pageno) + ($RPT(pagenostart) - 1)]
        switch -exact -- $RPT(pagenostyle) {
            N {}
            R {set pval [rptuNum2Rom $pval]}
            C {set pval [rptuNum2Rom $pval C]}
        }
        set pval "$RPT(pagenopre)$pval$RPT(pagenosuf)"
        if {[lindex $RPT(pagenoloc) 1]} {
            set l "$RPT(footerltext)"
            set r "$RPT(footerrtext)"
        }
        switch -exact -- [lindex $RPT(pagenoloc) 2] {
            L {set RPT(footerltext) "$pval"}
            C {set RPT(footerctext) "$pval"}
            R {set RPT(footerrtext) "$pval"}
        }
        if {[lindex $RPT(pagenoloc) 1]} {
            switch -exact -- [lindex $RPT(pagenoloc) 2] {
                L {set RPT(footerrtext) "$l"; set RPT(pagenoloc) "B 1 R"}
                C {}
                R {set RPT(footerltext) "$r"; set RPT(pagenoloc) "B 1 L"}
            }
        }
    }
    # -- reset line buffers
    set RPT(textline) $RPT(blankline)
    set RPT(utextline) $RPT(blankline)
    set RPT(spos) \
            [expr int($RPT(lmargin)/$RPT(ppc))]
    set RPT(epos) \
            [expr int(($RPT(pw)-$RPT(rmargin))/$RPT(ppc))]
    # -- figure how many blank lines are needed to get to 1st footer line
    set blines [expr int($RPT(lastbodylineno) - $RPT(lineno))]
    for {set i 0} {$i< $blines} {incr i} {puts $RPT(fd) ""}
    incr RPT(lineno) $blines
    # -- write out footer bar leading and footer bar
    if {$RPT(footerbarf)} {
        set blines [expr int($RPT(footerbargap)/$RPT(ppl))]
        for {set i 0} {$i< $blines} {incr i} {puts $RPT(fd) ""}
        incr RPT(lineno) $blines
        set barwidth [expr int((($RPT(pw)-($RPT(lmargin)+$RPT(rmargin))) - \
                (2.0*$RPT(footerbarindent))) / $RPT(ppc))-1]
        set bar "_________________________"
        set barline ""
        while {[string length $barline] < $barwidth} {
            set barline "${barline}${bar}"
        }
        set barline [string range $barline 0 $barwidth]
        set cpos [expr $RPT(spos) + ((($RPT(epos)-$RPT(spos))-$barwidth)/2)]
        for {set i 1} {$i<$cpos} {incr i} {puts -nonewline $RPT(fd) " "}
        puts $RPT(fd) $barline
        incr RPT(lineno) 
    }
    # -- write out footer text and any associated leading
    set blines [expr int([lindex $RPT(style-pagefooter) 2]/$RPT(ppl))]
    for {set i 0} {$i< $blines} {incr i} {puts $RPT(fd) ""}
    incr RPT(lineno) $blines
    set ftflag 0
    set pretext ""
    set posttext ""
    set text ""
    set line $RPT(blankline)
    set lastchar [expr [string length $line] - 1]
    if {$RPT(footerltext) != ""} {
        set cpos $RPT(spos)
        set len [string length $RPT(footerltext)]
        set pretext [string range $line 0 [expr $cpos - 1]]
        incr cpos $len
        set posttext [string range $line $cpos $lastchar]
        set line "${pretext}$RPT(footerltext)${posttext}"
        set ftflag 1
    }
    if {$RPT(footerctext) != ""} {
        set len [string length $RPT(footerctext)]
        set cpos [expr $RPT(spos) + ((($RPT(epos)-$RPT(spos))-$len)/2)]
        set pretext [string range $line 0 [expr $cpos - 1]]
        incr cpos $len
        set posttext [string range $line $cpos $lastchar]
        set line "${pretext}$RPT(footerctext)${posttext}"
        set ftflag 1
    }
    if {$RPT(footerrtext) != ""} {
        set len [string length $RPT(footerrtext)]
        set cpos [expr ($RPT(epos)-$len)]
        set pretext [string range $line 0 [expr $cpos - 1]]
        set line "${pretext}$RPT(footerrtext)"
        set ftflag 1
    }
    if {$ftflag} {
        puts $RPT(fd) "$line"
        incr RPT(lineno)
        if {$RPT(lineno) < [expr $RPT(ph)/$RPT(ppl)]} {
            puts -nonewline $RPT(fd) "\f"
        }
    }
    # -- set up for the next page
    set RPT(lineno) 1
    set RPT(textline) $RPT(blankline)
    set RPT(utextline) $RPT(blankline)
    if {$RPT(wrappingf)} {
        set RPT(spos) [expr int(($RPT(lmargin)+$RPT(wrapindent)+$RPT(indent))/$RPT(ppc))]
        set RPT(wrappingf) 0
    } else {
        set RPT(spos) [expr int(($RPT(lmargin)+$RPT(indent))/$RPT(ppc))]
    }
    set RPT(epos) [expr int(($RPT(pw)-($RPT(rmargin)+$RPT(rindent)))/$RPT(ppc))]
    switch -exact -- $RPT(justify) {
        L {set RPT(cpos) $RPT(spos)}
        J {set RPT(cpos) $RPT(spos)}
        C {set RPT(cpos) [expr $RPT(spos)+int(($RPT(epos)-$RPT(spos))/2.0)]}
        R {set RPT(cpos) $RPT(epos)}
    }
    set RPT(lasttabposused) $RPT(cpos)
}


#----------------------------
# rptuTXTheaderOut -- write page header to ASCII report
proc rptuTXTheaderOut {} {
    global RPT
    # -- newline down to the top margin line
    set RPT(lineno) 1
    set RPT(txtnocrs) 0
    rptNewLine [expr int($RPT(tmargin)/$RPT(ppl))]
    # -- reset text buffers
    set linejust $RPT(justify)
    set RPT(textline) $RPT(blankline)
    set RPT(utextline) $RPT(blankline)
    set RPT(spos) \
        [expr int($RPT(lmargin)/$RPT(ppc))]
    set RPT(epos) \
        [expr int(($RPT(pw)-$RPT(rmargin))/$RPT(ppc))]
    set RPT(cpos) $RPT(spos)
    if {$RPT(headerlines) == 0} {return}
    rptSetStyle pageheader
    set RPT(txtnocrs) 0
    # -- format page number
    if {[lindex $RPT(pagenoloc) 0] == "T" && $RPT(pagenof)} {
        set pval [expr $RPT(pageno) + ($RPT(pagenostart) - 1)]
        switch -exact -- $RPT(pagenostyle) {
            N {}
            R {set pval [rptuNum2Rom $pval]}
            C {set pval [rptuNum2Rom $pval C]}
        }
        set pval "$RPT(pagenopre)$pval$RPT(pagenosuf)"
        if {[lindex $RPT(pagenoloc) 1]} {
            set l "$RPT(subtitleltext)"
            set r "$RPT(subtitlertext)"
        }
        switch -exact -- [lindex $RPT(pagenoloc) 2] {
            L {set RPT(subtitleltext) "$pval"}
            C {set RPT(subtitlectext) "$pval"}
            R {set RPT(subtitlertext) "$pval"}
        }
        if {[lindex $RPT(pagenoloc) 1]} {
            switch -exact -- [lindex $RPT(pagenoloc) 2] {
                L {set RPT(subtitlertext) "$l"; set RPT(pagenoloc) "T 1 R"}
                C {}
                R {set RPT(subtitleltext) "$r"; set RPT(pagenoloc) "T 1 L"}
            }
        }
    }
    # -- handle title line and any leading to next line
    set RPT(cpos) [expr int($RPT(spos)+(($RPT(epos)-$RPT(spos))/2.0))]
    set RPT(justify) C
    rptAddLine "$RPT(titletext)"
    set RPT(cpos) $RPT(spos)
    # -- add new lines for leading between title and subtitle
    rptNewLine [expr int([lindex $RPT(style-pageheader) 2]/$RPT(ppl))]
    # -- handle subtitle line 
    set pretext ""
    set posttext ""
    set text ""
    set line $RPT(blankline)
    set lastchar [expr [string length $line] - 1]
    rptSetStyle pagesubheader
    set RPT(txtnocrs) 0
    set stflag 0
    if {$RPT(subtitleltext) != ""} {
        set cpos $RPT(spos)
        set len [string length $RPT(subtitleltext)]
        set pretext [string range $line 0 [expr $cpos - 1]]
        incr cpos $len
        set posttext [string range $line $cpos $lastchar]
        set line "${pretext}$RPT(subtitleltext)${posttext}"
        set stflag 1
    }
    if {$RPT(subtitlectext) != ""} {
        set len [string length $RPT(subtitlectext)]
        set cpos [expr $RPT(spos) + ((($RPT(epos)-$RPT(spos))-$len)/2)]
        set pretext [string range $line 0 [expr $cpos - 1]]
        incr cpos $len
        set posttext [string range $line $cpos $lastchar]
        set line "${pretext}$RPT(subtitlectext)${posttext}"
        set stflag 1
    }
    if {$RPT(subtitlertext) != ""} {
        set len [string length $RPT(subtitlertext)]
        set cpos [expr ($RPT(epos)-$len)]
        set pretext [string range $line 0 [expr $cpos - 1]]
        set line "${pretext}$RPT(subtitlertext)"
        set stflag 1
    }
    if {$stflag} {
        set RPT(textline) $line
        set RPT(cpos) $RPT(spos)
        set RPT(atlinestartf) 0
        rptNewLine 0
    }
    # -- add newlines for subtitle to title bar leading
    rptNewLine [expr int([lindex $RPT(style-pagesubheader) 2]/$RPT(ppl))]
    # -- handle any title bar and leading to first line of body
    if {$RPT(titlebarf)} {
        set barwidth [expr int((($RPT(pw)-($RPT(lmargin)+$RPT(rmargin))) - \
                (2.0*$RPT(titlebarindent))) / $RPT(ppc))-1]
        set bar "===================="
        set barline ""
        while {[string length $barline] < $barwidth} {
            set barline "${barline}${bar}"
        }
        set barline [string range $barline 0 $barwidth]
        set RPT(justify) C
        set RPT(cpos) [expr int($RPT(spos)+(($RPT(epos)-$RPT(spos))/2.0))]
        rptAddLine "$barline"
        rptNewLine [expr int($RPT(titlebargap)/$RPT(ppl))]
    }
    set RPT(textline) $RPT(blankline)
    set RPT(utextline) $RPT(blankline)
    set RPT(tunderf) 0
    if {$RPT(wrappingf)} {
        set RPT(spos) [expr int(($RPT(lmargin)+$RPT(wrapindent)+$RPT(indent))/$RPT(ppc))]
        set RPT(wrappingf) 0
    } else {
        set RPT(spos) [expr int(($RPT(lmargin)+$RPT(indent))/$RPT(ppc))]
    }
    set RPT(epos) [expr int(($RPT(pw)-($RPT(rmargin)+$RPT(rindent)))/$RPT(ppc))]
    switch -exact -- $RPT(justify) {
        L {set RPT(cpos) $RPT(spos)}
        J {set RPT(cpos) $RPT(spos)}
        C {set RPT(cpos) [expr $RPT(spos)+int(($RPT(epos)-$RPT(spos))/2.0)]}
        R {set RPT(cpos) $RPT(epos)}
    }
    set RPT(lasttabposused) $RPT(cpos)
    set RPT(atlinestartf) 1
    set RPT(curtab) -1
}

#----------------------------
# rptuTXTsetup -- set ASCII report specfic elements
proc rptuTXTsetup {} {
    global RPT
    # -- calculate lpp if not already provided
    if {$RPT(lpp) == 0} {
        set RPT(lpp) [expr int(($RPT(ph)-$RPT(bmargin))/$RPT(ppl))]
    }
    # -- calculate number of lines consumed by page header
    set RPT(headerlines) 0
    if {$RPT(titletext) != ""} {
        incr RPT(headerlines) [expr 1 + int([lindex $RPT(style-pageheader) 2]/(1.0*$RPT(ppl)))]
    }
    if {[lindex $RPT(pagenoloc) 0] == "T" && $RPT(pagenof)} {
        switch -exact [lindex $RPT(pagenoloc) 2] -- {
            L {set RPT(subtitleltext) "1"}
            C {set RPT(subtitlectext) "1"}
            R {set RPT(subtitlertext) "1"}
        }
    }
    if {$RPT(subtitleltext) != "" || $RPT(subtitlectext) != "" || \
            $RPT(subtitlertext) != ""} {
        incr RPT(headerlines) [expr 1 + int([lindex $RPT(style-pagesubheader) 2]/(1.0*$RPT(ppl)))]
    }
    if {$RPT(titlebarf)} {
        incr RPT(headerlines) [expr 1 + int($RPT(titlebargap)/(1.0*$RPT(ppl)))]
    }
    # -- calculate number of lines consumed by page footer
    set RPT(footerlines) 0
    if {[lindex $RPT(pagenoloc) 0] == "B" && $RPT(pagenof)} {
        switch -exact [lindex $RPT(pagenoloc) 2] -- {
            L {set RPT(footerltext) "1"}
            C {set RPT(footerctext) "1"}
            R {set RPT(footerrtext) "1"}
        }
    }
    if {$RPT(footerltext) != "" || $RPT(footerctext) != "" || \
            $RPT(footerrtext) != ""} {
        incr RPT(footerlines) [expr 1 + int([lindex $RPT(style-pagefooter) 2]/(1.0*$RPT(ppl)))]
    }
    if {$RPT(footerbarf)} {
        incr RPT(footerlines) [expr 1 + int($RPT(footerbargap)/(1.0*$RPT(ppl)))]
    }
    # -- build empty text buffer for ASCII reports
    set RPT(linwid) \
            [expr int(($RPT(pw)-$RPT(rmargin))/$RPT(ppc)) - 1]
    set RPT(blankline) "          "
    set loopend [expr $RPT(linwid) / 10]
    for {set i 0} {$i<$loopend} {incr i} {
        set RPT(blankline) "$RPT(blankline)          "
    }
    set RPT(blankline) [string range $RPT(blankline) 0 \
            [expr $RPT(linwid) - 1]]
    set RPT(textline) $RPT(blankline)
    set RPT(utextline) $RPT(blankline)
    # -- calculate indexes into output buffer for margins and current position
    if {$RPT(wrappingf)} {
        set RPT(spos) [expr int(($RPT(lmargin)+$RPT(wrapindent)+$RPT(indent))/$RPT(ppc))]
        set RPT(wrappingf) 0
    } else {
        set RPT(spos) [expr int(($RPT(lmargin)+$RPT(indent))/$RPT(ppc))]
    }
    set RPT(cpos) $RPT(spos)
    set RPT(lasttabposused) $RPT(cpos)
    set RPT(epos) \
            [expr int(($RPT(pw)-($RPT(rmargin)+$RPT(rindent)))/$RPT(ppl))]
    set RPT(tunderf) 0
    set RPT(atlinestartf) 1
    set RPT(curtab) -1
    set RPT(lastbodylineno) [expr 1 + int(($RPT(ph)/$RPT(ppl)) - \
            (($RPT(bmargin)/$RPT(ppl))+$RPT(footerlines)))]
}

#----------------------------
# rptuTXTtext -- add text to ASCII report at current position
proc rptuTXTtext {text} {
    global RPT
    set padstring "        "
    set slen [string length $text]
    # -- return on empty string
    if {$slen == 0} {return}
    set RPT(atlinestartf) 0
    set lineWrap $RPT(wrapf)
    # -- if we would be adding text next to text, insert a space
    if {$RPT(cpos) > $RPT(spos)} {
        if {[string index $RPT(textline) [expr $RPT(cpos) - 1]] != " " && \
                [string index $text 0] != " "} {
            set text " $text"
            incr slen
        }
    }
    # -- find justified starting position in buffer, right truncate to fit
    switch -exact -- $RPT(justify) {
        C {
            set RPT(cpos) [expr $RPT(cpos) - int($slen/2.0)]
            if {$RPT(cpos) < $RPT(spos)} {set RPT(cpos) $RPT(spos)}
        }
        R {
            set RPT(cpos) [expr $RPT(cpos) - $slen]
            if {$RPT(cpos) < $RPT(spos)} {set RPT(cpos) $RPT(spos)}
        }
    }
    # -- handle condition that start position beyond end of line
    if {$RPT(cpos) > $RPT(epos)} {
        if {$RPT(wrapf)} {
            # -- rptNewLine 0 will flush current line, recurse for this line
            if {$text != ""} {
                # -- drain off leading blanks in the wrapped text
                set start 0
                set last [expr [string length "$text"] - 1]
                while {"[string index $text $start]" == " "} {incr start}
                if {$start != 0} {set text "[string range $text $start $last]"}
            }
            if {$text != ""} {
                rptNewLine 0
                rptAddText "$text"
            }
        }
        return
    }
    # -- if we're wrapping text and string exceeds right limit,
    # -- capture excess text from current string at either nearest
    # -- word break, or if no word break found, at limit 
    # -- trim off un-wrapped excess line; otherwise, truncate string
    set wrappedtext ""
    set epos [expr $RPT(cpos)+$slen-1]
    if {$RPT(epos) < $epos} {
        if {!$RPT(wrapf)} {
            set endidx [expr $slen - ($epos - $RPT(epos))-1]
            set text [string range $text 0 $endidx]
            set slen [string length $text]
        }
    }
    set epos [expr $RPT(cpos)+($slen-1)]
    # -- put text string into the line buffer at current position
    set pretext ""
    set preutext ""
    set afttext ""
    set aftutext ""
    if {$RPT(cpos) > 0} {
        set pretext "[string range $RPT(textline) 0 [expr $RPT(cpos) - 1]]"
        if {$RPT(underlinef)} {
            set RPT(tunderf) 1
            set preutext "[string range $RPT(utextline) 0 [expr $RPT(cpos) - 1]]"
        }
    }
    if {$RPT(epos) >= $epos} {
        set afttext "[string range $RPT(textline) [expr $epos + 1] $RPT(epos)]"
        if {$RPT(underlinef)} {
            set aftutext "[string range $RPT(utextline) [expr $epos + 1] \
                    $RPT(epos)]"
        }
    }
    # -- rebuild text output buffer
    set RPT(textline) "${pretext}${text}${afttext}"
    # -- if wrapping and line extends beyond right margin, capture extra
    # -- text and adjust current text line appropriately
    if {$RPT(wrapf) && $epos > $RPT(epos)} {
        # -- find the appropriate word break to split at
        set tend [expr $RPT(epos) - $RPT(wrapcharwid)]
        set ttend $tend
        while {$tend >= $RPT(spos)} {
            if {[string index $RPT(textline) $tend] == " "} {
                break
            } else {
                incr tend -1
            }
        }
        if {$tend <= $RPT(spos)} {set tend $RPT(epos)}
        set wrappedtext [string range $RPT(textline) [expr $tend+1] $epos]
        if {$wrappedtext != ""} {
            # -- drain off leading blanks in the wrapped text
            set start 0
            set last [expr [string length "$wrappedtext"] - 1]
            while {"[string index $wrappedtext $start]" == " "} {incr start}
            if {$start != 0} {
                set wrappedtext "[string range $wrappedtext $start $last]"
            }
        }
        # -- drain off trailing blanks in current text line
        while {[string index $RPT(textline) $tend] == " "} {incr tend -1}
        set RPT(textline) [string range $RPT(textline) 0 $tend]
        set epos $tend
        # -- show line wrap character(s)
        if {$RPT(wrapcharwid) > 0} {
            set RPT(textline) \
                    "[string range $RPT(textline) 0 [expr $RPT(epos) - $RPT(wrapcharwid)]]$RPT(wrapchar)"
        }            
    }
   
    # -- handle word underlining (underline characters in separate buffer)
    if {$RPT(underlinef)} {
        set utext ""
        for {set i $RPT(cpos)} {$i<= $epos} {incr i} {
            if {[string index $RPT(textline) $i] != " "} {
                set utext "${utext}_"
            } else {
                set utext "${utext} "
            }
        }
        set RPT(utextline) "${preutext}${utext}${aftutext}"
    }
    # -- if we have left-right justification, see if we need to pad internal
    # -- blanks in text string
    if {$RPT(justify) == "J" && "$wrappedtext" != ""} {
        set endgap [expr (($RPT(epos)-$RPT(wrapcharwid))-$epos)]
        if {$endgap > 0} {
            # -- count internal blank strings in line, from last tab position,
            # -- remember their location
            set bcnt 0
            set bpos ""
            set prevblank -2
            for {set i $RPT(lasttabposused)} {$i <= $epos} {incr i} {
                if {[string index $RPT(textline) $i] == " "} {
                    if {[expr $i - 1] > $prevblank} {
                        incr bcnt
                        lappend bpos $i
                    }
                    set prevblank $i
                }
            }
            # -- don't do it if we have to add over 4 blanks per spot
            set everyblnk 0
            set extrablnk 0
            if {$bcnt > 0} {
                set everyblnk [expr $endgap / $bcnt]
                set extrablnk [expr $endgap % $bcnt]
                if {$everyblnk > 4} {
                    set everyblnk 0
                    set extrablnk 0
                    set bpos ""
                    set bcnt 0
                }
            }
            # -- if we have blanks to pad...
            if {$bcnt>0 && [expr int($endgap/$bcnt)] <= 4} {
                set ttext [string range $RPT(textline) 0 [expr $RPT(spos)-1]]
                set tutext [string range $RPT(textline) 0 [expr $RPT(spos)-1]]
                set spos $RPT(spos)
                foreach padloc $bpos {
                    # -- adjust padding with each word written
                    if {$extrablnk > 0} {
                        set padding [string range $padstring 0 $everyblnk]
                        incr extrablnk -1
                    } else {
                        set padding [string range $padstring 0 [expr $everyblnk-1]]
                    }
                    set word [string range $RPT(textline) $spos $padloc]
                    set ttext "${ttext}${word}${padding}"
                    set word [string range $RPT(utextline) $spos $padloc]
                    set tutext "${tutext}${word}${padding}"
                    set spos [expr $padloc + 1]
                }
                if {$spos < [expr $epos + 1]} {
                    set ttext "${ttext}[string range $RPT(textline) $spos $epos]"
                    set tutext "${tutext}[string range $RPT(utextline) $spos $epos]"
                }
                set RPT(textline) "$ttext$RPT(wrapchar)"
                set RPT(utextline) "$tutext"
                set slen [expr $RPT(epos)+1]
                set epos $RPT(epos)
            }
        }
    }
    # -- update next output position in line buffer
    set RPT(cpos) [expr $epos+1]
    # -- handle any wrapped text
    if {$RPT(wrapf)} {
        if {$wrappedtext != ""} {
            set RPT(wrappingf) 1
            rptNewLine 0
            rptAddText "$wrappedtext"
        } 
    } 
}

#============================
#       END OF FILE
#============================



