/*
 * Copyright (c) 1995 Sun Microsystems, Inc.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL SUN MICROSYSTEMS, INC. BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 * OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF SUN
 * MICROSYSTEMS, INC. HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * SUN MICROSYSTEMS, INC. SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THE SOFTWARE PROVIDED
 * HEREUNDER IS ON AN "AS IS" BASIS, AND SUN MICROSYSTEMS, INC. HAS NO
 * OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
 * MODIFICATIONS.
 */

#include <dlfcn.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/audioio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <thread.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>

/*
 * Macros
 */
#define CLICK_NSECONDS  300000000 /* Double-click timeout (nanoseconds) */
#define PING_NSECONDS 10000000000 /* Ping timeout (nanoseconds) */
#define KC_MUTE 52                /* X keycode for audio mute key */
#define KC_LOWER_VOLUME 9         /* X keycode for audio lower volume */
#define KC_RAISE_VOLUME 11        /* X keycode for audio raise key */

#define MIN(a,b) (a<b?a:b)
#define MAX(a,b) (a>b?a:b)

/*
 * Player structure
 */
typedef struct {
  int magicnum;
  int death_flag;
  int pad1[6];
  int health;
  int shield;
  int pad2[14];
  int frags[4];
  int pad3[11];
  int bullets;
  int shells;
  int plasma;
  int rockets;
  int pad4[8];
  int monsters_killed;
  int pad5[18];
} Player;

Player players[4];


/*
 * DOOM global variables
 */
int netgame, deathmatch, mousebfire, mousebstrafe, key_fire, key_speed, key_up;
int gameepisode, gamemap, gamestate, usemouse;
int consoleplayer, commercial, gametime, leveltime;
char *player_names[4];

/*
 * Frag information
 */
static int fragpipe, audioctl_fd, current_level, current_state, current_death;
static int p0_frags[4], p1_frags[4], p2_frags[4], p3_frags[4];

/*
 * Player names
 */
char *PlayerColors[] = 
{
  "DOOM_GREEN",
  "DOOM_INDIGO",
  "DOOM_BROWN",
  "DOOM_RED",
};
 
/*
 * Local variables
 */
static void *x11_handle = NULL;
static int (*next_event)(Display *, XEvent *) = NULL;

static int keycode_grab, keycode_fire, keycode_altfire, keycode_pause;
static int keycode_caps, keycode_numlock, keycode_speed, keycode_altspeed;
static int keycode_up, keycode_altup, keycode_type, keycode_return;
static int keycode_boss, keycode_noop;

static Window doom_window;
static Colormap doom_colormap = NULL;
static int grabbed = 0;
static int suppress_vertical = 0;
static int center_x, center_y;
static int curx, cury;

/*
 * Utility functions
 */
static
doomkey_to_keycode(Display *display, int keysym, int *keycode, int *altkeycode)
{
  *keycode = 0;
  *altkeycode = 0;

  switch (keysym) {
  case 157: /* Control */
    *keycode = XKeysymToKeycode(display, XK_Control_L);
    *altkeycode = XKeysymToKeycode(display, XK_Control_R);
    break;
  case 172:
  case 173:
  case 174:
  case 175: /* Arrow keys */
    *keycode = XKeysymToKeycode(display, XK_Left + keysym - 172);
    break;
  case 182: /* Shift */
    *keycode = XKeysymToKeycode(display, XK_Shift_L);
    *altkeycode = XKeysymToKeycode(display, XK_Shift_R);
    break;
  case 184: /* Meta */
    *keycode = XKeysymToKeycode(display, XK_Meta_L);
    *altkeycode = XKeysymToKeycode(display, XK_Meta_R);
    break;
  default:
    *keycode = XKeysymToKeycode(display, keysym);
    *altkeycode = *keycode;
    break;
  }
}

static void
init_keycodes(Display *display)
{
  keycode_boss = XKeysymToKeycode(display, XK_L1);
  keycode_noop = XKeysymToKeycode(display, XK_Home);
  keycode_caps = XKeysymToKeycode(display, XK_Caps_Lock);
  keycode_grab = XKeysymToKeycode(display, XK_Scroll_Lock);
  keycode_numlock = XKeysymToKeycode(display, XK_Num_Lock);
  keycode_pause = XKeysymToKeycode(display, XK_Pause);
  keycode_return = XKeysymToKeycode(display, XK_Return);
  keycode_type = XKeysymToKeycode(display, XK_t);

  doomkey_to_keycode(display, key_fire, &keycode_fire, &keycode_altfire);
  doomkey_to_keycode(display, key_speed, &keycode_speed, &keycode_altspeed);
  doomkey_to_keycode(display, key_up, &keycode_up, &keycode_altup);
}

static void *
no_input(void *arg)
{
  pid_t parent = (pid_t)arg;
  struct sigaction oact;

  while (!next_event) {
    sigaction(SIGPOLL, NULL, &oact);
    if (oact.sa_handler)
      kill(parent, SIGPOLL);
    sleep(1);
  }
  return 0;
}

static void
init_players()
{
  int i;
  pid_t parent;
  char buf[BUFSIZ];
  char *name, *fragpipestr;
  sigset_t orig_mask, no_input_mask;

  memset(p0_frags, -1, 16);
  memset(p1_frags, -1, 16);
  memset(p2_frags, -1, 16);
  memset(p3_frags, -1, 16);

  for (i=0; i<4; i++) {
    if (name = getenv(PlayerColors[i])) {
      sprintf(buf, "%s: ", name);
      player_names[i] = strdup(buf);
    }
  }

  fragpipestr = getenv("FRAGPIPE");
  if (fragpipestr)
    fragpipe = atoi(fragpipestr);

  /*
   * Hack to fix the no input problem
   */
  sigfillset(&no_input_mask);
  thr_sigsetmask(SIG_SETMASK, &no_input_mask, &orig_mask);
  thr_create(NULL, 0, no_input, (void *)getpid(),
	     THR_DETACHED | THR_DAEMON, NULL);
  thr_sigsetmask(SIG_SETMASK, &orig_mask, NULL);
}

static void
toggle_grab(Display *display)
{
  if (!grabbed) {
    if (XGrabPointer(display, doom_window, False,
		     ButtonPressMask | ButtonReleaseMask | PointerMotionMask,
		     GrabModeAsync, GrabModeAsync, None, None, CurrentTime)
	!= GrabSuccess) {
      fprintf(stderr, "Pointer Grab failed\n");
    }
    else if (XGrabKeyboard(display, doom_window, False,
			   GrabModeAsync, GrabModeAsync, CurrentTime)
	     != GrabSuccess) {
      fprintf(stderr, "Keyboard Grab failed\n");
      XUngrabPointer(display, CurrentTime);
    }
    else {
      printf("Mouse is grabbed.  Press \"Scroll Lock\" to toggle grab.\n");
      grabbed = 1;
    }
    if (doom_colormap != NULL)
      XInstallColormap(display, doom_colormap);
  }
  else {
    XUngrabPointer(display, CurrentTime);
    XUngrabKeyboard(display, CurrentTime);
    printf("Mouse is ungrabbed.\n");
    grabbed = 0;
  }
}

static void
pause_game(Display *display)
{
  static Cursor watch_cursor = NULL;
  static Cursor null_cursor = NULL;

  XEvent event;
  XSetWindowAttributes attr;
  int was_grabbed;

  /* Don't allow pause in multi-player games */
  if (players[1].magicnum)
    return;

  was_grabbed = grabbed;
  if (grabbed)
    toggle_grab(display);

  printf("Game has been paused\n");

  if (!watch_cursor) {
    Pixmap pix;
    XColor color;
    char data[1];

    color.pixel = 0;
    color.flags = 0;
    *data = 0;
    pix = XCreatePixmapFromBitmapData(display, doom_window,
				      data, 1, 1, 0, 0, 1);
    null_cursor = XCreatePixmapCursor(display, pix, pix,
				      &color, &color, 0, 0);
    watch_cursor = XCreateFontCursor(display, XC_watch);
  }

  attr.cursor = watch_cursor;
  XChangeWindowAttributes(display, doom_window, CWCursor, &attr);

  do {
    (*next_event)(display, &event);
  } while(event.type != KeyPress && event.type != MapNotify);

  if (event.type == KeyPress) {
    /* Discard the KeyRelease event */
    (*next_event)(display, &event);
  }

  attr.cursor = null_cursor;
  XChangeWindowAttributes(display, doom_window, CWCursor, &attr);

  printf("Game has been resumed\n");

  if (was_grabbed)
    toggle_grab(display);
}

static void
audio_config(int keycode)
{
  audio_info_t get_info;
  audio_info_t set_info;
  int status, gain;

  if (!audioctl_fd)
    audioctl_fd = open("/dev/audioctl", O_RDONLY);
  if (audioctl_fd == -1)
    return;

  AUDIO_INITINFO(&set_info);
  while ((status = ioctl(audioctl_fd, AUDIO_GETINFO, &get_info)) == -1 &&
	 errno == EINTR);
  if (status == -1)
    return;

  switch (keycode) {
  case KC_MUTE:
    set_info.output_muted = !get_info.output_muted;
    break;
  case KC_LOWER_VOLUME:
    if (get_info.play.gain == 255)
      gain = get_info.play.gain - 15;
    else
      gain = get_info.play.gain - 16;
    set_info.play.gain = MAX(0, gain);
    break;
  case KC_RAISE_VOLUME:
    gain = get_info.play.gain + 16;
    set_info.play.gain = MIN(AUDIO_MAX_GAIN, gain);
    break;
  }

  while (ioctl(audioctl_fd, AUDIO_SETINFO, &set_info) == -1 &&
	 errno == EINTR);
}

static void
boss_alert(Display *display, int enable)
{
  static int alert = 0;
  audio_info_t set_info;

  if (alert == enable)
    return;

  alert = enable;

  if (!audioctl_fd)
    audioctl_fd = open("/dev/audioctl", O_RDONLY);
  if (audioctl_fd == -1)
    return;

  AUDIO_INITINFO(&set_info);
  set_info.output_muted = enable;
  while (ioctl(audioctl_fd, AUDIO_SETINFO, &set_info) == -1 &&
	 errno == EINTR);

  if (enable && grabbed)
    toggle_grab(display);
  else if (!enable && !grabbed && usemouse)
    toggle_grab(display);

  if (enable) {
    XIconifyWindow(display, doom_window, DefaultScreen(display));
    if (!players[1].magicnum) {
      pause_game(display);
      boss_alert(display, 0);
    }
  }
}

static void
print_frags()
{
  if (memcmp(players[0].frags, p0_frags, 16) ||
      memcmp(players[1].frags, p1_frags, 16) ||
      memcmp(players[2].frags, p2_frags, 16) ||
      memcmp(players[3].frags, p3_frags, 16)) {

    if (fragpipe) {
      char buf[1024];

      sprintf(buf,
	      "{%d %d} "
	      "{%d %d %d %d} "
	      "{%d %d %d %d} "
	      "{%d %d %d %d} "
	      "{%d %d %d %d}\n",
	      gameepisode,
	      gamemap,
	      players[0].frags[0],
	      players[0].frags[1],
	      players[0].frags[2],
	      players[0].frags[3],
	      players[1].frags[0],
	      players[1].frags[1],
	      players[1].frags[2],
	      players[1].frags[3],
	      players[2].frags[0],
	      players[2].frags[1],
	      players[2].frags[2],
	      players[2].frags[3],
	      players[3].frags[0],
	      players[3].frags[1],
	      players[3].frags[2],
	      players[3].frags[3]);

      if (write(fragpipe, buf, strlen(buf)) == -1) {
	perror("fragpipe write");
	fragpipe = 0;
      }
    }

    if (!fragpipe) {
      printf("%12s %12s %12s %12s %12s %12s\n",
	     "",
	     player_names[0],
	     player_names[1],
	     player_names[2],
	     player_names[3],
	     "Total");
      
      printf("%12s %9d %12d %12d %12d   %12d\n",
	     player_names[0],
	     players[0].frags[0],
	     players[0].frags[1],
	     players[0].frags[2],
	     players[0].frags[3],
	     players[0].frags[1] 
	     + players[0].frags[2] 
	     + players[0].frags[3] 
	     - players[0].frags[0]);
      
      printf("%12s %9d %12d %12d %12d   %12d\n",
	     player_names[1],
	     players[1].frags[0],
	     players[1].frags[1],
	     players[1].frags[2],
	     players[1].frags[3],
	     players[1].frags[0] 
	     + players[1].frags[2] 
	     + players[1].frags[3] 
	     - players[1].frags[1]);
      
      printf("%12s %9d %12d %12d %12d   %12d\n",
	     player_names[2],
	     players[2].frags[0],
	     players[2].frags[1],
	     players[2].frags[2],
	     players[2].frags[3],
	     players[2].frags[0] 
	     + players[2].frags[1] 
	     + players[2].frags[3] 
	     - players[2].frags[2]);
      
      printf("%12s %9d %12d %12d %12d   %12d\n",
	     player_names[3],
	     players[3].frags[0],
	     players[3].frags[1],
	     players[3].frags[2],
	     players[3].frags[3],
	     players[3].frags[0] 
	     + players[3].frags[1] 
	     + players[3].frags[2] 
	     - players[3].frags[3]);
      
      printf("\n");
    }
      
    memcpy(p0_frags, players[0].frags, 16);
    memcpy(p1_frags, players[1].frags, 16);
    memcpy(p2_frags, players[2].frags, 16);
    memcpy(p3_frags, players[3].frags, 16);
  }
}

static void
ping()
{
  static hrtime_t pingtime = 0;
  hrtime_t curtime;

  curtime = gethrtime();
  if (curtime - pingtime > PING_NSECONDS) {
    if (write(fragpipe, "ping\n", 5) == -1) {
      perror("fragpipe write");
      fragpipe = 0;
    }
    pingtime = curtime;
  }
}

static void
loggame()
{
  if (fragpipe) {
    if (write(fragpipe, "loggame\n", 8) == -1) {
      perror("fragpipe write");
      fragpipe = 0;
    }
  }
}

/*
 * Overloaded X functions
 */

static void
fetch_x11_handle()
{
  if (x11_handle == NULL) {
    x11_handle = dlopen("libX11.so", RTLD_LAZY);
    if (x11_handle == NULL) {
      fprintf(stderr, "dlopen: %s\n", dlerror());
      exit(1);
    }
  }
}

Window
XCreateWindow(Display *display, Window parent, int x, int y,
	      unsigned int width, unsigned int height,
	      unsigned int bw, int depth, unsigned int class,
	      Visual *visual, unsigned long valuemask,
	      XSetWindowAttributes *attributes)
{
  static Window (*create_window)(Display *, Window, int, int,
			      unsigned int, unsigned int, unsigned int,
			      int, unsigned int, Visual *, unsigned long,
			      XSetWindowAttributes *) = NULL;
  Window w, root;

  if (create_window == NULL) {
    init_keycodes(display);
    init_players();

    fetch_x11_handle();

    create_window = (Window (*)(Display *, Window, int, int,
				unsigned int, unsigned int, unsigned int,
				int, unsigned int, Visual *, unsigned long,
				XSetWindowAttributes *))
      dlsym(x11_handle, "XCreateWindow");

    if (create_window == NULL) {
      fprintf(stderr, "dlsym XCreateWindow: %s\n", dlerror());
      exit(1);
    }
  }

  if (usemouse) {
    attributes->event_mask = attributes->event_mask |
      ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
  }

  if (valuemask & CWColormap)
    doom_colormap = attributes->colormap;

  w = (*create_window)(display, parent, x, y, width, height, bw, depth,
		       class, visual, valuemask, attributes);

  XGetGeometry(display, DefaultRootWindow(display),
	       &root, &x, &y, &width, &height, &bw, &depth);
  center_x = width/2;
  center_y = height/2;
  doom_window = w;

  return w;
}

int
XSelectInput(Display *display, Window w, long event_mask)
{
  static int (*select_input)(Display *, Window, long) = NULL;

  if (select_input == NULL) {
    fetch_x11_handle();

    select_input = (int (*)(Display *, Window, long))
      dlsym(x11_handle, "XSelectInput");

    if (select_input == NULL) {
      fprintf(stderr, "dlsym XSelectInput: %s\n", dlerror());
      exit(1);
    }
  }

  if (usemouse) {
    event_mask = event_mask |
      ButtonPressMask | ButtonReleaseMask | PointerMotionMask |
      StructureNotifyMask;
  }

  return (*select_input)(display, w, event_mask);
}

int
XNextEvent(Display *display, XEvent *event)
{
  static int dograb, checkstate, speedPress, resetSpeed, typing;
  static hrtime_t clicktime;
  hrtime_t curtime;
  int retval;

  if (next_event == NULL) {
    signal(SIGPIPE, SIG_IGN);
    fetch_x11_handle();

    next_event = (int (*)(Display *, XEvent *))dlsym(x11_handle, "XNextEvent");
    if (next_event == NULL) {
      fprintf(stderr, "dlsym XNextEvent: %s\n", dlerror());
      exit(1);
    }

    if (!usemouse)
      dograb = 0;
    else
      dograb = 1;

    checkstate = 1;
    speedPress = 0;
    resetSpeed = 0;
    clicktime = 0;
    typing = 0;
  }

  retval = (*next_event)(display, event);

  if (deathmatch)
    print_frags();

  if (fragpipe)
    ping();

  if (gamemap != current_level) {
    if (speedPress)
      resetSpeed = 1;
    current_level = gamemap;
  }
  if (gamestate != current_state) {
    if (gamestate && !current_state)
      loggame();
    if (!gamestate && current_state && speedPress)
      resetSpeed = 1;
    current_state = gamestate;
  }
  if (!netgame && players[0].death_flag != current_death) {
    if (!players[0].death_flag && current_death && speedPress)
      resetSpeed = 1;
    current_death = players[0].death_flag;
  }

  switch(event->type) {
  case ButtonPress:
    if (dograb) {
      toggle_grab(display);
      dograb = 0;
    }
    if (checkstate) {
      if (event->xbutton.state & LockMask ||
	  event->xbutton.state & Mod3Mask) {
	speedPress = 1;
	resetSpeed = 1;
      }
      checkstate = 0;
    }

    if (usemouse && event->xbutton.button == mousebfire+1) {
      event->type = KeyPress;
      event->xkey.keycode = keycode_fire;
    }
    if (usemouse && event->xbutton.button == mousebstrafe+1) {
      curtime = gethrtime();
      if (curtime - clicktime < CLICK_NSECONDS) {
	suppress_vertical = !suppress_vertical;
	if (suppress_vertical)
	  printf("Vertical mouse motion disabled.\n");
	else
	  printf("Vertical mouse motion enabled.\n");
      }
      clicktime = curtime;
    }
    break;

  case ButtonRelease:
    if (usemouse && event->xbutton.button == mousebfire+1) {
      event->type = KeyRelease;
      event->xkey.keycode = keycode_fire;
    }
    break;

  case KeyPress:
    if (dograb) {
      toggle_grab(display);
      dograb = 0;
    }
    if (checkstate) {
      if (event->xkey.state & LockMask ||
	  event->xkey.state & Mod3Mask) {
	speedPress = 1;
	resetSpeed = 1;
      }
      checkstate = 0;
    }

    if (usemouse && event->xkey.keycode == keycode_grab)
      toggle_grab(display);
    else if (event->xkey.keycode == keycode_pause)
      pause_game(display);
    else if (event->xkey.keycode == keycode_caps ||
	     event->xkey.keycode == keycode_numlock) {
      if (!typing) {
	event->xkey.keycode = keycode_speed;
	speedPress = 1;
	resetSpeed = 0;
      }
      else {
	speedPress = 1;
	resetSpeed = 1;
      }
    }
    else if (event->xkey.keycode == keycode_speed ||
	     event->xkey.keycode == keycode_altspeed) {
      if (!typing && speedPress)
	event->type = KeyRelease;
    }
    else if (event->xkey.keycode == keycode_up ||
	     event->xkey.keycode == keycode_altup) {
      if (!typing && resetSpeed) {
	event->xkey.keycode = keycode_speed;
	resetSpeed = 0;
      }
    }
    else if (event->xkey.keycode == KC_MUTE ||
	     event->xkey.keycode == KC_LOWER_VOLUME ||
	     event->xkey.keycode == KC_RAISE_VOLUME) {
      audio_config(event->xkey.keycode);
    }
    else if (event->xkey.keycode == keycode_type)
      typing = 1;
    else if (event->xkey.keycode == keycode_return)
      typing = 0;
    else if (event->xkey.keycode == keycode_boss) {
      boss_alert(display, 1);
      event->xkey.keycode = keycode_noop;
    }
    break;

  case KeyRelease:
    if (event->xkey.keycode == keycode_caps ||
	event->xkey.keycode == keycode_numlock) {
      event->xkey.keycode = keycode_speed;
      speedPress = 0;
      resetSpeed = 0;
    }
    else if (event->xkey.keycode == keycode_speed ||
	     event->xkey.keycode == keycode_altspeed) {
      if (!typing) {
	if (speedPress)
	  event->type = KeyPress;
	resetSpeed = 0;
      }
      else {
	event->xkey.keycode = keycode_type;
      }
    }
    else if (event->xkey.keycode == keycode_boss)
      event->xkey.keycode = keycode_noop;
    break;

  case MotionNotify:
    if (dograb) {
      toggle_grab(display);
      dograb = 0;
    }
    if (checkstate) {
      if (event->xmotion.state & LockMask ||
	  event->xmotion.state & Mod3Mask) {
	speedPress = 1;
	resetSpeed = 1;
      }
      checkstate = 0;
    }

    if (resetSpeed) {
      event->type = KeyPress;
      event->xkey.keycode = keycode_speed;
      resetSpeed = 0;
    }
    else if (grabbed) {
      if (event->xmotion.x_root != center_x ||
	  event->xmotion.y_root != center_y) {
	curx += (event->xmotion.x_root - center_x);
	if (!suppress_vertical)
	  cury += (event->xmotion.y_root - center_y);
	XWarpPointer(display, None, DefaultRootWindow(display),
		     0, 0, 0, 0, center_x, center_y);
      }

      event->xmotion.x = curx;
      event->xmotion.x_root = curx;
      event->xmotion.y = cury;
      event->xmotion.y_root = cury;
    }
    break;

  case MapNotify:
    boss_alert(display, 0);
    break;

  default:
    break;
  }

  return retval;
}

long
sysconf(int name)
{
  static long (*sysconf)(int name) = NULL;

  if (!sysconf) {
    void *handle;

    handle = dlopen("libc.so", RTLD_LAZY);
    if (!handle) {
      fprintf(stderr, "dlopen: %s\n", dlerror());
      exit(1);
    }
    sysconf = (long (*)(int))dlsym(handle, "sysconf");
    if (!sysconf) {
      fprintf(stderr, "dlsym sysconf: %s\n", dlerror());
      exit(1);
    }
  }

  /* Don't allow -mp in multi-player games */
  if (players[1].magicnum && (name == _SC_NPROCESSORS_ONLN))
    return 1;
  else
    return (*sysconf)(name);
}

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#define MAXPLAYER 4
#define MAXALIAS 16

static int checkAlias;
static unsigned long alias[MAXPLAYER-1][MAXALIAS];
static int nalias[MAXPLAYER-1];

int
recvfrom(int s, char *buf, int len, int flags,
	 struct sockaddr *from, int *fromlen)
{
  static int (*recvfrom)(int, char *, int, int, struct sockaddr *, int *)=NULL;
  struct sockaddr_in *from_in;
  int i, j, retval;

  if (!recvfrom) {
    void *handle;
    char *str, *ptr;
    char *strings[MAXPLAYER-1];
    int ncolons;

    handle = dlopen("libsocket.so", RTLD_LAZY);
    if (!handle) {
      fprintf(stderr, "dlopen: %s\n", dlerror());
      exit(1);
    }
    recvfrom = (int (*)(int, char *, int, int, struct sockaddr *, int *))
      dlsym(handle, "recvfrom");
    if (!recvfrom) {
      fprintf(stderr, "dlsym recvfrom: %s\n", dlerror());
      exit(1);
    }

    /* Need to figure out ip addresses for other hosts */
    str = getenv("DOOMALIASES");
    checkAlias = 0;
    if (str != NULL) {
      /* Separate alias strings */
      str = strdup(str);
      ptr = str;
      ncolons = 0;
      strings[ncolons++] = str;

      while (*ptr) {
	if (*ptr == ':') {
	  *ptr = '\0';
	  strings[ncolons++] = ptr+1;
	  if (ncolons == (MAXPLAYER-1))
	    break;
	}
	ptr++;
      }

      /* Parse each alias string */
      memset(nalias, 0, sizeof(int) * (MAXPLAYER-1));
      for (i=0; i<ncolons; i++) {
	for (j=0, ptr=strtok(strings[i], " \t\n");
	     j<MAXALIAS && ptr;
	     j++, ptr=strtok(NULL, " \t\n")) {
	  alias[i][j] = inet_addr(ptr);
	  if (alias[i][j] == -1)
	    j--;
	}
	nalias[i] = j;
      }
      free(str);

      /* Check if we got anything */
      for (i=0; i<(MAXPLAYER-1); i++) {
	if (nalias[i] != 0) {
	  checkAlias = 1;
	  break;
	}
      }
    }
  }

  retval = (*recvfrom)(s, buf, len, flags, from, fromlen);

  if (checkAlias) {
    from_in = (struct sockaddr_in *)from;
    for (i=0; i<(MAXPLAYER-1); i++) {
      for (j=0; j<nalias[i]; j++) {
	if (alias[i][j] == from_in->sin_addr.s_addr) {
	  from_in->sin_addr.s_addr = alias[i][0];
	  break;
	}
      }
    }
  }

  return retval;
}
