/*
*	SNDSERVER.C - NAS Implementation of the *nix DOOM(tm)
*	soundserver.
*
*	$Id: sndserver.c,v 1.3 1996/04/22 06:08:26 toor Exp $
*	$Log: sndserver.c,v $
*	Revision 1.3  1996/04/22 06:08:26  toor
*	Fixed up log/id placement..
*
*
*/

/*
Copyright (C) 1996 Stephen Hocking.
All Rights Reserved.

Permission to use, copy, modify and distribute this software
for any purpose is hereby granted without fee, provided that the
above copyright notice and this notice appear in all copies
and that both the copyright notice and this notice appear in
supporting documentation.  Stephen Hocking makes no representations about
the suitability of this software for any purpose.  It is provided
"AS IS" without express or implied warranty.

STEPHEN HOCKING DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
IN NO EVENT SHALL SCO BE LIABLE FOR ANY SPECIAL, INDIRECT,
PUNITIVE, CONSEQUENTIAL OR INCIDENTAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, LOSS OF DATA OR LOSS OF
PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.

"DOOM" is a trademark of iD Software Inc. 
*/

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include "config.h"

#define	NSAMPLES	2048
#define	WRITECHUNKS	256	/* How much sent off in between command polls */
#define	BUFSAMPS	(11025 * 2 * 12)	/* 12 seconds of stereo 16bit 11025Hz */
#define	NEUTRAL	128		/* The neutral position, pitch & volume */
/* 
* Copy the sounds off into a malloced area, rather than using them direct
* from the mmapped file. Hopefully this will give us better page locality.
*/
#define	USE_MALLOC
/*
 * The sounds, in the order that the id soundservers search for, deduced by
 * listing the wad directory and using truss to see the seeks and reads.
 * Lucky they didn't use mmap....
 */

static struct snd_data
{
	char            snd_name[9];
	unsigned short  snd_size;
	unsigned char  *snd_sound;
}               snd_array[] =
{
	{
		"DSPISTOL", 0, 0
	},
	{
		"DSSHOTGN", 0, 0
	},
	{
		"DSSGCOCK", 0, 0
	},
	{
		"DSDSHTGN", 0, 0
	},
	{
		"DSDBOPN", 0, 0
	},
	{
		"DSDBCLS", 0, 0
	},
	{
		"DSDBLOAD", 0, 0
	},
	{
		"DSPLASMA", 0, 0
	},
	{
		"DSBFG", 0, 0
	},
	{
		"DSSAWUP", 0, 0
	},
	{
		"DSSAWIDL", 0, 0
	},
	{
		"DSSAWFUL", 0, 0
	},
	{
		"DSSAWHIT", 0, 0
	},
	{
		"DSRLAUNC", 0, 0
	},
	{
		"DSRXPLOD", 0, 0
	},
	{
		"DSFIRSHT", 0, 0
	},
	{
		"DSFIRXPL", 0, 0
	},
	{
		"DSPSTART", 0, 0
	},
	{
		"DSPSTOP", 0, 0
	},
	{
		"DSDOROPN", 0, 0
	},
	{
		"DSDORCLS", 0, 0
	},
	{
		"DSSTNMOV", 0, 0
	},
	{
		"DSSWTCHN", 0, 0
	},
	{
		"DSSWTCHX", 0, 0
	},
	{
		"DSPLPAIN", 0, 0
	},
	{
		"DSDMPAIN", 0, 0
	},
	{
		"DSPOPAIN", 0, 0
	},
	{
		"DSVIPAIN", 0, 0
	},
	{
		"DSMNPAIN", 0, 0
	},
	{
		"DSPEPAIN", 0, 0
	},
	{
		"DSSLOP", 0, 0
	},
	{
		"DSITEMUP", 0, 0
	},
	{
		"DSWPNUP", 0, 0
	},
	{
		"DSOOF", 0, 0
	},
	{
		"DSTELEPT", 0, 0
	},
	{
		"DSPOSIT1", 0, 0
	},
	{
		"DSPOSIT2", 0, 0
	},
	{
		"DSPOSIT3", 0, 0
	},
	{
		"DSBGSIT1", 0, 0
	},
	{
		"DSBGSIT2", 0, 0
	},
	{
		"DSSGTSIT", 0, 0
	},
	{
		"DSCACSIT", 0, 0
	},
	{
		"DSBRSSIT", 0, 0
	},
	{
		"DSCYBSIT", 0, 0
	},
	{
		"DSSPISIT", 0, 0
	},
	{
		"DSBSPSIT", 0, 0
	},
	{
		"DSKNTSIT", 0, 0
	},
	{
		"DSVILSIT", 0, 0
	},
	{
		"DSMANSIT", 0, 0
	},
	{
		"DSPESIT", 0, 0
	},
	{
		"DSSKLATK", 0, 0
	},
	{
		"DSSGTATK", 0, 0
	},
	{
		"DSSKEPCH", 0, 0
	},
	{
		"DSVILATK", 0, 0
	},
	{
		"DSCLAW", 0, 0
	},
	{
		"DSSKESWG", 0, 0
	},
	{
		"DSPLDETH", 0, 0
	},
	{
		"DSPDIEHI", 0, 0
	},
	{
		"DSPODTH1", 0, 0
	},
	{
		"DSPODTH2", 0, 0
	},
	{
		"DSPODTH3", 0, 0
	},
	{
		"DSBGDTH1", 0, 0
	},
	{
		"DSBGDTH2", 0, 0
	},
	{
		"DSSGTDTH", 0, 0
	},
	{
		"DSCACDTH", 0, 0
	},
	{
		"DSSKLDTH", 0, 0
	},
	{
		"DSBRSDTH", 0, 0
	},
	{
		"DSCYBDTH", 0, 0
	},
	{
		"DSSPIDTH", 0, 0
	},
	{
		"DSBSPDTH", 0, 0
	},
	{
		"DSVILDTH", 0, 0
	},
	{
		"DSKNTDTH", 0, 0
	},
	{
		"DSPEDTH", 0, 0
	},
	{
		"DSSKEDTH", 0, 0
	},
	{
		"DSPOSACT", 0, 0
	},
	{
		"DSBGACT", 0, 0
	},
	{
		"DSDMACT", 0, 0
	},
	{
		"DSBSPACT", 0, 0
	},
	{
		"DSBSPWLK", 0, 0
	},
	{
		"DSVILACT", 0, 0
	},
	{
		"DSNOWAY", 0, 0
	},
	{
		"DSBAREXP", 0, 0
	},
	{
		"DSPUNCH", 0, 0
	},
	{
		"DSHOOF", 0, 0
	},
	{
		"DSMETAL", 0, 0
	},
	{
		"DSTINK", 0, 0
	},
	{
		"DSBDOPN", 0, 0
	},
	{
		"DSBDCLS", 0, 0
	},
	{
		"DSITMBK", 0, 0
	},
	{
		"DSFLAME", 0, 0
	},
	{
		"DSFLAMST", 0, 0
	},
	{
		"DSGETPOW", 0, 0
	},
	{
		"DSBOSPIT", 0, 0
	},
	{
		"DSBOSCUB", 0, 0
	},
	{
		"DSBOSSIT", 0, 0
	},
	{
		"DSBOSPN", 0, 0
	},
	{
		"DSBOSDTH", 0, 0
	},
	{
		"DSMANATK", 0, 0
	},
	{
		"DSMANDTH", 0, 0
	},
	{
		"DSSSSIT", 0, 0
	},
	{
		"DSSSDTH", 0, 0
	},
	{
		"DSKEENPN", 0, 0
	},
	{
		"DSKEENDT", 0, 0
	},
	{
		"DSSKEACT", 0, 0
	},
	{
		"DSSKESIT", 0, 0
	},
	{
		"DSSKEATK", 0, 0
	},
	{
		"DSRADIO", 0, 0
	},
	{
		"", 0, 0
	}
};

static int      quiet = 0;

static void
fatalError(char *message)
{
	fprintf(stderr, message);
	fprintf(stderr, "\n");
	exit(1);
}

/*
 * Look for the sounds in the current wadfile's directory and replace the
 * ones in our directory. Note that this function should only be called after
 * the wadfile has been mmaped in.
 */
static void
setupSounds()
{
	int             i, j;
	unsigned int    offset;
	char           *name;

	for (j = 0; j < (sizeof(snd_array) / sizeof(struct snd_data)) - 1; j++)
	{
		for (i = 0; i < NUMLUMPS; i++)
		{
			name = DIR_ITEM_NAME(i);
			offset = DIR_ITEM_OFFSET(i);
			if (!strncmp(name, snd_array[j].snd_name, 8))
			{
				snd_array[j].snd_size = SOUND_NUM_SAMPLES(offset);
#ifdef USE_MALLOC
				if (snd_array[j].snd_sound != NULL)
					free(snd_array[j].snd_sound);
				if ((snd_array[j].snd_sound = malloc(snd_array[j].snd_size)) == NULL)
					fatalError("Not enough memory!\n");
				memcpy(snd_array[j].snd_sound, SOUND_DATA(offset), snd_array[j].snd_size);
#else
				snd_array[j].snd_sound = SOUND_DATA(offset);
#endif
			}
		}
	}
}

/*
 * Map the wadfile into the address space, then extract any sounds into our
 * sound directory.
 */

static void
setupWadfile(int wadfile)
{
	struct stat     wb;
	int             filesize;

	if (fstat(wadfile, &wb) < 0)
	{
		perror("stat");
		exit(1);
	}
	filesize = wb.st_size;
	if ((wadbase = mmap((caddr_t) 0, (size_t) filesize, PROT_READ, MAP_PRIVATE, wadfile, (off_t) 0)) == NULL)
	{
		perror("mmap");
		exit(1);
	}
	dir_offset = (unsigned int) wadbase + DIR_OFFSET;
	setupSounds();
#ifdef USE_MALLOC
	if (munmap((caddr_t) wadbase, (size_t) filesize))
	{
		perror("munmap");
		exit(1);
	}
#endif
	close(wadfile);
}

/*
 * The circular list of buffers we'll be using
 */

static short   *soundbuf;
static short   *cursamp;
static short   *playsamp;
static short   *endsamp;

static int
soundBufAllocate()
{
	if ((playsamp = cursamp = soundbuf = (short *) calloc(BUFSAMPS, sizeof(short))) == NULL)
		return 1;
	endsamp = soundbuf + BUFSAMPS - 2;
	return 0;
}


/*
 * This one just drops the sample into the buffer, adding it to whatever's
 * already there.
 */

inline static void
putSamples(int left, int right)
{
	*cursamp++ += left;
	*cursamp++ += right;
	if (cursamp >= endsamp)
		cursamp = soundbuf;
}

/*
 * This function takes a command for a sound, calculates the sample after
 * volume changes, stereo positioning and calls putSamples to place the
 * sample into the buffer.
 */

static void
putSound(int snd_id, int pitch, int volume, int posn)
{
	int             x;
	short           sample;
	int             left, right, size, voldiff;
	int             nextSample, integ, fract;
	int             want;
	register int inc, currentSample;
	register unsigned char *data;

	cursamp = playsamp;
	size = snd_array[snd_id].snd_size;
	data = snd_array[snd_id].snd_sound;

	/* set up pitch shifting parameters */
	want = (size * NEUTRAL) / pitch;
	integ = pitch / NEUTRAL;
	fract = ((pitch - (integ * NEUTRAL)) << 16) / NEUTRAL;
	currentSample = 0;
	nextSample = (integ << 16) | fract;

	/* setup volume (with squashed range) & balance */
	voldiff = (volume - NEUTRAL) / 2;
	volume = NEUTRAL + voldiff;
	x = NEUTRAL - posn + NEUTRAL;

	if (currentSample & 0xffff0000)
		goto NEXT;

	while (want && size)
	{
		sample = (*data << 8) ^ 0x8000;	/* u_int8 to int16 */
		left = ((int) sample * (x + 1) * volume) >> 16;
		right = ((int) sample * (posn + 1) * volume) >> 16;
		putSamples(left, right);
		want--;

		currentSample += nextSample;
NEXT:
		if ((inc = currentSample >> 16))
		{
			currentSample &= 0xffff;

			if (inc > size)
			{
				currentSample |= (inc - size) << 16;
				inc = size;
			}
			data++;
			size--;
		}
	}
}

/*
 * Checks stdin to see if there's a command, if so it acts accordingly
 */
static void
doCmdCheck()
{
	static char     buf[10];/* where we read each command */
	int             n, snd_id, snd_pitch, snd_vol, snd_posn, i;

	n = read(0, buf, 9);
	if (n > 0)
	{
		buf[9] = 0;
		if (buf[0] == 'p')
		{
			i = (int) strtol(buf + 1, (char **) NULL, 16);
			snd_id = ((i & 0xff000000) >> 24) - 1;
			snd_pitch = (i & 0xff0000) >> 16;
			snd_vol = (i & 0xff00) >> 8;
			snd_posn = i & 0xff;
			if (!quiet)
				printf("id %d %8.8s pitch %d, vol %d, posn %d,\n",
				       snd_id, snd_array[snd_id].snd_name, snd_pitch,
				       snd_vol, snd_posn);
			putSound(snd_id, snd_pitch, snd_vol, snd_posn);
			for (i = 0; i < 10; i++)
				buf[i] = 0;
		}
		else if (buf[0] == 'q')
		{
			if (!quiet)
				printf("Goodbye...\n");
			exit(0);
		}
	}
	else if (n < 0 && errno != EAGAIN)
		exit(1);
}

static int      dataFormat;
static int      sampleRate = 11025;
static int      numTracks = 2;

typedef struct
{
	AuServer       *aud;
	AuFlowID        flow;
}               InfoRec, *InfoPtr;

static InfoRec  infor;
static char    *server = NULL;

static void
sendFile(AuServer * aud, InfoPtr i, AuUint32 numBytes)
{
	int             n, m;

	if ((playsamp + (numBytes / sizeof(short))) >= endsamp)
	{
		m = (endsamp - playsamp) * sizeof(short);
		while (m)
		{
			if (m < WRITECHUNKS)
				n = m;
			else
				n = WRITECHUNKS;
			AuWriteElement(aud, i->flow, 0, n, (char *) playsamp, AuFalse, NULL);
			memset(playsamp, 0, n);
			playsamp += (n / sizeof(short));
			m -= n;
			numBytes -= n;
			doCmdCheck();
		}
		playsamp = soundbuf;
	}
	while (numBytes)
	{
		if (numBytes < WRITECHUNKS)
			n = numBytes;
		else
			n = WRITECHUNKS;
		doCmdCheck();
		AuWriteElement(aud, i->flow, 0, n, (char *) playsamp, AuFalse, NULL);
		memset((void *) playsamp, 0, n);
		playsamp += (n / sizeof(short));
		numBytes -= n;
	}

}

static          AuBool
eventHandler(AuServer * aud, AuEvent * ev,
	     AuEventHandlerRec * handler)
{
	InfoPtr         i = (InfoPtr) handler->data;

	switch (ev->type)
	{
	case AuEventTypeElementNotify:
		{
			AuElementNotifyEvent *event = (AuElementNotifyEvent *) ev;

			switch (event->kind)
			{
			case AuElementNotifyKindLowWater:
				sendFile(aud, i, event->num_bytes);
				break;
			case AuElementNotifyKindState:
				switch (event->cur_state)
				{
				case AuStatePause:
					if (event->reason != AuReasonUser)
						sendFile(aud, i, event->num_bytes);
					break;
				}
			}
		}
	}
	return AuTrue;
}

static void
doNetAudioConn()
{
	AuDeviceID      device = AuNone;
	AuElement       elements[3];
	int             i;


	if (!(infor.aud = AuOpenServer(server, 0, NULL, 0, NULL, NULL)))
		fatalError("Can't open audio server\n");

	dataFormat = (littleEndian) ?
		AuFormatLinearSigned16LSB : AuFormatLinearSigned16MSB;

	/* look for an output device */
	for (i = 0; i < AuServerNumDevices(infor.aud); i++)
		if ((AuDeviceKind(AuServerDevice(infor.aud, i)) ==
		     AuComponentKindPhysicalOutput) &&
		AuDeviceNumTracks(AuServerDevice(infor.aud, i)) == numTracks)
		{
			device = AuDeviceIdentifier(AuServerDevice(infor.aud, i));
			break;
		}

	if (device == AuNone)
		fatalError("Couldn't find an output device");

	if (!(infor.flow = AuCreateFlow(infor.aud, NULL)))
		fatalError("Couldn't create flow");

	AuMakeElementImportClient(&elements[0], sampleRate, dataFormat,
			numTracks, AuTrue, NSAMPLES * 2, NSAMPLES, 0, NULL);
	AuMakeElementExportDevice(&elements[1], 0, device, sampleRate,
				  AuUnlimitedSamples, 0, NULL);
	AuSetElements(infor.aud, infor.flow, AuTrue, 2, elements, NULL);

	AuRegisterEventHandler(infor.aud, AuEventHandlerIDMask, 0, infor.flow,
			       eventHandler, (AuPointer) & infor);
}

static void
NASHandleEvents()
{
	AuStartFlow(infor.aud, infor.flow, NULL);
	for (;;)
	{
		AuEvent         ev;

		doCmdCheck();
		AuNextEvent(infor.aud, AuTrue, &ev);
		doCmdCheck();
		AuDispatchEvent(infor.aud, &ev);
	}
}


#if defined(USE_VOXWARE)
static int      soundfd;

static int
openVOXwareDevice()
{
	int             chans = 2, wordsize = 0x10;
	if ((soundfd = open("/dev/dsp", O_RDWR)) < 0)
		fprintf(stderr, "Could not open /dev/dsp");
	else
	{
		int             frag = (2 << 16) | 11;	/* 2048 sized frags */
		ioctl(soundfd, SNDCTL_DSP_SAMPLESIZE, &wordsize);
		ioctl(soundfd, SND_DSP_STEREO, &chans);
		ioctl(soundfd, SNDCTL_DSP_SPEED, &sample_speed);
		ioctl(soundfd, SOUND_PCM_SETFRAGMENT, &frag);
	}
}

static int
doVOXwareLoop()
{
	while (1)
	{
		doCmdCheck();
		write(soundfd, (char *) playsamp, 2048);
		playsamp += 1024;
		if (playsamp >= endsamp)
			playsamp = soundbuf;
	}
}
#endif


int
main(int argc, char *argv[])
{
	int             i, wadfile = -1;
	char           *doomwaddir = NULL;
	char           *extrawads, *tmp, *tmp2;
	char            wadnamebuf[BUFSIZ];
	char           *wadnames[] = {
		"doom2f.wad",
		"doom2.wad",
		"doom.wad",
		"doom1.wad",
	NULL};

	for (i = 1; i < argc; i++)
	{
		if (!strcmp("-quiet", argv[i]))
			quiet++;
	}

	if (soundBufAllocate())
		fatalError("Not enough memory!");

	if ((doomwaddir = getenv("DOOMWADDIR")) == NULL)
		doomwaddir = ".";

	for (i = 0; wadnames[i] != NULL && wadfile < 0; i++)
	{
		strcpy(wadnamebuf, doomwaddir);
		strcat(wadnamebuf, "/");
		strcat(wadnamebuf, wadnames[i]);
		wadfile = open(wadnamebuf, O_RDONLY);
	}
	if (wadfile < 0)
		fatalError("Could not find any wadfiles!");
	if (!quiet)
		printf("[%s]\n", wadnamebuf);

	doNetAudioConn();	/* Open up sound server */
	fcntl(0, F_SETFL, O_NONBLOCK);	/* don't block waiting for commands */

	setupWadfile(wadfile);
	if ((extrawads = getenv("WADFILES")) != NULL && strlen(extrawads) > 0)
	{
		tmp = extrawads + strlen(extrawads);
		while (tmp2 < tmp)
		{
			if ((tmp2 = strchr(extrawads, ' ')) != NULL)
				*tmp2 = 0;
			if ((wadfile = open(extrawads, O_RDONLY)) < 0)
				fatalError("Could not  open secondary wadfile!");
			if (!quiet)
				printf("[%s]\n", extrawads);
			setupWadfile(wadfile);
			if (tmp2)
				extrawads = tmp2 + 1;
			else
				break;
		}
	}

	if (!quiet)
		printf("Ready\n");

	NASHandleEvents();	/* Wait for things to happen */
	return 0;
}
