/***********************************************************
*  joystick4doom - Playing Linux DOOM with a joystick      *
*----------------------------------------------------------*
*  1994  Artsoft Development                               *
*        Holger Schemel                                    *
*        33659 Bielefeld-Senne                             *
*        Telefon: (0521) 493245                            *
*        eMail: aeglos@valinor.ms.sub.org                  *
*               aeglos@uni-paderborn.de                    *
*               q99492@pbhrzx.uni-paderborn.de             *
*----------------------------------------------------------*
*  programming date: 02.10.94                              *
*----------------------------------------------------------*
*  Fine tuned and enhanced October 19, 1994, by            *
*     Sam Lantinga (slouken@cs.ucdavis.edu)                *
*                                                          *
#  Modified to support SVGA DOOM! December 14, 1994, by    *
*     Sam Lantinga (slouken@cs.ucdavis.edu)                *
***********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#ifdef X11
#define XK_MISCELLANY
#define XK_LATIN1

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Intrinsic.h>
#include <X11/keysymdef.h>
#else
#include <sys/termios.h>

#define TRUE	1
#define FALSE	0
#endif /* X11 */

#include <linux/joystick.h>

#define PIDFILE		"/tmp/joystick4doom.pid"
#ifndef JOYSTICK_DAT
#define JOYSTICK_DAT	"/usr/local/lib/joystick.dat"
#endif

#define PAUSETIME	100000

#define RELEASED	0
#define PRESSED		1
#define HOLDDOWN	2
#define UNPRESSED	3

#define FIRST_BUTTON	0
#define SECOND_BUTTON	1

#define JB_PRESSED(b)	(button[b]==PRESSED || button[b]==HOLDDOWN)
#define JB_UNPRESSED(b)	(button[b]==RELEASED || button[b]==UNPRESSED)
#define ABS(x)		((x)>0 ? (x) : (-x))

/* Thresholds (in percentages) */
#ifdef THRESHOLD
#define KEYRELEASE_AT	THRESHOLD
#define KEYPRESS_AT	THRESHOLD
#else
#define SENSITIVE
#ifdef SENSITIVE
#define KEYRELEASE_AT	10
#define KEYPRESS_AT	15
#else
#define KEYRELEASE_AT	15
#define KEYPRESS_AT	20
#endif
#endif /* THRESHOLD */
#define RUNNING_AT	90

#define JK_SHIFT	0
#define JK_ALT		1
#define JK_META		2
#define JK_CONTROL	3
#define JK_SPACE	4
#define JK_ESCAPE	5
#ifndef X11
#define JK_UP		6
#define JK_DOWN		7
#define JK_RIGHT	8
#define JK_LEFT		9
#endif /* ! X11 */

#ifdef X11
Display			*display;
#else
int                     console;
#endif
char			*progname;
int			joystick_device;
char			*device_name;
struct JS_DATA_TYPE	js;
int			button[2];
int			xpos_left, xpos_middle, xpos_right, xpos_actual;
int			ypos_upper,ypos_middle, ypos_lower, ypos_actual;
int			i,j;

/* Keep track of where the joystick is */
#define	CENTERED	0x00
#define CUTRIGHT	0x01
#define CUTLEFT		0x02
#define CUTUP		0x10
#define CUTDOWN		0x20
#define RUNNING		0x40

#define SET(var, flag)		(var |= flag)
#define UNSET(var, flag)	(var &= ~flag)
#define ISSET(var, flag)	((var|flag) == var)
#define ISRIGHT(var)		ISSET(var, CUTRIGHT)
#define ISLEFT(var)		ISSET(var, CUTLEFT)
#define ISUP(var)		ISSET(var, CUTUP)
#define ISDOWN(var)		ISSET(var, CUTDOWN)
#define ISRUNNING(var)		ISSET(var, RUNNING)

char *str2lower(char *);
void GetJoystick(void);
int GetMovement(int, int, int);
void WaitButton(void);
#ifdef X11
void SendKeyEvent(int, int, KeySym);
#else
void SendKeyEvent(char);
#endif
void Calibrate(void);
void quit();

int main(int argc, char *argv[])
{
  int up,down,left,right;
  int kp=KEYPRESS_AT;		/* threshold for pressed cursor key */
  int kr=KEYRELEASE_AT;		/* threshold for released cursor key */
  int kh=RUNNING_AT;		/* threshold for high speed motion */
  int key[2];
  short special_strafe=FALSE;	/* Do we want sideways movement? */
  char joystate=CENTERED;	/* Current state of the joystick */
  FILE *pidfile; int pid;
  FILE *datfile;

#ifdef X11
  struct keys
  {
    char *keyname;
    int pressed, released;
    KeySym key;
  };
  static struct keys joykey[6] =
  {
    "shift", 0,ShiftMask, XK_Shift_L,
    "alt", 0,Mod1Mask, XK_Meta_L,
    "meta", 0,Mod2Mask, XK_Alt_L,
    "control", 0,ControlMask, XK_Control_L,
    "space", 0,0, XK_space,
    "escape", 0,0, XK_Escape
  };
#else
  struct keys
  {
    char *keyname;
    char pressed, released;
  };
  static struct keys joykey[10] =
  {
    "shift", 0x2a, 0xaa,
    "alt", 0x38, 0xb8,
    "meta", 0x38, 0xb8,
    "control", 0x1d, 0x9d,
    "space", 0x39, 0xb9,
    "escape", 0x01, 0x81,
    "up", 0x67, 0xe7,
    "down", 0x6c, 0xec,
    "right", 0x6a, 0xea,
    "left", 0x69, 0xe9,
  };
#endif /* X11 */

  fprintf(stdout, "Joystick4Doom, 1994 by Artsoft Development\n");

  /* Kill an existing joystick process if requested */
  if ( (argc == 2) && (strcmp(argv[1], "-k") == 0) ) {
    if ( (pidfile=fopen(PIDFILE, "r")) != NULL ) {
      fscanf(pidfile, "%d", &pid);
      (void) kill(pid, SIGINT);
    }
    exit(0);
  }
  /* Calibrate the joystick if requested */
  if ( (argc == 3) && (strcmp(argv[1], "-c") == 0) ) {
    device_name=argv[2];
    if ( (joystick_device=open(device_name,O_RDONLY)) < 0 ) {
      perror(device_name);
      exit(-1);
    }
    Calibrate();
    printf("Joystick calibrated.\n");
    exit(0);
  }

  if (argc<2 || argc>5 || *argv[1]=='?')
  {
    fprintf(stderr, "Usage: %s {-c device}|{-k}\nor..\n" ,argv[0]);
    fprintf(stderr, "Usage: %s device key1 key2 [special]\n",argv[0]);
    fprintf(stderr, "   device:  the device of your joystick, e.g. /dev/js0\n");
    fprintf(stderr, "   key1/2:  the key you want to map to button 1/2;\n");
    fprintf(stderr, "            this can be Shift, Alt/Meta, Control, Space or Escape\n");
    fprintf(stderr, "   special: set this to '-strafe' for sideways movement with button 2\n");
    exit(-1);
  }

  for(i=0;i<2;i++)
  {
    key[i]=-1;
    for(j=0;j<6;j++)
    {
      if (!strcmp(str2lower(argv[i+2]),joykey[j].keyname))
	key[i]=j;
    }
    if (key[i]<0)
    {
      fprintf(stderr, "%s: unknown key\n",argv[i+2]);
      exit(-1);
    }
  }

  if (argc==5 && !strcmp(str2lower(argv[4]),"-strafe"))
    special_strafe=TRUE;

  progname=argv[0];
  device_name=argv[1];

  if ((joystick_device=open(device_name,O_RDONLY))<0)
  {
    perror(device_name);
    exit(-1);
  }

#ifdef X11
  /* connect to X server */
  if (!(display=XOpenDisplay(NULL)))
  {
    fprintf(stderr, "%s: cannot connect to X server\n", progname);
    exit(-1);
  }
#else
  if ( (console=open("/dev/tty0", O_RDONLY, 0)) < 0 )
  {
    fprintf(stderr, "%s: cannot connect to SVGA server\n", progname);
    exit(-1);
  }
#endif /* X11 */

  button[0]=button[1]=RELEASED;

#ifdef PRECALIBRATED
  xpos_left   = 15;
  ypos_upper  = 25;
  xpos_right  = 1920;
  ypos_lower  = 1950;
  xpos_middle = 750;
  ypos_middle = 600;
#else
  if ( (datfile=fopen(JOYSTICK_DAT, "r")) == NULL )
    Calibrate();
  else {
    fscanf(datfile, "%d %d %d\n", &xpos_left, &xpos_middle, &xpos_right);
    fscanf(datfile, "%d %d %d\n", &ypos_upper, &ypos_middle, &ypos_lower);
    fclose(datfile);
  }
#endif

  /* Register this process */
  if ( (pidfile=fopen(PIDFILE, "w")) != NULL ) {
    fprintf(pidfile, "%d\n", getpid());
    fclose(pidfile);
  }
  signal(SIGINT, quit);
  signal(SIGTERM,  quit);
  fprintf(stdout, "(Interrupt with 'Control-C' or '-k' option after playing...)\n");

  for(;;)
  {
#ifdef DEBUG
	printf("loop: <- %%%d ^ %%%d V %%%d  %%%d ->\n", left, up, down, right);
#endif
    GetJoystick();

    left =  GetMovement(xpos_middle, xpos_left,  xpos_actual);
    right = GetMovement(xpos_middle, xpos_right, xpos_actual);
    up =    GetMovement(ypos_middle, ypos_upper, ypos_actual);
    down =  GetMovement(ypos_middle, ypos_lower, ypos_actual);

    for(i=0;i<2;i++)
    {
      if (button[i]==PRESSED)
      {
#ifdef X11
	SendKeyEvent(KeyPress,joykey[key[i]].pressed,joykey[key[i]].key);
#else
        SendKeyEvent(joykey[key[i]].pressed);
#endif
#ifdef DEBUG
	printf("Sent V key #%d\n", i);
#endif
	if (special_strafe && (i == SECOND_BUTTON)) {
#ifdef X11
	  SendKeyEvent(KeyPress,joykey[JK_ALT].pressed,joykey[JK_ALT].key);
#else
          SendKeyEvent(joykey[JK_ALT].pressed);
#endif /* X11 */
#ifdef DEBUG
	  printf("sent V key 'ALT'\n", SECOND_BUTTON);
#endif
	}
      }
      if (button[i]==RELEASED)
      {
#ifdef X11
	SendKeyEvent(KeyRelease,joykey[key[i]].released,joykey[key[i]].key);
#else
        SendKeyEvent(joykey[key[i]].released);
#endif
#ifdef DEBUG
	printf("Sent ^ key #%d\n", i);
#endif
	if (special_strafe && (i == SECOND_BUTTON)) {
#ifdef X11
	  SendKeyEvent(KeyRelease,joykey[JK_ALT].released,joykey[JK_ALT].key);
#else
          SendKeyEvent(joykey[JK_ALT].released);
#endif
#ifdef DEBUG
	  printf("sent ^ key 'ALT'\n", SECOND_BUTTON);
#endif
	}
      }
    }

    /* Check to see if joystick is pushed far enough for us to run */
    if ( !ISRUNNING(joystate) ) {
      if ( (up>kh) || (left>kh) || (right>kh) || (down>kh) ) {
#ifdef X11
        SendKeyEvent(KeyPress,0,XK_Shift_L);
#else
        SendKeyEvent(joykey[JK_SHIFT].pressed);
#endif
        SET(joystate, RUNNING);
#ifdef DEBUG
        printf("Sent V <Shift>\n");
#endif
      }
    } else {
      if ( (up<kh) && (left<kh) && (right<kh) && (down<kh) ) {
#ifdef X11
        SendKeyEvent(KeyRelease,ShiftMask,XK_Shift_L);
#else
        SendKeyEvent(joykey[JK_SHIFT].released);
#endif
        UNSET(joystate, RUNNING);
#ifdef DEBUG
        printf("Sent ^ <Shift>\n");
#endif
      }
    }

    /* Crosscheck motion of joystick with current state of the keys */
    if (!ISLEFT(joystate) && (left>kp)) {
#ifdef X11
      SendKeyEvent(KeyPress,0,XK_Left);
#else
      SendKeyEvent(joykey[JK_LEFT].pressed);
#endif
      SET(joystate, CUTLEFT);
#ifdef DEBUG
	printf("Sent V Left\n");
#endif
    }
    if (!ISRIGHT(joystate) && (right>kp)) {
#ifdef X11
      SendKeyEvent(KeyPress,0,XK_Right);
#else
      SendKeyEvent(joykey[JK_RIGHT].pressed);
#endif
      SET(joystate, CUTRIGHT);
#ifdef DEBUG
	printf("Sent V Right\n");
#endif
    }
    if (!ISUP(joystate) && (up>kp)) {
#ifdef X11
      SendKeyEvent(KeyPress,0,XK_Up);
#else
      SendKeyEvent(joykey[JK_UP].pressed);
#endif
      SET(joystate, CUTUP);
#ifdef DEBUG
	printf("Sent V Up\n");
#endif
    }
    if (!ISDOWN(joystate) && (down>kp)) {
#ifdef X11
      SendKeyEvent(KeyPress,0,XK_Down);
#else
      SendKeyEvent(joykey[JK_DOWN].pressed);
#endif
      SET(joystate, CUTDOWN);
#ifdef DEBUG
	printf("Sent V Down\n");
#endif
    }

    if (ISLEFT(joystate) && (left<kr)) {
#ifdef X11
      SendKeyEvent(KeyRelease,0,XK_Left);
#else
      SendKeyEvent(joykey[JK_LEFT].released);
#endif
      UNSET(joystate, CUTLEFT);
#ifdef DEBUG
	printf("Sent ^ Left\n");
#endif
    }
    if (ISRIGHT(joystate) && (right<kr)) {
#ifdef X11
      SendKeyEvent(KeyRelease,0,XK_Right);
#else
      SendKeyEvent(joykey[JK_RIGHT].released);
#endif
      UNSET(joystate, CUTRIGHT);
#ifdef DEBUG
	printf("Sent ^ Right\n");
#endif
    }
    if (ISUP(joystate) && (up<kr)) {
#ifdef X11
      SendKeyEvent(KeyRelease,0,XK_Up);
#else
      SendKeyEvent(joykey[JK_UP].released);
#endif
      UNSET(joystate, CUTUP);
#ifdef DEBUG
	printf("Sent ^ Up\n");
#endif
    }
    if (ISDOWN(joystate) && (down<kr)) {
#ifdef X11
      SendKeyEvent(KeyRelease,0,XK_Down);
#else
      SendKeyEvent(joykey[JK_DOWN].released);
#endif
      UNSET(joystate, CUTDOWN);
#ifdef DEBUG
	printf("Sent ^ Down\n");
#endif
    }

    usleep(PAUSETIME);
  }
  quit();
}

void quit() {
  (void) unlink(PIDFILE);
#ifdef X11
  XCloseDisplay(display);
#endif
  exit(0);
}

char *str2lower(char *input)
{
  int i;

  for(i=0;input[i];i++)
    input[i]=tolower(input[i]);
  return(input);
}

void Calibrate(void)
{
  FILE *datfile;

  fprintf(stdout, "Move joystick to the upper left and press any button!\n");
  WaitButton();
  xpos_left =js.x;
  ypos_upper=js.y;
  fprintf(stdout, "xpos_left = %d\nypos_upper = %d\n", js.x,  js.y);

  fprintf(stdout, "Move joystick to the lower right and press any button!\n");
  WaitButton();
  xpos_right=js.x;
  ypos_lower=js.y;
  fprintf(stdout, "xpos_right = %d\nypos_lower = %d\n", js.x,  js.y);

  fprintf(stdout, "Move joystick to the middle and press any button!\n");
  WaitButton();
  xpos_middle=js.x;
  ypos_middle=js.y;
  fprintf(stdout, "xpos_middle = %d\nypos_middle = %d\n", js.x,  js.y);

  /* Record calibration in joystick.dat */
  if ( (datfile=fopen(JOYSTICK_DAT, "w")) == NULL ) {
    fprintf(stdout, "Warning: Can't open %s for writing.\n", JOYSTICK_DAT);
    return;
  }
  fprintf(datfile, "%d %d %d\n", xpos_left, xpos_middle, xpos_right);
  fprintf(datfile, "%d %d %d\n", ypos_upper, ypos_middle, ypos_lower);
  fclose(datfile);
}

void GetJoystick()
{
  if (read(joystick_device,&js,JS_RETURN) != JS_RETURN)
  {
    perror(device_name);
    exit(-1);
  }

  for(i=0;i<2;i++)
  {
    if (js.buttons & (1<<i))
      button[i] = (JB_UNPRESSED(i) ? PRESSED : HOLDDOWN);
    else
      button[i] = (JB_PRESSED(i) ? RELEASED : UNPRESSED);
  }

  xpos_actual=js.x;
  ypos_actual=js.y;
}

int GetMovement(int middle, int margin, int actual)
{
  long range, pos;
  int percentage;

  range=ABS(margin-middle);
  pos=actual-middle;
  percentage=(int)(pos*100/range);

  return(percentage);
}

void WaitButton()
{
  while(JB_UNPRESSED(0) && JB_UNPRESSED(1))
    GetJoystick();
  while(JB_PRESSED(0) || JB_PRESSED(1))
    GetJoystick();
}

#ifdef X11
void SendKeyEvent(int type, int state, KeySym key)
{
  long mask = (type==KeyPress ? KeyPressMask : KeyReleaseMask);
  int keycode = XKeysymToKeycode(display,key);
  static XKeyEvent keyevent =
  {
    0,
    0L,True,
    (Display *)NULL,
    (Window)0L,(Window)0L,(Window)0L,
    (Time)0L,
    0,0, 0,0,
    0, 0,
    True
  };

  keyevent.type=type;
  keyevent.state=state;
  keyevent.keycode=keycode;

  XSendEvent(display,InputFocus,False,mask,(XEvent *)&keyevent);
  XFlush(display);
}
#else
void SendKeyEvent(char key)
{
  if ( ioctl(console, TIOCSTI, &key) < 0 )
        perror("ioctl(TIOCSTI)");
}
#endif /* X11 */

