/* map2wad.c

Copyright (C) 2002 Nicholai Main

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

// defines
#define VERSIONSTR "0.85"

// these can be superceded by command line options
#define WALLSTR "SLADWALL"
#define CEILSTR "FLOOR4_6"
#define FLRSTR "FLOOR0_6"
#define NOWALLSTR "-"

// structs
// created from info in buildhlp.txt
typedef struct
{
  short wallptr, wallnum;
  long ceilingz, floorz;
  short ceilingstat, floorstat;
  short ceilingpicnum, ceilingheinum;
  signed char ceilingshade;
  char ceilingpal, ceilingxpanning, ceilingypanning;
  short floorpicnum, floorheinum;
  signed char floorshade;
  char floorpal, floorxpanning, floorypanning;
  char visibility, filler;
  short lotag, hitag, extra;
} buildsector;

typedef struct
{
  long x, y;
  short point2, nextwall, nextsector, cstat;
  short picnum, overpicnum;
  signed char shade;
  char pal, xrepeat, yrepeat, xpanning, ypanning;
  short lotag, hitag, extra;
} buildwall;

// created from info in dmspec16.txt

typedef struct
{
  short v1, v2;
  short flags, type;
  short tag;
  short sidedef1, sidedef2; // right and left sidedefs respectively
} linedef;

typedef struct
{
  short xoff, yoff;
  char uptex[8], lowtex[8], midtex[8];
  short sectoref;
} sidedef;

typedef struct
{
  short x, y;
} vertex;

typedef struct
{
  short floorheight, ceilingheight;
  char floortex[8], ceilingtex[8];
  short light, type, tag;
} sector;

typedef struct
{
  short x, y;
  short angle;
  short type, flags;
} thing;


// globals

FILE *mapfile, *wadfile;
int numbuildsectors, numbuildwalls, numbuildsprites;
buildsector *buildsectors;
buildwall *buildwalls;
linedef *linedefs;
sidedef *sidedefs;
vertex *vertexes;
sector *sectors;
thing *things;
int numlinedefs, numsidedefs, numvertexes, numsectors, numthings;
int *wallhit;

// code

void *safe_malloc (size_t size, char *file, int line)
{ // malloc() and if fail, exit with error
  void *ptr = malloc (size);
  if (!ptr)
  {
    printf ("error: failed on malloc of %i bytes\n", size);
    printf ("       source: %s:%i\n", file, line);
    // just in case they're open
    fclose (mapfile);
    fclose (wadfile);
    exit (EXIT_FAILURE);
  }
  return ptr;
}

void safe_fwrite (void *buf, size_t size, size_t num, FILE *fil, char *file, int line)
{
  if (fwrite (buf, size, num, fil) == -1)
  {
    printf ("error: failed on fwrite of %i bytes\n", size * num);
    printf ("       source: %s:%i\n", file, line);
    printf ("       Does map2wad have permission to write to the selected file?\n");
    printf ("       Are you out of disk space?\n");
    fclose (mapfile);
    fclose (wadfile);
    exit (EXIT_FAILURE);
  }
}

void safe_fread (void *buf, size_t size, size_t num, FILE *fil, char *file, int line)
{
  if (fread (buf, size, num, fil) == -1)
  {
    printf ("error: failed on fread of %i bytes\n", size * num);
    printf ("       source: %s:%i\n", file, line);
    printf ("       Does map2wad have permission to read from the selected file?\n");
    printf ("       Is this a valid BUILD mapfile?\n");
    fclose (mapfile);
    fclose (wadfile);
    exit (EXIT_FAILURE);
  }
}


#define s_fread(a,b,c,d) safe_fread (a, b, c, d, __FILE__, __LINE__)
#define s_fwrite(a,b,c,d) safe_fwrite (a, b, c, d, __FILE__, __LINE__)
#define s_malloc(x) safe_malloc (x, __FILE__, __LINE__) // print file and line info

int main (int argc, char *argv[])
{
  int i, j, k;
  char mapstr[8];
  int mapver;
  long posx, posy;
  short ang;
  char nullstr[4];
  char scratch[4];
  char wallstring[8];
  char floorstring[8];
  char ceilingstring[8];
  char nowallstring[8];
  long lineoff, linesize, veroff, versize;
  long sectoff, sectsize, sideoff, sidesize;
  long thingoff, thingsize;
  long diroff, dirnums, zeroff, zersize;


  nullstr[0] = nullstr[1] = nullstr[2] = nullstr[3] = 0;

  printf ("\nMAP2WAD version %s Copyright (C) 2002 Nicholai Main\n", VERSIONSTR);
  printf ("Built on %s\n", __DATE__);

  printf ("This program is distributed in the hope that it will be useful,\n"
          "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
          "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
          "GNU General Public License for more details.\n"
          "This is however free software, and you are welcome to distribute and\n"
          "modify it in accordance to the GNU General Public Licence,\n"
          "see file COPYING for more information.\n\n");

  if (argc < 4)
  {
    printf ("Usage: map2wad infile outfile mapstring [floor] [ceiling] [wall]\n"
            "  infile = a suitable BUILD created .map file\n"
            "  outfile = the .wad file created by map2wad,\n"
            "      (will overwrite any existing file)\n"
            "  mapstring = desired target map string, should be\n"
            "      no more than 8 characters, most likely in the\n"
            "      format of MAPxx or ExMx, and is case sensitive\n"
            "  floor = (optional) default floor texture\n"
            "  ceiling = (optional) default ceiling texture\n"
            "  wall = (optional) default wall texture\n"
            "    You cannot set ceiling unless you specifiy floor, and\n"
            "    you cannot set wall unless you specifiy ceiling and floor\n");


    return EXIT_FAILURE;
  }
  strncpy (mapstr, argv[3], 8);
  if (!(mapfile = fopen (argv[1], "rb")))
  {
    printf ("Error opening infile!\n"
            "Does map2wad have permission to read from this file?\n"
            "Did your crappy IBM 75GXP just break on you? ;)\n");
    return EXIT_FAILURE;
  }
  if (!(wadfile = fopen (argv[2], "wb")))
  {
    printf ("Error creating outfile!\n"
            "Does map2wad have permission to write to this file?\n");
    fclose (mapfile); // map file might have successfully opened
    return EXIT_FAILURE;
  }
  if (argc > 4)
    strncpy (floorstring, argv[4], 8);
  else
    strncpy (floorstring, FLRSTR, 8);
  if (argc > 5)
    strncpy (ceilingstring, argv[5], 8);
  else
    strncpy (ceilingstring, CEILSTR, 8);
  if (argc > 6)
    strncpy (wallstring, argv[6], 8);
  else
    strncpy (wallstring, WALLSTR, 8);
  strncpy (nowallstring, NOWALLSTR, 8);


  // load and parse mapfile
  printf ("Loading infile...\n");
  s_fread (&mapver, 1, 4, mapfile);
  printf ("Version is %i (dec)\n", mapver);
  if (mapver != 7) // only can handle version 7 maps
  {
    printf ("map2wad only processes version 7 (dec)\n"
            "aborting now...\n");
    fclose (mapfile);
    fclose (wadfile);
    return EXIT_FAILURE;
  }
  printf ("Loading start coords...\n");
  s_fread (&posx, 4, 1, mapfile);
  s_fread (&posy, 4, 1, mapfile);
  s_fread (scratch, 1, 4, mapfile); // posz
  s_fread (&ang, 2, 1, mapfile);
  s_fread (scratch, 1, 2, mapfile); // current sector num

  s_fread (&numbuildsectors, 2, 1, mapfile);
  printf ("Loading %i sectors...\n", numbuildsectors);
  buildsectors = s_malloc (sizeof (buildsector) * numbuildsectors);
  s_fread (buildsectors, sizeof (buildsector), numbuildsectors, mapfile);

  s_fread (&numbuildwalls, 2, 1, mapfile);
  printf ("Loading %i walls...\n", numbuildwalls);
  buildwalls = s_malloc (sizeof (buildwall) * numbuildwalls);
  s_fread (buildwalls, sizeof (buildwall), numbuildwalls, mapfile);
  // cache wallhit
  wallhit = s_malloc (sizeof (int) * numbuildwalls);


  s_fread (&numbuildsprites, 2, 1, mapfile);
  printf ("Ignoring %i sprites...\n", numbuildsprites);
  fclose (mapfile);

  printf ("Processing .map into .wad...\n");

  // allocate the absolute max possible amound of doom data space needed
  // at most one linedef for each buildwall
  linedefs = s_malloc (sizeof (linedef) * numbuildwalls);
  // exactly 1 sidedef for each buildwall
  sidedefs = s_malloc (sizeof (sidedef) * numbuildwalls);
  // at most 1 vertex for each buildwall
  vertexes = s_malloc (sizeof (vertex) * numbuildwalls);
  // exactly one sector for each buildse.cx :o
  sectors = s_malloc (sizeof (sector) * numbuildsectors);
  // one thing for player start
  things = s_malloc (sizeof (thing) * 1);

  // geometric misc comments
  /*
  One doom x or y unit = 16 build x or y units.  Therefore, doom can make much larger maps
  than build which restricts max map size to (-65536,-65536) to (65536, 65536) which is
  equivalent to (-4096, -4096) to (4096, 4096) in doom.  If you are not happy with this
  limiation, use dck or something.
  Note that while the build map format allows much larger xy coords, the 64k limit is
  in the editor.
  One doom z unit = 256 build z units.  thus the max z heights in doom (-32767 to 32768)
  correspond to a bit over -8 million to 8 million.  I do not know what the build limits
  are here; I put a rock on the pgdn key and left for 5 minutes, i came back and it was
  around 5 million.  So it seems likely that build can exceed doom limits here, so keep
  your ceilings under 8 million, eh?  With all this z buillshit its really a moot point
  though, who ever makes 32k high rooms in doom?
  Another XY restriction i just remembered: dont make things too close together!
  Build allows finer granulity and closer verticies, and map2wad will round to doom size
  1 and put these together, which will probably fuck your map.
  */

  // use buildsectors to create sectors - easy part :)
  // remember to set numsectors!!!!

  numsectors = numbuildsectors;
  for (i = 0; i < numbuildsectors; i++)
  {
    sectors[i].floorheight = -1 * (buildsectors[i].floorz >> 8); // reduce by 256 and flip
    sectors[i].ceilingheight = -1 * (buildsectors[i].ceilingz >> 8);
    strncpy (sectors[i].floortex, floorstring, 8); // textures would go here
    strncpy (sectors[i].ceilingtex, ceilingstring, 8);
    sectors[i].light = 255,
    sectors[i].type = sectors[i].tag = 0;
  }

  // create vertexes from buildwalls

  numvertexes = numbuildwalls;
  for (i = 0; i < numbuildwalls; i++)
  {
    vertexes[i].x = (buildwalls[i].x >> 4); // reduce by 16
    vertexes[i].y = -1 * (buildwalls[i].y >> 4); // reduce by 16 and flip
  }

  // create sidedefs from buildwalls
  numsidedefs = numbuildwalls;
  for (i = 0; i < numbuildwalls; i++)
  {
    sidedefs[i].xoff = sidedefs[i].yoff = 0;
    if (buildwalls[i].nextwall != -1) // wall is 2sided
    {
      strncpy (sidedefs[i].uptex, wallstring, 8);
      strncpy (sidedefs[i].lowtex, wallstring, 8);
      strncpy (sidedefs[i].midtex, nowallstring, 8);
    }
    else // wall is 1sided
    {
      strncpy (sidedefs[i].midtex, wallstring, 8);
      strncpy (sidedefs[i].uptex, nowallstring, 8);
      strncpy (sidedefs[i].lowtex, nowallstring, 8);
    }
  }
  // create linedefs from buildwalls
  for (i = 0; i < numbuildwalls; i++)
    wallhit[i] = 0; // a refrence of whether or not the wall has been bound to a linedef

  numlinedefs = 0;
  for (i = 0; i < numbuildwalls; i++)
  {
    if (wallhit[i] == 0) // wall has not been hit yet
    {
      if (buildwalls[i].nextwall == -1) // wall is onesided
      { // bind verticies and sidedef
        linedefs[numlinedefs].v1 = i;
        linedefs[numlinedefs].v2 = buildwalls[i].point2;
        linedefs[numlinedefs].flags = 1; // impassible
        linedefs[numlinedefs].type = linedefs[numlinedefs].tag = 0;
        linedefs[numlinedefs].sidedef1 = i;
        linedefs[numlinedefs].sidedef2 = -1;
        wallhit[i] = 1;
        numlinedefs++;
      }
      else // wall is twosided
      {
        linedefs[numlinedefs].v1 = i;
        linedefs[numlinedefs].v2 = buildwalls[i].point2;
        // i THINK that the vertex # buildwalls[i].nextwall is unused, but a voice in my
        // head says no
        linedefs[numlinedefs].flags = 4; // 2s, !imp
        linedefs[numlinedefs].type = linedefs[numlinedefs].tag = 0;
        linedefs[numlinedefs].sidedef1 = i;
        linedefs[numlinedefs].sidedef2 = buildwalls[i].nextwall;
        wallhit[i] = 1;
        wallhit[buildwalls[i].nextwall] = 1;
        numlinedefs++;
      }
    } // wallhit[i] = 0
  }

  // the sector refrences were not set during sidedef creation, because of build's pain
  // in the ass way of storing them, they will be computed now

  for (i = 0; i < numbuildwalls; i++)
    wallhit[i] = 0; // reuse to determine whether or not a sidedef has been offsetted

  // first we go through the sectors, begin with each sectors first wall, and work around
  // according to bldform.txt point2 is always in the same sector, wheee.
  /* obsolete code -- see below
  for (i = 0; i < numbuildsectors; i++)
  {
    j = buildsectors[i].wallptr;
    sidedefs[j].sectoref = i;
    k = j;
    j = buildwalls[k].point2;
    wallhit[k] = 1;
    while (wallhit[j] == 0)
    {
      sidedefs[j].sectoref = i;
      k = j;
      j = buildwalls[k].point2;
      wallhit[k] = 1;
    }
  }
  */
  // the above method covers everything but inner loops.
  /* I was tooling around some various code for other things, when I read a line of code
     in wad2map that suggested that you can guarantee that the walls in a sector are
     numbered sequentally from wallptr on up to wallptr + wallnum - 1;
     grrr!!!!  this was not mentioned in Ken's format docs.  It makes my
     other algorithm stupid (not that there was much thought in my other algorthim, but
     moreso all the thought i wasted trying to figure out how to adapt it to inner loops
  */
  k = 0; // use k to store number of "strange errors"
  for (i = 0; i < numbuildsectors; i++)
  {
    for (j = buildsectors[i].wallptr;
         j < buildsectors[i].wallptr + buildsectors[i].wallnum; j++)
    {
      if (j < numbuildwalls) // should never be false, but to protect from errors
      {
        if (wallhit[j] == 1)
        {
          k++; // this should never happen unless a map is fubared
        }
          else
        {
          sidedefs[j].sectoref = i;
          wallhit[j] = 1;
        }
      }
    }
  }
  if (k > 0)
    printf ("%i strange wall errors (double process) found on map.\n"
            "Did build fubar this map?  Attempting to continue normally.\n"
            "Make sure to check all sidedef sector refrences in output .wad\n", k);
  k = 0;
  for (i = 0; i < numbuildwalls; i++)
    if (wallhit[i] == 0)
      k++;
  if (k > 0)
    printf ("%i strange wall errors (zero process) found on map.\n"
            "Did build fubar this map?  Attempting to continue normally.\n"
            "Make sure to check all sidedef sector refrences in output .wad\n", k);

  // a very bad hacky method to eliminate unused verticies that really sucks
  // doesnt actually delete them but removes refrence to them

  for (i = 0; i < numbuildwalls; i++)
    wallhit[i] = 0; // used to tell whether or not a vertex has been remapped

  for (i = 0; i < numbuildwalls; i++) // cycle through vertexes
  {
    for (j = i + 1; j < numbuildwalls; j++) // cycle through all vertexes after this one
    {
      if ((wallhit[i] == 0) &&
          (vertexes[j].x == vertexes[i].x) &&
          (vertexes[j].y == vertexes[i].y))
      { // two vertexes are the same and the second has not been remapped to the first
        wallhit[j] = 1; // vertex j has been remapped to i
        for (k = 0; k < numbuildwalls; k++) // cycle through linedefs
        {
          if (linedefs[k].v1 == j)
            linedefs[k].v1 = i;
          if (linedefs[k].v2 == j)
            linedefs[k].v2 = i;
        }
      }
    }
  }

  /* why doesnt this commented code block work?  you tell me, i give you one dolla ;)
  // removes all vertices that were marked as unused in last loop
  for (i = 0; i < numbuildwalls; i++)
  {
    if (wallhit[i] == 1) // from last loop, this means that vertex at i has been remapped
    {
      numvertexes--; // keep track of how many vertexes we've lopped off the end
      // we move every vertex back in line, stepping over the unneeded one and leaving
      // garbage at the end
      for (j = i + 1; j < numvertexes + 1; j++) // numbuildwalls?
      {
        vertexes[j - 1].x = vertexes[j].x;
        vertexes[j - 1].y = vertexes[j].y;
        for ( k = 0; k < numlinedefs; k++)
        { // all linedefs with vertex j must now carry vertex j - 1
          if (linedefs[k].v1 == j)
          {
            linedefs[k].v1--;
          }
          if (linedefs[k].v2 == j)
          {
            linedefs[k].v2--;
          }
        }
      }
    }
  }
  */
  // finally, create one thing for player start
  numthings = 1;
  things[0].type = 1; // P1 start
  things[0].flags = 7; // skill 12345
  things[0].x = (posx >> 4); // reduce by 16
  things[0].y = -1 * (posy >> 4); // reduce by 16 and flip
  things[0].angle = (-ang) >> 8; // remap 0 to 2047 cw to 0 to 7 ccw
  things[0].angle *= 360; // FIX - this method floors angles instead of
                          // rounding them properly up or down
  /* ok, now, hopefully :), the linedefs, sidedefs, vertexes, and sectors structures
     are all full of the proper data.  now to save the .wadfile, per dmspec16
     in order, we write
     MAPXX or whatever
     THINGS
     LINEDEFS
     SIDEDEFS
     VERTEXES
     SEGS
     SSECTORS
     NODES
     SECTORS
     REJECT
     BLOCKMAP
  */
  printf ("Writing outfile PWAD...\n");

  diroff = 12;
  dirnums = 11; // 11 entries in wadfile

  zeroff = 188;
  zersize = 0; // offset and size for all 0 byte lumps

  thingoff = 188;
  thingsize = sizeof (thing) * numthings;
  lineoff = thingoff + thingsize;
  linesize = sizeof (linedef) * numlinedefs;
  sideoff = lineoff + linesize;
  sidesize = sizeof (sidedef) * numsidedefs;
  veroff = sideoff + sidesize;
  versize = sizeof (vertex) * numvertexes;
  sectoff = veroff + versize;
  sectsize = sizeof (sector) * numsectors;


  s_fwrite ("PWAD", 1, 4, wadfile); // PWAD header
  s_fwrite (&dirnums, 4, 1, wadfile); // 6 entries
  s_fwrite (&diroff, 4, 1, wadfile); // directory starts at offset 0x0c (12)

  // data starts at offset 188 dec

  s_fwrite (&zeroff, 4, 1, wadfile); // data offset to MAPxx or whatever
  s_fwrite (&zersize, 4, 1, wadfile); // MAPxx or whatever is 0big
  s_fwrite (mapstr, 1, 8, wadfile); // ascii MAPxx or whatever

  s_fwrite (&thingoff, 4, 1, wadfile); // same thing with THINGS
  s_fwrite (&thingsize, 4, 1, wadfile);
  s_fwrite ("THINGS", 1, 6, wadfile);
  s_fwrite (nullstr, 1, 2, wadfile); // null chars at end of "THINGS"

  s_fwrite (&lineoff, 4, 1, wadfile);
  s_fwrite (&linesize, 4, 1, wadfile);
  s_fwrite ("LINEDEFS", 1, 8, wadfile);

  s_fwrite (&sideoff, 4, 1, wadfile);
  s_fwrite (&sidesize, 4, 1, wadfile);
  s_fwrite ("SIDEDEFS", 1, 8, wadfile);

  s_fwrite (&veroff, 4, 1, wadfile);
  s_fwrite (&versize, 4, 1, wadfile);
  s_fwrite ("VERTEXES", 1, 8, wadfile);

  s_fwrite (&zeroff, 4, 1, wadfile);
  s_fwrite (&zersize, 4, 1, wadfile);
  s_fwrite ("SEGS", 1, 4, wadfile);
  s_fwrite (nullstr, 1, 4, wadfile);

  s_fwrite (&zeroff, 4, 1, wadfile);
  s_fwrite (&zersize, 4, 1, wadfile);
  s_fwrite ("SSECTORS", 1, 8, wadfile);

  s_fwrite (&zeroff, 4, 1, wadfile);
  s_fwrite (&zersize, 4, 1, wadfile);
  s_fwrite ("NODES", 1, 5, wadfile);
  s_fwrite (nullstr, 1, 3, wadfile);

  s_fwrite (&sectoff, 4, 1, wadfile);
  s_fwrite (&sectsize, 4, 1, wadfile);
  s_fwrite ("SECTORS", 1, 7, wadfile);
  s_fwrite (nullstr, 1, 1, wadfile);

  s_fwrite (&zeroff, 4, 1, wadfile);
  s_fwrite (&zersize, 4, 1, wadfile);
  s_fwrite ("REJECT", 1, 6, wadfile);
  s_fwrite (nullstr, 1, 2, wadfile);

  s_fwrite (&zeroff, 4, 1, wadfile);
  s_fwrite (&zersize, 4, 1, wadfile);
  s_fwrite ("BLOCKMAP", 1, 8, wadfile);

  // now data
  printf ("Writing %i THINGS...\n", numthings);
  s_fwrite (things, sizeof (thing), numthings, wadfile);
  printf ("Writing %i LINEDEFS...\n", numlinedefs);
  s_fwrite (linedefs, sizeof (linedef), numlinedefs, wadfile);
  printf ("Writing %i SIDEDEFS...\n", numsidedefs);
  s_fwrite (sidedefs, sizeof (sidedef), numsidedefs, wadfile);
  printf ("Writing %i VERTEXES...\n", numvertexes);
  s_fwrite (vertexes, sizeof (vertex), numvertexes, wadfile);
  printf ("Writing %i SECTORS...\n", numsectors);
  s_fwrite (sectors, sizeof (sector), numsectors, wadfile);

  fclose (wadfile);

  printf ("Complete!  Now buy me a beer.\n");

  // free for shutdown
  free (buildsectors);
  free (buildwalls);
  free (wallhit);
  free (linedefs);
  free (sidedefs);
  free (vertexes);
  free (sectors);
  return EXIT_SUCCESS;
}

